//------------------------------------------------------------------------------
// Copyright (c) 2005, 2006 IBM Corporation and others.
// All rights reserved. This program and the accompanying materials
// are made available under the terms of the Eclipse Public License v1.0
// which accompanies this distribution, and is available at
// http://www.eclipse.org/legal/epl-v10.html
//
// Contributors:
// IBM Corporation - initial implementation
//------------------------------------------------------------------------------
package org.eclipse.epf.authoring.ui.forms;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.edit.ui.provider.AdapterFactoryContentProvider;
import org.eclipse.epf.authoring.ui.AuthoringUIResources;
import org.eclipse.epf.authoring.ui.AuthoringUIText;
import org.eclipse.epf.authoring.ui.filters.CategoryFilter;
import org.eclipse.epf.authoring.ui.filters.CustomCategoryFilter;
import org.eclipse.epf.library.edit.IFilter;
import org.eclipse.epf.library.edit.TngAdapterFactory;
import org.eclipse.epf.library.edit.itemsfilter.FilterConstants;
import org.eclipse.epf.library.util.LibraryManager;
import org.eclipse.epf.uma.CustomCategory;
import org.eclipse.epf.uma.Domain;
import org.eclipse.epf.uma.WorkProduct;
import org.eclipse.epf.uma.WorkProductType;
import org.eclipse.epf.uma.util.AssociationHelper;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.forms.editor.FormEditor;


/**
 * The Categories page in the Work Product editor.
 * 
 * @author Jeff Hardy
 * @author Kelvin Low
 * @since 1.0
 */
public class WorkProductCategoriesPage extends AssociationFormPage {

	private static final String FORM_PAGE_ID = "workProductCategoriesPage"; //$NON-NLS-1$

	private WorkProduct workProduct;

	/**
	 * Creates a new instance.
	 */
	public WorkProductCategoriesPage(FormEditor editor) {
		super(editor, FORM_PAGE_ID, AuthoringUIText.CATEGORIES_PAGE_TITLE);
	}

	/**
	 * @see org.eclipse.ui.forms.editor.init(IEditorSite, IEditorInput)
	 */
	public void init(IEditorSite site, IEditorInput input) {
		super.init(site, input);
		workProduct = (WorkProduct) contentElement;
		setUseCategory3(true);
		setCategoryIsSingleSelection1(true);
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.AssociationFormPage#initContentProviderSelected()
	 */
	protected void initContentProviderSelected() {
		contentProviderSelected = new AdapterFactoryContentProvider(
				TngAdapterFactory.INSTANCE
						.getNavigatorView_ComposedAdapterFactory()) {
			public Object[] getElements(Object object) {
				List list = new ArrayList();
				Domain domain = AssociationHelper
						.getDomain((WorkProduct) object);
				if (domain != null)
					list.add(domain);
				return list.toArray();
			}
		};
		viewer_selected.setContentProvider(contentProviderSelected);
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.AssociationFormPage#addItemsToModel1(ArrayList)
	 */
	protected void addItemsToModel1(ArrayList addItems) {
		// Update the model.
		if (!addItems.isEmpty()) {
			Object domain = addItems.get(0);
			LibraryManager.getInstance().addToDomain(getActionManager(),
					(Domain) domain, workProduct);
			// RATLC00383094: Standard Categories: contributing standard
			// category is not deleted when it is empty
			if (usedCategories.contains(domain)) {
				usedCategories.remove(domain);
			}
		}
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.AssociationFormPage#removeItemsFromModel1(ArrayList)
	 */
	protected void removeItemsFromModel1(ArrayList rmItems) {
		// Update the model.
		if (!rmItems.isEmpty()) {
			for (Iterator it = rmItems.iterator(); it.hasNext();) {
				Domain domain = (Domain) it.next();
				LibraryManager.getInstance().removeFromDomain(
						getActionManager(), domain, workProduct);
				// RATLC00383094: Standard Categories: contributing standard
				// category is not deleted when it is empty
				if (!usedCategories.contains(domain)) {
					usedCategories.add(domain);
				}
			}
		}
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.AssociationFormPage#initContentProviderSelected2()
	 */
	protected void initContentProviderSelected2() {
		contentProviderSelected2 = new AdapterFactoryContentProvider(
				TngAdapterFactory.INSTANCE
						.getNavigatorView_ComposedAdapterFactory()) {
			public Object[] getElements(Object object) {
				return AssociationHelper.getWorkProductTypes(
						(WorkProduct) object).toArray();
			}
		};
		viewer_selected2.setContentProvider(contentProviderSelected2);
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.AssociationFormPage#addItemsToModel2()
	 */
	protected void addItemsToModel2(ArrayList addItems) {
		// Update the model.
		if (!addItems.isEmpty()) {
			for (java.util.Iterator it = addItems.iterator(); it.hasNext();) {
				WorkProductType wpType = (WorkProductType) it.next();
				LibraryManager.getInstance().addToWorkProductType(
						getActionManager(), wpType, workProduct);
				// RATLC00383094: Standard Categories: contributing standard
				// category is not deleted when it is empty
				if (usedCategories.contains(wpType)) {
					usedCategories.remove(wpType);
				}
			}
		}
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.AssociationFormPage#removeItemsFromModel2()
	 */
	protected void removeItemsFromModel2(ArrayList rmItems) {
		// Update the model.
		if (!rmItems.isEmpty()) {
			for (Iterator it = rmItems.iterator(); it.hasNext();) {
				WorkProductType wpType = (WorkProductType) it.next();
				LibraryManager.getInstance().removeFromWorkProductType(
						getActionManager(), wpType, workProduct);
				if (!usedCategories.contains(wpType)) {
					usedCategories.add(wpType);
				}
			}
		}
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.AssociationFormPage#initContentProviderSelected3()
	 */
	protected void initContentProviderSelected3() {
		contentProviderSelected3 = new AdapterFactoryContentProvider(
				TngAdapterFactory.INSTANCE
						.getNavigatorView_ComposedAdapterFactory()) {
			public Object[] getElements(Object object) {
				return AssociationHelper.getCustomCategories(
						(WorkProduct) object).toArray();
			}
		};
		viewer_selected3.setContentProvider(contentProviderSelected3);
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.AssociationFormPage#addItemsToModel3(ArrayList)
	 */
	protected void addItemsToModel3(ArrayList addItems) {
		// Update the model.
		if (!addItems.isEmpty()) {
			for (java.util.Iterator it = addItems.iterator(); it.hasNext();) {
				CustomCategory customCategory = (CustomCategory) it.next();
				LibraryManager.getInstance().addToCustomCategory(
						getActionManager(), customCategory, workProduct);
				if (usedCategories.contains(customCategory)) {
					usedCategories.remove(customCategory);
				}
			}
		}
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.AssociationFormPage#removeItemsFromModel3(ArrayList)
	 */
	protected void removeItemsFromModel3(ArrayList rmItems) {
		// Update the model.
		if (!rmItems.isEmpty()) {
			for (java.util.Iterator it = rmItems.iterator(); it.hasNext();) {
				CustomCategory customCategory = (CustomCategory) it.next();
				LibraryManager.getInstance().removeFromCustomCategory(
						getActionManager(), customCategory, workProduct);
				if (!usedCategories.contains(customCategory)) {
					usedCategories.add(customCategory);
				}
			}
		}
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.DescriptionFormPage#getContentElement()
	 */
	protected Object getContentElement() {
		return workProduct;
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.DescriptionFormPage#getTabString()
	 */
	protected String getTabString() {
		return FilterConstants.DOMAINS;
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.DescriptionFormPage#getTabString2()
	 */
	protected String getTabString2() {
		return FilterConstants.WORKPRODUCTTYPES;
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.DescriptionFormPage#getTabString3()
	 */
	protected String getTabString3() {
		return FilterConstants.CUSTOM_CATEGORIES;
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.DescriptionFormPage#getFilter()
	 */
	protected IFilter getFilter() {
		return filter = new CategoryFilter() {
			protected boolean childAccept(Object obj) {
				return (obj instanceof Domain);
			}
		};
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.DescriptionFormPage#getFilter2()
	 */
	protected IFilter getFilter2() {
		return filter2 = new CategoryFilter() {
			protected boolean childAccept(Object obj) {
				return (obj instanceof WorkProductType);
			}
		};
	}

	/**
	 * @see org.eclipse.epf.authoring.ui.forms.DescriptionFormPage#getFilter3()
	 */
	protected IFilter getFilter3() {
		return filter3 = new CustomCategoryFilter();
	}
	protected String getSectionDescription() {
		return AuthoringUIResources.workProductCategoriesPage_sectionDescription;
	}

	protected String getSectionName() {
		return AuthoringUIResources.workProductCategoriesPage_sectionName;
	}

	protected String getSelectedLabel() {
		return AuthoringUIResources.workProductCategoriesPage_selectedLabel;
	}
	
	protected String getSelectedLabel2() {
		return AuthoringUIResources.workProductCategoriesPage_selectedLabel2;
	}
	
	protected String getSelectedLabel3() {
		return AuthoringUIResources.workProductCategoriesPage_selectedLabel3;
	}
}
