//------------------------------------------------------------------------------
// Copyright (c) 2005, 2006 IBM Corporation and others.
// All rights reserved. This program and the accompanying materials
// are made available under the terms of the Eclipse Public License v1.0
// which accompanies this distribution, and is available at
// http://www.eclipse.org/legal/epl-v10.html
//
// Contributors:
// IBM Corporation - initial implementation
//------------------------------------------------------------------------------
package org.eclipse.epf.authoring.ui.wizards;

import java.io.File;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.epf.authoring.ui.AuthoringUIPlugin;
import org.eclipse.epf.authoring.ui.AuthoringUIResources;
import org.eclipse.epf.authoring.ui.preferences.LibraryLocationData;
import org.eclipse.epf.library.edit.util.TngUtil;
import org.eclipse.epf.library.ui.LibraryUIManager;
import org.eclipse.epf.library.ui.LibraryUIText;
import org.eclipse.epf.library.xmi.XMILibraryUtil;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;


/**
 * A wizard that guides the user to create a new Method Library.
 * 
 * @author Bingxue Xu
 * @author Kelvin Low
 * @since 1.0
 */
public class NewLibraryWizard extends Wizard implements INewWizard {

	protected NewLibraryMainPage mainPage;

	protected boolean okToComplete = false;

	protected LibraryLocationData llData;

	/**
	 * Creates a new instance.
	 */
	public NewLibraryWizard(LibraryLocationData llData) {
		super();
		this.llData = llData;
		setWindowTitle(AuthoringUIResources.newLibraryWizard_title); //$NON-NLS-1$
	}

	/**
	 * Creates a new instance.
	 */
	public NewLibraryWizard() {
		this(null);
	}

	/**
	 * @see org.eclipse.ui.IWorkbenchWizard#init(IWorkbench,
	 *      IStructuredSelection)
	 */
	public void init(IWorkbench workbench, IStructuredSelection selection) {
	}

	/**
	 * @see org.eclipse.jface.wizard.Wizard#addPages()
	 */
	public void addPages() {
		mainPage = new NewLibraryMainPage(llData); //$NON-NLS-1$
		addPage(mainPage);
	}

	/**
	 * @see org.eclipse.jface.wizard.IWizard#canFinish()
	 */
	public boolean canFinish() {
		return okToComplete;
	}

	/**
	 * @see org.eclipse.jface.wizard.IWizard#performFinish()
	 */
	public boolean performFinish() {
		// Create the new Library.
		mainPage.saveData();

		// Check the library name field value, this will not be needed if we
		// remove that field.
		String libName = llData.getLibName();
		String msg = TngUtil.checkElementName(libName,
				LibraryUIText.TEXT_METHOD_LIBARARY);
		if (msg != null) {
			AuthoringUIPlugin.getDefault().getMsgDialog().displayError(
					AuthoringUIResources.newLibraryWizard_title, //$NON-NLS-1$
					msg);
			llData.setOkPressed(false);
			return false;
		}

		llData.setOkPressed(true);
		boolean answer = checkAndCreateDir();
		if (answer) {
			return true;
		} else {
			llData.setOkPressed(false);
			return false;
		}
	}

	private boolean checkAndCreateDir() {
		boolean answer = false;

		// Convert relative path to absolute one, and check the validity of the
		// path.
		String tmpLibPath = llData.getParentFolder();
		tmpLibPath = LibraryUIManager.toAbsoluteLibraryPath(tmpLibPath);
		IStatus status = XMILibraryUtil.isValidLibrary(tmpLibPath);
		if (!status.isOK()) {
			AuthoringUIPlugin
					.getDefault()
					.getMsgDialog()
					.displayError(
							AuthoringUIResources.newLibraryWizard_title, //$NON-NLS-1$
							AuthoringUIResources.newLibraryError_msg, //$NON-NLS-1$
							AuthoringUIResources.bind(AuthoringUIResources.newLibraryError_reason1, tmpLibPath) + "\n\n" + //$NON-NLS-1$ //$NON-NLS-2$
									AuthoringUIResources.newLibraryError_advice); //$NON-NLS-1$ 
			return false;
		}

		llData.setParentFolder(tmpLibPath);

		String dir = llData.getParentFolder();
		File file = new File(dir);
		if (file.exists()) {
			// The lbrary path already exist, check for the presence of the
			// library.xmi file.
			String libraryFilePath = dir
					+ ((dir.endsWith(File.separator)) ? "" : File.separator) + "library.xmi"; //$NON-NLS-1$ //$NON-NLS-2$
			File libraryFile = new File(libraryFilePath);
			if (libraryFile.exists()) {
				answer = false;
				AuthoringUIPlugin
						.getDefault()
						.getMsgDialog()
						.displayError(
								AuthoringUIResources.newLibraryWizard_title, //$NON-NLS-1$
								AuthoringUIResources.newLibraryError_msg, //$NON-NLS-1$
								AuthoringUIResources.bind(AuthoringUIResources.newLibraryError_reason2, tmpLibPath) + "\n\n" + //$NON-NLS-1$ //$NON-NLS-2$
										AuthoringUIResources.newLibraryError_advice); //$NON-NLS-1$ 
			} else
				answer = true;
		} else {
			if (!file.mkdirs()) {
				AuthoringUIPlugin
						.getDefault()
						.getMsgDialog()
						.displayError(
								AuthoringUIResources.newLibraryWizard_title, //$NON-NLS-1$
								AuthoringUIResources.newLibraryError_msg, //$NON-NLS-1$
								AuthoringUIResources.bind(AuthoringUIResources.newLibraryError_reason3, file.getAbsolutePath()) + "\n\n" + //$NON-NLS-1$ //$NON-NLS-2$
										AuthoringUIResources.newLibraryError_advice); //$NON-NLS-1$ 
				answer = false;
			} else {
				answer = true;
			}
		}

		return answer;
	}

}
