//------------------------------------------------------------------------------
// Copyright (c) 2005, 2006 IBM Corporation and others.
// All rights reserved. This program and the accompanying materials
// are made available under the terms of the Eclipse Public License v1.0
// which accompanies this distribution, and is available at
// http://www.eclipse.org/legal/epl-v10.html
//
// Contributors:
// IBM Corporation - initial implementation
//------------------------------------------------------------------------------
package org.eclipse.epf.importing.wizards;

import java.util.Iterator;

import org.eclipse.epf.importing.ImportPlugin;
import org.eclipse.epf.importing.ImportResources;
import org.eclipse.epf.importing.services.ConfigSpecs;
import org.eclipse.epf.importing.services.ConfigurationImportData;
import org.eclipse.epf.library.ui.LibraryUIImages;
import org.eclipse.epf.library.ui.wizards.BaseWizardPage;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;


/**
 * A wizard page that prompts the user to select the configuration
 * specificaitons to import.
 * 
 * @author Jeff Hardy
 * @author Kelvin Low
 * @since 1.0
 */
public class SelectConfigSpecsToImportPage extends BaseWizardPage implements
		ISelectionChangedListener, ICheckStateListener {

	public static final String PAGE_NAME = SelectConfigSpecsToImportPage.class
			.getName();

	private static final String libraryConfigSpecExists = ImportResources.SelectConfigSpecsToImportPage_libraryConfigSpecExists; //$NON-NLS-1$

	private static final String libraryConfigSpecNotExists = ImportResources.SelectConfigSpecsToImportPage_libraryConfigSpecNotExists; //$NON-NLS-1$

	private static final String configSpecNoSelection = ImportResources.SelectConfigSpecsToImportPage_configSpecNoSelection; //$NON-NLS-1$

	private static final String configSpecDataLabel = ImportResources.SelectConfigSpecsToImportPage_configSpecDataLabel; //$NON-NLS-1$

	private CheckboxTableViewer ctrl_chkboxTableViewer;

	// 2 controls for displaying configspec info for plugins in import directory
	private Text ctrl_briefDescImport;

	private Label importConfigSpecLabel;

	// 2 controls for displaying configspec info for plugins in the library
	private Text ctrl_briefDescLibrary;

	private Label libraryConfigSpecLabel;

	private Composite container;

	private ConfigurationImportData data;

	/**
	 * Creates a new instance.
	 */
	public SelectConfigSpecsToImportPage(ConfigurationImportData data) {
		super(PAGE_NAME);
		setTitle(ImportResources.selectConfigSpecsWizardPage_title); //$NON-NLS-1$
		setDescription(ImportResources.selectConfigSpecsWizardPage_text); //$NON-NLS-1$
		setImageDescriptor(ImportPlugin.getDefault().getImageDescriptor(
				"full/wizban/ImportLibraryConfiguration.gif")); //$NON-NLS-1$
		this.data = data;
	}

	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(Composite)
	 */
	public void createControl(Composite parent) {
		container = new Composite(parent, SWT.NONE);
		container.setLayout(new GridLayout(2, true));

		Composite tableContainer = new Composite(container, SWT.NONE);
		tableContainer.setLayout(new GridLayout());
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.horizontalSpan = 2;
		tableContainer.setLayoutData(gridData);

		createLabel(
				tableContainer,
				ImportResources.SelectConfigSpecsToImportPage_label_configs, 2); //$NON-NLS-1$
		ctrl_chkboxTableViewer = createCheckboxTableViewer(tableContainer, 1);

		ILabelProvider labelProvider = new LabelProvider() {
			public Image getImage(Object element) {
				return LibraryUIImages.IMG_METHOD_CONFIGURATON;
			}

			public String getText(Object element) {
				return super.getText(element);
			}
		};

		ctrl_chkboxTableViewer.setLabelProvider(labelProvider);
		ctrl_chkboxTableViewer.setContentProvider(new ArrayContentProvider());

		Composite importContainer = new Composite(container, SWT.NONE);
		importContainer.setLayout(new GridLayout());
		importContainer.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		createImportConfigSpecInfo(importContainer);

		Composite libraryContainer = new Composite(container, SWT.NONE);
		libraryContainer.setLayout(new GridLayout());
		libraryContainer.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		createLibraryConfigSpecInfo(libraryContainer);

		addListeners();

		setControl(container);
		setPageComplete(false);
	}

	private void createImportConfigSpecInfo(Composite container) {
		importConfigSpecLabel = createLabel(container, configSpecNoSelection, 1);
		createLine(container, 1);

		createLabel(
				container,
				ImportResources.SelectConfigSpecsToImportPage_label_desc, 1); //$NON-NLS-1$
		ctrl_briefDescImport = createTextMulti(container, "", 275, 100, 1); //$NON-NLS-1$
	}

	private void createLibraryConfigSpecInfo(Composite container) {
		libraryConfigSpecLabel = createLabel(container, configSpecNoSelection,
				1);
		createLine(container, 1);

		createLabel(
				container,
				ImportResources.SelectConfigSpecsToImportPage_label_desc, 1); //$NON-NLS-1$
		ctrl_briefDescLibrary = createTextMulti(container, "", 275, 100, 1); //$NON-NLS-1$
	}

	private void addListeners() {
		ctrl_chkboxTableViewer.addSelectionChangedListener(this);
		ctrl_chkboxTableViewer.addCheckStateListener(this);
	}

	public void checkStateChanged(CheckStateChangedEvent event) {
		setPageComplete(isPageComplete());
		getWizard().getContainer().updateButtons();
	}

	public void selectionChanged(SelectionChangedEvent event) {
		StructuredSelection selection = (StructuredSelection) event
				.getSelection();
		if (!selection.isEmpty()) {
			Object[] configs = selection.toArray();
			setDisplayAttributes((ConfigSpecs.Entry) configs[0]);
		} else {
			clearDisplayAttributes();
		}
	}

	private void setDisplayAttributes(ConfigSpecs.Entry spec) {
		ctrl_briefDescImport
				.setText(spec.configSpec.brief_desc == null ? "" : spec.configSpec.brief_desc); //$NON-NLS-1$
		importConfigSpecLabel.setText(configSpecDataLabel);

		if (spec.existingConfig != null) {
			ctrl_briefDescLibrary.setText(spec.existingConfig
					.getBriefDescription());
			libraryConfigSpecLabel.setText(libraryConfigSpecExists);
		} else {
			libraryConfigSpecLabel.setText(libraryConfigSpecNotExists);
			ctrl_briefDescLibrary.setText(""); //$NON-NLS-1$
		}
	}

	private void clearDisplayAttributes() {
		ctrl_briefDescImport.setText(""); //$NON-NLS-1$
		ctrl_briefDescLibrary.setText(""); //$NON-NLS-1$
		libraryConfigSpecLabel.setText(configSpecNoSelection);
		importConfigSpecLabel.setText(configSpecNoSelection);
	}

	/**
	 * @see org.eclipse.jface.wizard.WizardPage#isCompleted()
	 */
	public boolean isPageComplete() {
		if (getErrorMessage() != null)
			return false;

		if (ctrl_chkboxTableViewer.getCheckedElements().length > 0) {
			// Set data.specs.configs list, fix the selected field of each
			// entry.
			Iterator iter = data.specs.configs.iterator();
			while (iter.hasNext()) {
				ConfigSpecs.Entry e = (ConfigSpecs.Entry) iter.next();
				if (ctrl_chkboxTableViewer.getChecked(e))
					e.selected = true;
				else
					e.selected = false;
			}
			((ImportConfigurationWizard) getWizard()).okToComplete = true;
		} else {
			((ImportConfigurationWizard) getWizard()).okToComplete = false;
		}

		// Always return false so that the "Next" button is never enabled.
		return false;
	}

	public boolean canFlipToNextPage() {
		return isPageComplete();
	}

	public void onEnterPage() {
		ctrl_chkboxTableViewer.setInput(data.specs.configs);
		if (data.specs.configs.size() > 0) {
			IStructuredSelection sel = new StructuredSelection(
					data.specs.configs.get(0));
			ctrl_chkboxTableViewer.setSelection(sel);
		}
	}

}
