//------------------------------------------------------------------------------
// Copyright (c) 2005, 2006 IBM Corporation and others.
// All rights reserved. This program and the accompanying materials
// are made available under the terms of the Eclipse Public License v1.0
// which accompanies this distribution, and is available at
// http://www.eclipse.org/legal/epl-v10.html
//
// Contributors:
// IBM Corporation - initial implementation
//------------------------------------------------------------------------------
package org.eclipse.epf.library;

import java.util.List;

import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryContentProvider;
import org.eclipse.epf.library.configuration.ConfigurationClosure;
import org.eclipse.epf.library.layout.ElementLayoutManager;
import org.eclipse.epf.library.services.DependencyManager;
import org.eclipse.epf.uma.MethodConfiguration;
import org.eclipse.epf.uma.MethodLibrary;

/**
 * Manages a method configuration.
 * 
 * @author Kelvin Low
 * @since 1.0
 */
public class ConfigurationManager implements IConfigurationManager {

	// If true, generate debug traces.
	protected static boolean debug = LibraryPlugin.getDefault().isDebugging();

	// The managed method configuraiton.
	protected MethodConfiguration config;

	// The containing method library for the managed method configuration.
	protected MethodLibrary library;

	// The library manager for the containing method library.
	protected ILibraryManager libraryManager;

	// The dependency manager.
	protected DependencyManager dependencyManager;

	// The layout manager.
	protected ElementLayoutManager layoutManager;

	// The configuration closure.
	protected ConfigurationClosure closure;

	protected AdapterFactoryContentProvider afcp;

	/**
	 * Creates a new instance.
	 * 
	 * @param config
	 *            a method configuration
	 */
	public ConfigurationManager(MethodConfiguration config) {
		this.config = config;

		library = LibraryServiceUtil.getMethodLibrary(config);

		libraryManager = LibraryService.getInstance()
				.getLibraryManager(library);
		afcp = new AdapterFactoryContentProvider(libraryManager
				.getAdapterFactory());

		dependencyManager = new DependencyManager(library);

		if (config == null) {
			layoutManager = new ElementLayoutManager();
		} else {
			layoutManager = new ElementLayoutManager(config);
			// closure = new ConfigurationClosure(this, config);
		}
	}

	/**
	 * Gets the method configuration managed by this configuration manager.
	 * 
	 * @return a method configuration
	 */
	public MethodConfiguration getMethodConfiguration() {
		return config;
	}

	/**
	 * Gets the containing method library for the managed method configuration.
	 * 
	 * @return a method library
	 */
	public MethodLibrary getMethodLibrary() {
		return library;
	}

	/**
	 * Gets the adapter factory.
	 * 
	 * @return an adapter factory
	 */
	public AdapterFactory getAdapterFactory() {
		return libraryManager.getAdapterFactory();
	}

	/**
	 * Gets the adapter factory content provider.
	 * 
	 * @return an adapter factory content provider
	 */
	public AdapterFactoryContentProvider getContentProvider() {
		return afcp;
	}

	/**
	 * Gets the dependency manager.
	 * 
	 * @return a dependency manager
	 */
	public DependencyManager getDependencyManager() {
		return dependencyManager;
	}

	/**
	 * Returns the element layout manager.
	 * 
	 * @return an element layout manager
	 */
	public ElementLayoutManager getElementLayoutManager() {
		return layoutManager;
	}

	/**
	 * Gets the configuration closure.
	 * 
	 * @return a configuration closure
	 */
	public ConfigurationClosure getConfigurationClosure() {
		return closure;
	}

	/**
	 * Makes a configuration closure.
	 */
	public void makeConfigurationClosure() {
		if (closure != null) {
			closure.makeClosure();
		}
	}

	/**
	 * Returns a list of method plug-ins in the containing method library.
	 * 
	 * @return a list of method plug-ins
	 */
	public List getMethodPlugins() {
		return library.getMethodPlugins();
	}

	/**
	 * Disposes all resources allocated by this configuration manager.
	 */
	public void dispose() {
		config = null;
		library = null;
		libraryManager = null;
		dependencyManager = null;
		layoutManager = null;
		closure = null;
	}

}
