//------------------------------------------------------------------------------
// Copyright (c) 2005, 2006 IBM Corporation and others.
// All rights reserved. This program and the accompanying materials
// are made available under the terms of the Eclipse Public License v1.0
// which accompanies this distribution, and is available at
// http://www.eclipse.org/legal/epl-v10.html
//
// Contributors:
// IBM Corporation - initial implementation
//------------------------------------------------------------------------------
package org.eclipse.epf.library.configuration;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * @author Jinhua Xi
 * @since 1.0
 */
public class ElementDependencyError {

	private Object errorElement;

	/**
	 * map of causeelement to the ErrorInfo object
	 */
	private Map errorInfoMap = new HashMap();

	private int errorBits = 0;

	public ElementDependencyError(Object element) {
		this.errorElement = element;
	}

	public Object getErrorElement() {
		return errorElement;
	}

	public Object[] getCauseElements() {
		return errorInfoMap.keySet().toArray();
	}

	public int size() {
		return errorInfoMap.size();
	}

	public void addError(ErrorInfo errorInfo) {
		errorInfoMap.put(errorInfo.getCauseElement(), errorInfo);
		errorBits |= errorInfo.getErrorType();
	}

	/**
	 * remove the error associated with the causeElement
	 * 
	 * @param causeElement
	 *            Object the element that caused the error
	 */
	public void removeError(Object causeElement) {
		errorInfoMap.remove(causeElement);

		// recalculate the error bits
		calculateErrorBits();
	}

	/**
	 * remove the error associated with the causeElement
	 * 
	 * @param causeElement
	 *            Object the element that caused the error
	 */
	public ErrorInfo getError(Object causeElement) {
		return (ErrorInfo) errorInfoMap.get(causeElement);
	}

	/**
	 * get a list of all ErrInfo objects
	 * 
	 * @return List a list of ErrorInfo objects
	 */
	public List getAll() {
		return new ArrayList(errorInfoMap.values());
	}

	public boolean isError() {
		return (errorBits & ErrorInfo.ERROR) > 0;
	}

	public boolean isWarning() {
		return (errorBits & ErrorInfo.WARNING) > 0;
	}

	public boolean isChildError() {
		return (errorBits & ErrorInfo.CHILD_ERROR) > 0;
	}

	public boolean isChildWarning() {
		return (errorBits & ErrorInfo.CHILD_WARNING) > 0;
	}

	private void calculateErrorBits() {
		errorBits = 0;
		for (Iterator it = errorInfoMap.values().iterator(); it.hasNext();) {
			errorBits |= ((ErrorInfo) it.next()).getErrorType();
		}
	}

}
