//------------------------------------------------------------------------------
// Copyright (c) 2005, 2006 IBM Corporation and others.
// All rights reserved. This program and the accompanying materials
// are made available under the terms of the Eclipse Public License v1.0
// which accompanies this distribution, and is available at
// http://www.eclipse.org/legal/epl-v10.html
//
// Contributors:
// IBM Corporation - initial implementation
//------------------------------------------------------------------------------
package org.eclipse.epf.richtext.html;

import java.util.*;

import org.eclipse.epf.richtext.RichTextResources;

/**
 * Models a HTML font name.
 * 
 * @author Kelvin Low
 * @since 1.0
 */
public class FontName {

	// The user friendly names.
	private static final String NAME_DEFAULT = RichTextResources.fontName_default; //$NON-NLS-1$

	private static final String NAME_ARIAL = RichTextResources.fontName_arial; //$NON-NLS-1$

	private static final String NAME_COURIER_NEW = RichTextResources.fontName_courierNew; //$NON-NLS-1$

	private static final String NAME_TIMES_NEW_ROMAN = RichTextResources.fontName_timesNewRoman; //$NON-NLS-1$

	private static final String NAME_VERDANA = RichTextResources.fontName_verdana; //$NON-NLS-1$

	// The internal values.
	private static final String VALUE_DEFAULT = ""; //$NON-NLS-1$

	private static final String VALUE_ARIAL = "Arial, Helvetica, sans-serif"; //$NON-NLS-1$

	private static final String VALUE_COURIER_NEW = "Courier New, Courier, mono"; //$NON-NLS-1$

	private static final String VALUE_TIMES_NEW_ROMAN = "Times New Roman, Times, serif"; //$NON-NLS-1$

	private static final String VALUE_VERDANA = "Verdana, Arial, Helvetica, sans-serif"; //$NON-NLS-1$

	static public final FontName DEFAULT = new FontName(NAME_DEFAULT,
			VALUE_DEFAULT);

	/**
	 * Enumeration of all supported <code>HTMLBlockTag</code>.
	 */
	static public final FontName ARIAL = new FontName(NAME_ARIAL, VALUE_ARIAL);

	static public final FontName COURIER_NEW = new FontName(NAME_COURIER_NEW,
			VALUE_COURIER_NEW);

	static public final FontName TIMES_NEW_ROMAN = new FontName(
			NAME_TIMES_NEW_ROMAN, VALUE_TIMES_NEW_ROMAN);

	static public final FontName VERDANA = new FontName(NAME_VERDANA,
			VALUE_VERDANA);

	// A list of <code>FontName</code> objects.
	static private final List FONT_NAMES = new ArrayList();
	static {
		FONT_NAMES.add(DEFAULT);
		FONT_NAMES.add(ARIAL);
		FONT_NAMES.add(COURIER_NEW);
		FONT_NAMES.add(TIMES_NEW_ROMAN);
		FONT_NAMES.add(VERDANA);
	}

	// The font name.
	private String name;

	// The font value.
	private String value;

	/**
	 * Creates a new instance.
	 * 
	 * @param name
	 *            the font name
	 * @param value
	 *            the font value
	 */
	public FontName(String name, String value) {
		this.name = name;
		this.value = value;
	}

	/**
	 * Returns the font name.
	 * 
	 * @return the font name
	 */
	public String getName() {
		return name;
	}

	/**
	 * Returns the font value.
	 * 
	 * @return the font value
	 */
	public String getValue() {
		return value;
	}

	/**
	 * Returns the <code>FontName</code> object that is mapped to the given
	 * index.
	 * 
	 * @param index
	 *            an index into the <code>FontName</code> list
	 * @return a <code>FontName</code> object
	 */
	public static FontName getFontName(int index) {
		FontName result = (FontName) FONT_NAMES.get(index);
		if (result != null) {
			return result;
		}
		return DEFAULT;
	}

}