//------------------------------------------------------------------------------
// Copyright (c) 2005, 2006 IBM Corporation and others.
// All rights reserved. This program and the accompanying materials
// are made available under the terms of the Eclipse Public License v1.0
// which accompanies this distribution, and is available at
// http://www.eclipse.org/legal/epl-v10.html
//
// Contributors:
// IBM Corporation - initial implementation
//------------------------------------------------------------------------------
package org.eclipse.epf.richtext.html;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.epf.richtext.RichTextResources;

/**
 * Models a HTML font style.
 * 
 * @author Kelvin Low
 * @since 1.0
 */
public class FontStyle {

	// The user friendly names.
	private static final String NAME_NORMAL = RichTextResources.fontStyle_normal; //$NON-NLS-1$

	private static final String NAME_SECTION_HEADING = RichTextResources.fontStyle_sectionHeading; //$NON-NLS-1$

	private static final String NAME_SUBSECTION_HEADING = RichTextResources.fontStyle_subsectionHeading; //$NON-NLS-1$

	private static final String NAME_SUB_SUBSECTION_HEADING = RichTextResources.fontStyle_subSubsectionHeading; //$NON-NLS-1$

	private static final String NAME_QUOTE = RichTextResources.fontStyle_quote; //$NON-NLS-1$

	private static final String NAME_CODE_SAMPLE = RichTextResources.fontStyle_codeSample; //$NON-NLS-1$

	// The internal values.
	private static final String VALUE_NORMAL = "<p>"; //$NON-NLS-1$

	private static final String VALUE_SECTION_HEADING = "<h3>"; //$NON-NLS-1$

	private static final String VALUE_SUBSECTION_HEADING = "<h4>"; //$NON-NLS-1$

	private static final String VALUE_SUB_SUBSECTION_HEADING = "<h5>"; //$NON-NLS-1$

	private static final String VALUE_QUOTE = "<quote>"; //$NON-NLS-1$

	private static final String VALUE_CODE_SAMPLE = "<code>"; //$NON-NLS-1$

	/**
	 * Enumeration of all supported <code>FontStyle</code>.
	 */
	static public final FontStyle NORMAL = new FontStyle(NAME_NORMAL,
			VALUE_NORMAL);

	static public final FontStyle SECTION_HEADING = new FontStyle(
			NAME_SECTION_HEADING, VALUE_SECTION_HEADING);

	static public final FontStyle SUBSECTION_HEADING = new FontStyle(
			NAME_SUBSECTION_HEADING, VALUE_SUBSECTION_HEADING);

	static public final FontStyle SUB_SUBSECTION_HEADING = new FontStyle(
			NAME_SUB_SUBSECTION_HEADING, VALUE_SUB_SUBSECTION_HEADING);

	static public final FontStyle QUOTE = new FontStyle(NAME_QUOTE, VALUE_QUOTE);

	static public final FontStyle CODE_SAMPLE = new FontStyle(NAME_CODE_SAMPLE,
			VALUE_CODE_SAMPLE);

	// A list of <code>FontStyle</code> objects.
	static private final List FONT_STYLES = new ArrayList();
	static {
		FONT_STYLES.add(NORMAL);
		FONT_STYLES.add(SECTION_HEADING);
		FONT_STYLES.add(SUBSECTION_HEADING);
		FONT_STYLES.add(SUB_SUBSECTION_HEADING);
		FONT_STYLES.add(QUOTE);
		FONT_STYLES.add(CODE_SAMPLE);
	}

	// The font style name.
	private String name;

	// The font style value.
	private String value;

	/**
	 * Creates a new instance.
	 * 
	 * @param name
	 *            the font style name
	 * @param value
	 *            the font style value
	 */
	public FontStyle(String name, String value) {
		this.name = name;
		this.value = value;
	}

	/**
	 * Returns the font style name.
	 * 
	 * @return the font style name
	 */
	public String getName() {
		return name;
	}

	/**
	 * Returns the font style value.
	 * 
	 * @return the font style value
	 */
	public String getValue() {
		return value;
	}

	/**
	 * Returns the <code>FontStyle</code> object that is mapped to the given
	 * index.
	 * 
	 * @param index
	 *            an index into the <code>FontStyle</code> list
	 * @return a <code>FontStyle</code> object
	 */
	public static FontStyle getFontStyle(int index) {
		FontStyle result = (FontStyle) FONT_STYLES.get(index);
		if (result != null) {
			return result;
		}
		return NORMAL;
	}

}