//------------------------------------------------------------------------------
// Copyright (c) 2005, 2006 IBM Corporation and others.
// All rights reserved. This program and the accompanying materials
// are made available under the terms of the Eclipse Public License v1.0
// which accompanies this distribution, and is available at
// http://www.eclipse.org/legal/epl-v10.html
//
// Contributors:
// IBM Corporation - initial implementation
//------------------------------------------------------------------------------
package org.eclipse.epf.search.ui;

import org.eclipse.epf.library.LibraryService;
import org.eclipse.epf.search.ui.internal.MethodSearchInput;
import org.eclipse.epf.search.ui.internal.MethodSearchQuery;
import org.eclipse.epf.search.ui.internal.MethodSearchScope;
import org.eclipse.epf.search.ui.internal.MethodSearchScopeViewer;
import org.eclipse.epf.search.ui.preferences.SearchUIPreferences;
import org.eclipse.epf.uma.MethodLibrary;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.DialogPage;
import org.eclipse.search.ui.ISearchPage;
import org.eclipse.search.ui.ISearchPageContainer;
import org.eclipse.search.ui.NewSearchUI;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;

/**
 * The Method Search page.
 * 
 * @author Kelvin Low
 * @since 1.0
 */
public class MethodSearchPage extends DialogPage implements ISearchPage {

	public static final String SEARCH_PAGE_ID = MethodSearchPage.class
			.getName();

	private Combo searchStringCombo;

	private Combo namePatternCombo;

	private Button caseSensitiveCheckbox;

	private MethodSearchScopeViewer searchScopeViewer;

	private ISearchPageContainer container;

	/**
	 * Creates a new instance.
	 */
	public MethodSearchPage() {
		super();
	}

	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(Composite)
	 */
	public void createControl(Composite parent) {
		initializeDialogUnits(parent);

		Composite composite = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout(2, false);
		layout.horizontalSpacing = 10;
		composite.setLayout(layout);
		composite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		Label searchStringLabel = new Label(composite, SWT.NONE);
		searchStringLabel.setText(SearchUIResources.searchStringLabel_text); //$NON-NLS-1$

		new Label(composite, SWT.NONE);

		searchStringCombo = new Combo(composite, SWT.NONE);
		searchStringCombo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		String savedSearchString = SearchUIPreferences.getSearchString();
		if (savedSearchString != null && savedSearchString.length() > 0) {
			searchStringCombo.setText(savedSearchString);
		}

		caseSensitiveCheckbox = new Button(composite, SWT.CHECK);
		caseSensitiveCheckbox.setText(SearchUIResources.caseSensitiveCheckbox_text); //$NON-NLS-1$
		caseSensitiveCheckbox.setSelection(SearchUIPreferences
				.getCaseSensitive());

		Label elementNameLabel = new Label(composite, SWT.NONE);
		elementNameLabel.setText(SearchUIResources.elementNameLabel_text); //$NON-NLS-1$

		new Label(composite, SWT.NONE);

		namePatternCombo = new Combo(composite, SWT.NONE);
		namePatternCombo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		String savedNamePattern = SearchUIPreferences.getNamePattern();
		if (savedNamePattern != null && savedNamePattern.length() > 0) {
			namePatternCombo.setText(savedNamePattern);
		} else {
			namePatternCombo.setText("*"); //$NON-NLS-1$
		}

		namePatternCombo.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				container.setPerformActionEnabled(getSearchButtonEnabled());
			}
		});
		namePatternCombo.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent event) {
				container.setPerformActionEnabled(getSearchButtonEnabled());
			}
		});

		new Label(composite, SWT.NONE);

		Group searchScopeGroup = new Group(composite, SWT.NONE);
		searchScopeGroup.setLayout(new GridLayout(1, false));
		GridData searchScopeGroupGridData = new GridData(GridData.FILL_BOTH);
		searchScopeGroupGridData.heightHint = 200;
		searchScopeGroup.setLayoutData(searchScopeGroupGridData);
		searchScopeGroup.setText(SearchUIResources.scopeGroup_text); //$NON-NLS-1$

		searchScopeViewer = new MethodSearchScopeViewer(searchScopeGroup,
				SWT.NONE);

		container.setPerformActionEnabled(getSearchButtonEnabled());

		setControl(composite);
		Dialog.applyDialogFont(composite);
	}

	/**
	 * @see org.eclipse.search.ui.ISearchPage#performAction()
	 */
	public boolean performAction() {
		MethodLibrary library = LibraryService.getInstance().getCurrentMethodLibrary();
		if (library == null) {
			SearchUIPlugin.getDefault().getMsgDialog().displayError(
					SearchUIResources.searchError_title, 
					SearchUIResources.searchError_msg, 
					SearchUIResources.searchError_reason); 
			return false;
		}
		String searchString = searchStringCombo.getText();
		String namePattern = namePatternCombo.getText();
		MethodSearchScope searchScope = searchScopeViewer.getSearchScope();
		MethodSearchInput searchInput = new MethodSearchInput(searchString,
				namePattern, caseSensitiveCheckbox.getSelection(), false,
				false, searchScope);
		MethodSearchQuery searchQuery = new MethodSearchQuery(searchInput);
		NewSearchUI.activateSearchResultView();
		NewSearchUI.runQueryInBackground(searchQuery);
		SearchUIPreferences.setSearchString(searchStringCombo.getText());
		SearchUIPreferences.setNamePattern(namePatternCombo.getText());
		SearchUIPreferences.setCaseSensitive(caseSensitiveCheckbox
				.getSelection());
		return true;
	}

	/**
	 * @see org.eclipse.search.ui.ISearchPage#setContainer(ISearchPageContainer)
	 */
	public void setContainer(ISearchPageContainer container) {
		this.container = container;
	}

	/**
	 * @see org.eclipse.search.ui.IReplacePage#performReplace()
	 */
	public boolean performReplace() {
		return true;
	}

	/**
	 * @see org.eclipse.jface.dialogs.DialogPage#dispose()
	 */
	public void dispose() {
		super.dispose();
	}

	/**
	 * Gets the enabled status of the Search button.
	 * 
	 * @return <code>true<code> if the Search button should be enabled
	 */
	private boolean getSearchButtonEnabled() {
		return namePatternCombo.getText().trim().length() > 0;
	}

}
