//------------------------------------------------------------------------------
// Copyright (c) 2005, 2006 IBM Corporation and others.
// All rights reserved. This program and the accompanying materials
// are made available under the terms of the Eclipse Public License v1.0
// which accompanies this distribution, and is available at
// http://www.eclipse.org/legal/epl-v10.html
//
// Contributors:
// IBM Corporation - initial implementation
//------------------------------------------------------------------------------
package org.eclipse.epf.uma.util;

import java.util.Collection;
import java.util.List;

import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.epf.uma.MethodElement;

/**
 * The interface for a Method Library Persister.
 * <p>
 * A Method Library Persister is responsible for persisting the method library
 * content.
 * 
 * @author Phong Nguyen Le
 * @since 1.0
 */
public interface IMethodLibraryPersister {

	/**
	 * Saves an existing resource.
	 * 
	 * @param resource
	 *            a resource
	 */
	public void save(Resource resource) throws Exception;
	
	/**
	 * Saves the given MethodElement in its own file. The specified MethodElement must be part of a library
	 * and it is allowed to have its own resource, to which no other resource refers. In other words, calling 
	 * {@link #hasOwnResourceWithoutReferrer(Object)} on the given MethodElement must return <code>true</code>.
	 * 
	 * @param element
	 * @throws Exception
	 */
	public void save(MethodElement element) throws Exception;

	/**
	 * Deletes the files associated with a method element.
	 * 
	 * @param e
	 *            a method element
	 */
	public void delete(MethodElement e);

	/**
	 * Adjusts the location of a resource and saves all the resources that have
	 * been changed as the result of this adjustment.
	 * 
	 * @param resource
	 *            a resource
	 */
	public void adjustLocation(Resource resource);

	/**
	 * Gets the path of a method element's folder relative to its plug-in or
	 * library folder.
	 * 
	 * @param e
	 *            a method element
	 * @return a relative path name
	 */
	public String getFolderRelativePath(MethodElement e);

	/**
	 * Gets the file extension of a method element's resource file.
	 * 
	 * @param e
	 *            a method element
	 * @return a file extension name
	 */
	public String getFileExtension(Object e);

	/**
	 * Gets a list of warnings associated with a resource.
	 * 
	 * @return a list of <code>Exception</code> objects
	 */
	public List getWarnings();

	/**
	 * Checks whether a method element can have its own resource.
	 * 
	 * @param e
	 *            a method element
	 * @return <code>true</code> if the method element is owned by a resource
	 */
	public boolean hasOwnResource(Object e);

	/**
	 * Checks whether a method element stores its content in a folder named
	 * after it.
	 * 
	 * @param e
	 *            a method element
	 * @return <code>true</code> if the method element stores its content in a
	 *         folder named after it
	 */
	public boolean hasOwnFolder(Object e);
	
	/**
	 * Checks whether a method element can have its own resource, which no other resource refers to.
	 * 
	 * @param e
	 * @return
	 */
	public boolean hasOwnResourceWithoutReferrer(Object e);

	/**
	 * Gets the fail safe method library persister associated with this library
	 * persister.
	 * 
	 * @return a fail safe method library persister
	 */
	public FailSafeMethodLibraryPersister getFailSafePersister();

	public static interface FailSafeMethodLibraryPersister extends
			IMethodLibraryPersister {

		/**
		 * Gets the current transaction ID.
		 * 
		 * @return the current transaction ID
		 */
		String getCurrentTxID();

		/**
		 * Adjusts the location of the given resource and save all the resources
		 * that have been changed as the result of this adjustment. This call
		 * needs a new transaction and the transaction will be committed in this
		 * method.
		 * 
		 * @param resources
		 *            the resources whose locations need to be adjusted
		 */
		void adjustLocation(Collection resources);

		/**
		 * Commits the transaction.
		 */
		void commit();

		/**
		 * Roll backs the transaction.
		 */
		void rollback();
	}

}