# A Site can be a BaselineProcess (static, published website from EPF) or a Wiki, 
# which is an enhanced EPFC site. Baseline Processes are 
# used to create or update Wiki sites.
#
# Creation or update of a Wiki is a two step process for performance reasons.
# This way the second step can be performed using a job that runs at night.
# 
# More information:
# * {EPF Wiki Data model}[link:files/doc/DATAMODEL.html]
#--######################################################################
# Copyright (c) 2006 Logica
# 
# All rights reserved. This program and the accompanying materials
# are made available under the terms of the Eclipse Public License v1.0
# which accompanies this distribution, and is available at
# http://www.eclipse.org/legal/epl-v10.html
#
# Contributors:
# 
# Onno van der Straaten:: initial implementation
#++######################################################################
# {Copyright (c) 2006 Logica}[link:files/COPYRIGHT.html]

class Site < ActiveRecord::Base
  
  belongs_to :user
  has_many :pages
 
  validates_presence_of  :user_id, :title, :folder # TODO validate :user not :user_id
  
  # TODO Folde can contain . (dot) because it is created from a filename
  validates_format_of :folder, :message => 'should consist of letters, digits and underscores', :with =>  /([-_.\dA-Za-z])*/  
  
  # A wikifiable file is a HTML file
  HTML_FILE_PATTERN = /.*.htm(l)?/i
  
  # A wikifiable is not a Wiki file (a version file created using the Wiki)
  WIKI_FILE_PATTERN = /(.)*wiki(.)*/i
  
   def content_scanned?
    return !self.content_scanned_on.nil?
  end
  
  def baseline_processes_candidate
    returning bp_candidate = [] do
      if self.status == 'Ready' # TODO
        Site.find_baseline_processes.each do |bp|
          bp_candidate << bp unless self.baselines.include?(bp.baseline)
        end
      end    
    end
  end
  
  def wiki?
    return self.class.name == 'Wiki'
  end
  
  def baseline_process?
    return self.class.name == 'BaselineProcess'
  end
  
  def path 
    return "#{ENV['EPFWIKI_ROOT_DIR']}#{ENV['EPFWIKI_PUBLIC_FOLDER']}/#{ENV['EPFWIKI_WIKIS_FOLDER']}/#{self.folder}" if self.wiki?
    return "#{ENV['EPFWIKI_ROOT_DIR']}#{ENV['EPFWIKI_PUBLIC_FOLDER']}/#{ENV['EPFWIKI_SITES_FOLDER']}/#{self.folder}"
  end
  
  def rel_path
    return self.path.gsub(ENV['EPFWIKI_ROOT_DIR'] + "#{ENV['EPFWIKI_PUBLIC_FOLDER']}/",'')
  end
  
  def path2zip 
    return self.path + '.zip'
  end
  
  # Method #templates returns current versions of pages from the Wiki 'Templates'. 
  # These pages are templates for creating new pages. 
  def self.templates
    w = Wiki.find_by_title('Templates')
    raise "No Templates Wiki was found. There should always be a Wiki with title 'Templates' to provide templates for creating new pages" if !w
    returning tpls = [] do
      w.pages.each do |p|
        tpls << p.current_version if p.current_version # NOTE: a new checked out Template has no current version
      end
    end
  end
  
  def unzip_upload
    logger.info("Unzipping upload #{self.path2zip}")
    raise "Folder #{self.path} exists" if File.exists?(self.path)
    cmdline = "unzip -q \"#{self.path2zip}\" -d \"#{self.path}\""
    logger.debug("Executing command #{cmdline}")
    unzip_success =  system(cmdline)
    raise "Error executing command #{cmdline}: #{$?}" if !unzip_success
  end
  
  # array of HTML files that are candate Wikification
  def self.files_html(path)
    paths = Array.new
     (Dir.entries(path) - [".", ".."]).each do |entry| 
      new_path = File.expand_path(entry, path)
      if FileTest.directory?(new_path)   
        paths = paths + Site.files_html(new_path)
      else
        paths << new_path if !HTML_FILE_PATTERN.match(entry).nil? && WIKI_FILE_PATTERN.match(entry).nil?
      end
    end
    return paths
  end
  
  def files_wikifiable
    raise 'Path can\'t be blank' if self.path.blank?
    logger.info("Finding wikifiable files in #{self.path}")
    returning paths = [] do
      Site.files_html(self.path).each do |path2|
        #logger.info("#{path2}") 
        paths << path2 unless Page::ELEMENT_TYPE_PATTERN.match(IO.readlines(path2).join("\n")).nil?
      end
    end
  end
  
  ###########
  private 
  ###########
  
 
  # action #copy_to copies the content of a site to another site (theDestSite). Files are overwritten if they exist in the destination site.
  # NOTE: Ruby does not have a copy + overwrite command?
  def copy_to(theDestSite, theFolderPath = nil)
    if  theFolderPath
     (Dir.entries(theFolderPath) - [".", ".."]).each do |aEntry|
        aPath = File.expand_path(aEntry, theFolderPath)
        aDestPath = aPath.gsub(self.path, theDestSite.path)
        if  FileTest.directory?(aPath)
          logger.info("Copying folder " + aPath + " to " + aDestPath)
          File.makedirs(aDestPath)
          copy_to(theDestSite, aPath)
        else
          if  !FileTest.exists?(aDestPath)
            logger.info("New file copied " + aPath + " to " + aDestPath)
            File.copy(aPath, aDestPath)
          else
            if  FileUtils.cmp(aPath, aDestPath)
              logger.info("Not copied because equal: " + aPath)
            else
              logger.info("Overwritten: " + aPath)
              File.delete(aDestPath) 
              File.copy(aPath, aDestPath)
            end
          end
        end
      end 
    else
      logger.info("Copying content from site " + self.title + " to " + theDestSite.title)
      logger.info("Source folder: " + self.path + ". Destination folder: " + theDestSite.path)
      copy_to(theDestSite, self.path)
    end
  end
  
end
