require File.dirname(__FILE__) + '/../test_helper'
require 'openssl'
require 'digest/sha1'

#--######################################################################
# Copyright (c) 2006 Logica
# 
# All rights reserved. This program and the accompanying materials
# are made available under the terms of the Eclipse Public License v1.0
# which accompanies this distribution, and is available at
# http://www.eclipse.org/legal/epl-v10.html
#
# Contributors:
# 
# Onno van der Straaten:: initial implementation
#++######################################################################
# {Copyright (c) 2006 Logica}[link:files/COPYRIGHT.html]

class UserTest < Test::Unit::TestCase
  
  fixtures :users
  
  def setup
    logger.debug "Test Case: #{name}"  
    @emails = ActionMailer::Base::deliveries
    @emails.clear
    @andy, @cash, @tony, @george = users(:andy), users(:cash), users(:tony), users(:george)
  end
  
  def test_create
    assert_kind_of User, @andy
    #assert_equal 1, @andy.id
    assert_equal "andy.kaufman@epf.eclipse.org", @andy.email
    assert_equal "Andy Kaufman", @andy.name
    assert_equal "localhost", @andy.ip_address
    assert_equal hash_pw(@andy.name), @andy.hashed_password
    assert_equal "Y", @andy.admin
  end
  
  def test_new_signup
    assert User.count > 0
    assert_equal "0", ENV['EPFWIKI_GENERATE_PASSWORDS']
    assert_not_nil ENV['EPFWIKI_DOMAINS']
    # we need params
    user = User.new_signup({}) 
    assert !user.save
    assert_equal "Name can't be blank, Password confirmation can't be blank, Password can't be blank, Email can't be blank, Email is invalid, Email domain not valid", user.errors.full_messages.join(", ")
    # password needs to be confirmed
    user = User.new_signup({:name => "User10", :password => "User10", :email => "User10@epf.eclipse.org"}) #, :i_agree_to_the_terms_of_use => '1'
    assert !user.save
    assert_equal "Password confirmation can't be blank", user.errors.full_messages.join(", ")    
    # password needs to be confirmed 2
    user = User.new_signup({:name => "User10", :password => "User10", :password_confirmation => "xyz", :email => "User10@epf.eclipse.org"})  # , :i_agree_to_the_terms_of_use => '1'
    assert !user.save
    assert_equal "Password doesn't match confirmation", user.errors.full_messages.join(", ")    
    # user created
    user = User.new_signup({:name => "User10", :password => "User10", :password_confirmation => "User10", :email => "User10@epf.eclipse.org"})  # , :i_agree_to_the_terms_of_use => '1'
    assert user.save
    assert_equal "user10@epf.eclipse.org", user.email
    assert_equal nil, user.confirmed_on # account needs to be confirmed
    assert_equal hash_pw('User10'), user.hashed_password
    assert_equal nil, user.hashed_password_new
    # cannot login, not confirmed
    login_user = user.try_to_login
    assert_equal nil, login_user
    # confirm account
    # assert_equal hash_pw('User10'), user.hashed_password 
    user.confirm_account(hash_pw(user.hashed_password))  
    assert user.save
    assert_not_nil user.confirmed_on
    # can login
    login_user = user.try_to_login
    assert_not_nil login_user
  end
  
  def test_set_new_pw
    # user created
    user = User.new_signup({:name => "User11", :password => "User11", :password_confirmation => "User11", :email => "User11@epf.eclipse.org"})  # , :i_agree_to_the_terms_of_use => '1'
    assert user.save
    # confirm account
    user.confirm_account(hash_pw(user.hashed_password))  
    assert user.save
    assert_equal "", user.errors.full_messages.join(", ")    
    user = User.find_by_name('User11')
    assert_not_nil user.confirmed_on
    # can login
    user.password = 'User11'
    login_user = user.try_to_login
    assert_not_nil login_user
    # set new password  
    hashed_password = user.hashed_password
    user.set_new_pw
    new_pw = user.password
    assert user.save
    assert_equal "", user.errors.full_messages.join(", ")
    user.reload
    assert_not_nil user.confirmed_on 
    assert_equal hash_pw(new_pw), user.hashed_password_new
    # we can still sign in with the old password
    user.password = "User11"
    login_user = user.try_to_login
    assert_not_nil login_user
    # we cannot sign in with the new password
    user.password = new_pw
    login_user = user.try_to_login
    assert_equal nil, login_user
    # cannot confirm with the wrong token
    user = User.find_by_name('User11')
    assert_equal false, user.confirm_account("somewrongtoken")
    # confirm the account
    user = User.find_by_name('User11')
    #assert_equal hash_pw(hash_pw(new_pw)), hash_pw(user.hashed_password_new)
    user.confirm_account(hash_pw(hash_pw(new_pw)))
    assert_equal hash_pw(new_pw), user.hashed_password
    user.save
    assert_equal "", user.errors.full_messages.join(", ")
    user = User.find_by_name('User11')
    assert_not_equal hashed_password, user.hashed_password
    assert_equal hash_pw(new_pw), user.hashed_password
    assert_equal nil, user.hashed_password_new
    assert_not_nil user.confirmed_on
    # we can sign in with the new password
    user.password = new_pw
    login_user = user.try_to_login
    assert_not_nil login_user
  end
  
  def test_updates
    user = User.find_by_name('Andy Kaufman')
    user.name = "test04_updates"
    assert user.save
    assert_equal "", user.errors.full_messages.join(", ")
  end
  
  def test_change_password
    user = User.find_by_name('User2')
    assert_raise(RuntimeError) {user.change_password(User.new)}
    assert_raise(RuntimeError) {user.change_password(User.new(:password =>'', :password_confirmation => ''))}
    user.change_password(User.new(:password =>'xyz', :password_confirmation => '123'))    
    assert !user.save
    assert_equal "Password doesn't match confirmation", user.errors.full_messages.join(", ")
    user.change_password(User.new(:password =>'xyz', :password_confirmation => 'xyz'))    
    assert user.save
    assert_equal '', user.errors.full_messages.join(', ')
    user = User.find_by_name('User2')
    user.password = 'xyz'
    login_user = user.try_to_login
    assert_equal hash_pw('xyz'), login_user.hashed_password
    assert_not_nil login_user
  end
  
  # Shows:
  # 1 User can logon using basic authentication, first logon -> an account is created
  # 2 Second logon succesfull, account is not created
  # 3 Wrong password cannot logon
  # 4 Failed to save, sends email
  def tst_login_basicauthentication
    # get password
    pw = IO.readlines('S:/Keys/epfwiki_basic_authentication_key')[0]
    # 1
    ENV['EPFWIKI_DOMAINS'] = ENV['EPFWIKI_DOMAINS'] + ' @logicacmg.com'
    user_count = User.count
    user = User.new(:account => 'ostraaten', :password => pw)
    logon_user = User.login(user.account, user.password)
    assert_equal user_count + 1, User.count
    # 2
    logon_user = User.login(user.account, user.password)
    assert_not_nil logon_user
    assert_equal user_count + 1, User.count
    # 3
    logon_user = User.login(user.account, user.password + 'xyz')
    assert_nil logon_user
    assert_equal user_count + 1, User.count
    # 4
    User.find_by_account(user.account).destroy
    assert_equal user_count, User.count
    ActiveRecord::Migration::drop_column 'users', 'account'
    assert_equal 0,@emails.size
    logon_user = nil
    logon_user = User.login(user.account, user.password)
    assert_nil logon_user
    assert_equal user_count, User.count
  end
  
  # Shows:
  # 1. Cannot update a user to Y or C without specifying the user
  # 2. Cadmin can upgrade user to admin, downgrade to user, admin kan upgrade user to admin but not downgrade admin to user
  # 3. Cadmin can make another user the cadmin
  # . C -> Y or C -> N not possible
  def test_admin
    cadmin = User.find_central_admin
    user = User.find_all_by_admin('N')[0]
    admin = User.find_all_by_admin('Y')[0]
    assert_not_nil cadmin
    assert_not_nil user
    assert_not_nil admin
    user.admin = 'Y'
    assert !user.save
    assert_equal 'Admin can only be set by an admin', user.errors.full_messages.join(", ")
    user.admin = 'C'
    assert !user.save
    assert_equal 'Admin can only be set to C by the central admin', user.errors.full_messages.join(", ")
    # 2
    user.admin = 'Y' 
    user.user = cadmin
    assert user.save
    user.admin = 'N' 
    user.user = cadmin
    assert user.save
    user.admin = 'Y' 
    user.user = admin
    assert user.save
    user.admin = 'N' 
    user.user = admin
    assert !user.save
    assert_equal 'Admin can only be revoked by the central admin', user.errors.full_messages.join(", ")    
    user.user = cadmin
    assert user.save
    # 3
    assert cadmin.cadmin?
    User.cadmin(cadmin, user) 
    user.save
    cadmin.save
    assert_equal '', user.errors.full_messages.join(", ")        
    assert_equal '', cadmin.errors.full_messages.join(", ")            
    user.reload
    cadmin.reload
    assert user.cadmin?
    assert !cadmin.cadmin?
    assert_equal 'Y', cadmin.admin
    assert_equal 'C', user.admin
  end

  # Shows:
  # 1 User can logon using bugzilla authentication, on first logon an account is created
  # 2 Second logon succesfull, account is not created
  # 3 Wrong password cannot logon
  # 4 User tries to switch validemail to Bugzilla authentication when using the same email -> TODO
  def tst_login_bugzilla 
    # get password
    pw = IO.readlines('S:/Keys/epfwiki_bugzilla_authentication_key')[0]
    # 1
    ENV['EPFWIKI_DOMAINS'] = ENV['EPFWIKI_DOMAINS'] + ' @logicacmg.com'
    user_count = User.count
    user = User.new(:email => 'onno.van.der.straaten@logicacmg.com', :password => pw)
    logon_user = User.login(user.email, user.password)
    assert_equal user_count + 1, User.count # TODO fails
    # 2
    logon_user = User.login(user.email, user.password)
    assert_not_nil logon_user
    assert_equal user_count + 1, User.count
    # 3
    logon_user = User.login(user.email, user.password + 'xyz')
    assert_nil logon_user
    assert_equal user_count + 1, User.count
    # 4
  end
  
  # Shows:
  # 1. When the central admin is created, the 'Templates' Wiki is created
  def test_new_cadmin
    params = {:name => 'onno', :email => 'Onno@epf.eclipse.org', :password => 'xyz', :password_confirmation => 'xyz'}
    # cannot create cadmin if there are users
    assert_raise(RuntimeError) {cadmin = User.new_cadmin(params)}
    User.delete_all
    # params are needed
    cadmin = User.new_cadmin({})
    assert !cadmin.save
    assert_equal "Name can't be blank, Password confirmation can't be blank, Password can't be blank, Email can't be blank, Email is invalid", cadmin.errors.full_messages.join(", ")
    # password needs to be confirmed    
    cadmin = User.new_cadmin(:name => 'onno', :email => 'Onno@epf.eclipse.org', :password => 'xyz', :password_confirmation => '123')
    assert !cadmin.save
    assert_equal "Password doesn't match confirmation", cadmin.errors.full_messages.join(", ")
    # valid email is required
    cadmin = User.new_cadmin(:name => 'onno', :email => 'Onno(at)epf.org', :password => 'xyz', :password_confirmation => 'xyz')
    assert !cadmin.save
    assert_equal "Email is invalid", cadmin.errors.full_messages.join(", ")
    # cadmin is created, note domain restriction does not apply to cadmin account
    cadmin = User.new_cadmin(:id => 5, :name => 'onno', :email => 'Onno@noneExistingDomain.Com', :password => 'xyz', :password_confirmation => 'xyz')
    assert cadmin.save 
    assert_equal 26, Site.templates.size
    # 1
    assert_equal "", cadmin.errors.full_messages.join(", ")
    assert_equal 'onno@noneexistingdomain.com', cadmin.email # email set to downcase
    assert_not_nil cadmin.hashed_password
    assert_equal hash_pw('xyz'), cadmin.hashed_password # password is hashed and stored
    assert_not_nil cadmin.confirmed_on # account or email does not need to be confirmed
    # TODO: we can update attributes
  end
  
end
