/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.eworkbench;

import java.util.*;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IPluginContribution;
import org.eclipse.ui.internal.IWorkbenchGraphicConstants;
import org.eclipse.ui.internal.WorkbenchImages;
import org.eclipse.ui.internal.WorkbenchPlugin;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.eclipse.ui.internal.registry.RegistryReader;
import org.eclipse.ui.internal.registry.IWorkbenchRegistryConstants;


/** class used to describe an eRCP application
 * 
 */
public class ApplicationDescriptor implements IApplicationDescriptor,
        IPluginContribution {

	private String id;
    private String pluginId;
    private String label;
    private String className;
    private String description;
    private boolean singleton;
    private ImageDescriptor image; 
    private IConfigurationElement configElement;
	private Hashtable views = new Hashtable();
	private String icon;
	

    /**
     * Create a new empty descriptor.
     * 
     * @param id the id of the new descriptor
     * @param label the label of the new descriptor
     * @param originalDescriptor the descriptor that this descriptor is based on
     */
    public ApplicationDescriptor(String id, String label) {
        this.id = id;
        this.label = label;
    }

    /**
     * Create a descriptor from a config element.
     * 
     * @param id the id of the element to create
     * @param configElement the element to base this perspective on
     * @throws CoreException thrown if there are any missing attributes
     */
    public ApplicationDescriptor(String id, IConfigurationElement configElement) throws CoreException {
        this.configElement = configElement;
        this.id = id;
        // Sanity check.
        if ((getId() == null) || (getLabel() == null) ) {
            throw new CoreException(new Status(IStatus.ERROR,
                    WorkbenchPlugin.PI_WORKBENCH, 0,
                    "Invalid extension (missing label, id or class name): " + getId(),//$NON-NLS-1$
                    null));
        }
    }
    
    public String getDescription() {
        return configElement == null ? description : RegistryReader.getDescription(configElement);
    }

    public String getId() {
        return id;
    }

    public ImageDescriptor getImageDescriptor() {
    	if (image == null) {
            if (configElement != null) {
                String icon = configElement.getAttribute(IWorkbenchRegistryConstants.ATT_ICON);
                if (icon != null) {
                    image = AbstractUIPlugin.imageDescriptorFromPlugin(
                            configElement.getNamespace(), icon);
                }
                if (image == null) {
                    image = WorkbenchImages.getImageDescriptor(IWorkbenchGraphicConstants.IMG_ETOOL_DEF_PERSPECTIVE);
                }
            }
    	}
        return image;
    }

    public void setImage(String id, String icon){
    	this.icon = icon;
        if (id != null && icon != null) {
            image = AbstractUIPlugin.imageDescriptorFromPlugin(
                    id, icon);
        }

        if (image == null) {
    		//need to get image on GUI thread due to dependency in JFace resources
        	Display.getDefault().syncExec(new Runnable() {
                public void run() {
                	image = WorkbenchImages.getImageDescriptor(IWorkbenchGraphicConstants.IMG_ETOOL_DEF_PERSPECTIVE);
                }
            });
        }
    }

    public String getLabel() {
        return configElement == null ? label : configElement.getAttribute(IWorkbenchRegistryConstants.ATT_NAME);
    }

    /**
     * Returns <code>true</code> if this perspective is a singleton.
     * 
     * @return whether this perspective is a singleton
     */
    public boolean isSingleton() {
        return configElement == null ? singleton : configElement.getAttributeAsIs(IWorkbenchRegistryConstants.ATT_SINGLETON) != null;
    }

    /**
     * Return the configuration element used to create this perspective, if one was used.
     * 
     * @return the configuration element used to create this perspective
     */
    public IConfigurationElement getConfigElement() {
        return configElement;
    }

    public String getLocalId() {
        return getId();
    }

    public String getPluginId() {
        return configElement == null ? pluginId : configElement.getNamespace();
    }
    
    /**
     * Returns the factory class name for this descriptor.
     * 
     * @return the factory class name for this descriptor
     */
    public String getClassName() {
    	return configElement == null ? className : RegistryReader.getClassValue(configElement, IWorkbenchRegistryConstants.ATT_CLASS);
    }
	
	public void setViewId(String type, String id) {
		views.put(type, id);
	}
	
	public String getViewId(String type) {
		return (String)views.get(type);
	}
}
