/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.eworkbench;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.dynamichelpers.IExtensionChangeHandler;
import org.eclipse.core.runtime.dynamichelpers.IExtensionTracker;
import org.eclipse.ui.PlatformUI;

/**
 * eRCP Application registry.
 */
public class ApplicationRegistry implements IApplicationRegistry, IExtensionChangeHandler {
  
    private List applications = new ArrayList(10);
    
    /**
     * Construct a new registry.
     */
    public ApplicationRegistry() {
        IExtensionTracker tracker = PlatformUI.getWorkbench().getExtensionTracker();
    	tracker.registerHandler(this, null);
    }

    /**
     * Adds an application.  This is typically used by the reader.
     * 
     * @param desc
     */
    public void addApplication(ApplicationDescriptor desc) {
        if (desc == null)
            return;
        add(desc);
    }

    /**
	 * @param desc
	 */
	private void add(ApplicationDescriptor desc) {
		applications.add(desc);
		IConfigurationElement element = desc.getConfigElement();
		if (element != null) {
			PlatformUI.getWorkbench().getExtensionTracker().registerObject(
					element.getDeclaringExtension(), desc,
					IExtensionTracker.REF_WEAK);
		}
	}

	/**
	 * Create a new application.
	 * 
	 * @param label
	 *            the name of the new descriptor
	 * @param originalDescriptor
	 *            the descriptor on which to base the new descriptor
	 * @return a new application descriptor or <code>null</code> if the
	 *         creation failed.
	 */
    public ApplicationDescriptor createApplication(String label) {
        // Sanity check to avoid invalid or duplicate labels.
        if (!validateLabel(label))
            return null;
        if (findApplicationWithLabel(label) != null)
            return null;

        // Calculate ID.
        String id = label.replace(' ', '_');
        id = id.trim();

        // Create descriptor.
        ApplicationDescriptor desc = new ApplicationDescriptor(id, label);
        add(desc);
        return desc;
    }

    /**
     * Delete an application.
     * Has no effect if the application is defined in an extension.
     * 
     * @param in
     */
    public void deleteApplication(IApplicationDescriptor in) {
        ApplicationDescriptor desc = (ApplicationDescriptor) in;
        applications.remove(desc);
    }
    
    public IApplicationDescriptor findApplicationWithId(String id) {
    	for (Iterator i = applications.iterator(); i.hasNext();) {
			ApplicationDescriptor desc = (ApplicationDescriptor) i.next();
			if (desc.getId().equals(id))
				return desc;
		}

        return null;
    }

    public IApplicationDescriptor findApplicationWithLabel(String label) {
    	for (Iterator i = applications.iterator(); i.hasNext();) {
			ApplicationDescriptor desc = (ApplicationDescriptor) i.next();
			if (desc.getLabel().equals(label)) 
				return desc;
		}
        return null;
    }

     public IApplicationDescriptor[] getApplications() {
    	return (IApplicationDescriptor[]) applications.toArray(new IApplicationDescriptor[applications.size()]);
    }

    /**
     * Loads the registry.
     */
    public void load() {
        // Load the registries.
		ApplicationRegistryReader reader = new ApplicationRegistryReader(this);
	    reader.readApplications(Platform.getExtensionRegistry());
    }

    /**
     * Return <code>true</code> if a label is valid.
     * This checks only the given label in isolation.  It does not
     * check whether the given label is used by any
     * existing applications.
     * 
     * @param label the label to test
     * @return whether the label is valid
     */
    public boolean validateLabel(String label) {
        label = label.trim();
        if (label.length() <= 0)
            return false;
        return true;
    }

    public void dispose() {
    	PlatformUI.getWorkbench().getExtensionTracker().unregisterHandler(this);
    }

	public void removeExtension(IExtension source, Object[] objects) {
        for (int i = 0; i < objects.length; i++) {
            if (objects[i] instanceof ApplicationDescriptor) {
                deleteApplication((ApplicationDescriptor)objects[i]);
            }
        }
	}

	public void addExtension(IExtensionTracker tracker, IExtension addedExtension) {
        IConfigurationElement[] addedElements = addedExtension.getConfigurationElements();
        for (int i = 0; i < addedElements.length; i++) {
            ApplicationRegistryReader reader = new ApplicationRegistryReader(this);
            reader.readElement(addedElements[i]);
        }
    }
}
