/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.update.internal.core.connection;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.net.HttpURLConnection;
import java.net.URL;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.update.internal.core.UpdateCore;
import org.eclipse.update.internal.core.IStatusCodes;
import org.eclipse.update.internal.core.Messages;

public class HttpResponse extends AbstractResponse {
	/**
	 * Monitored InputStream.  Upon IOException, discards
	 * connection so it is not resused.
	 *
	 */
	Class Base64Class = null;
	private class MonitoringInputStream extends FilterInputStream {
		InputStream in;

		public MonitoringInputStream(InputStream in) {
			super(in);
			this.in = in;
		}

		public int available() throws IOException {
			try {
				return super.available();
			} catch (IOException ioe) {
				connection = null;
				throw ioe;
			}
		}

		public void close() throws IOException {
			try {
				super.close();
			} catch (IOException ioe) {
				connection = null;
				throw ioe;
			}
		}

		public int read() throws IOException {
			try {
				return super.read();
			} catch (IOException ioe) {
				connection = null;
				throw ioe;
			}
		}

		public synchronized void reset() throws IOException {
			try {
				super.reset();
			} catch (IOException ioe) {
				connection = null;
				throw ioe;
			}
		}

		public int read(byte[] b) throws IOException {
			try {
				return super.read(b);
			} catch (IOException ioe) {
				connection = null;
				throw ioe;
			}
		}

		public int read(byte[] b, int off, int len) throws IOException {
			try {
				return super.read(b, off, len);
			} catch (IOException ioe) {
				connection = null;
				throw ioe;
			}
		}

		public long skip(long n) throws IOException {
			try {
				return super.skip(n);
			} catch (IOException ioe) {
				connection = null;
				throw ioe;
			}
		}

	}
	
	protected URL url;
	protected InputStream in;
	protected long lastModified;
	protected long offset;
	//Added by Julian Chen
	private boolean isPoorNetwork = false;

	protected HttpResponse(URL url) {
		
        this.url = url;
	}

	public InputStream getInputStream() throws IOException {
		if (in == null && url != null) {
			if (connection == null || offset > 0)
				connection = url.openConnection();
			if (offset > 0)
				connection.setRequestProperty("Range", "bytes=" + offset + "-"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			try {
				InputStream connInputStream = connection.getInputStream();
				if (connInputStream != null)
					in = new MonitoringInputStream(connInputStream);
			} catch (IOException ioe) {
				connection = null;
				throw ioe;
			}
			checkOffset();
		}
		return in;
	}
	/**
	 * @see IResponse#getInputStream(IProgressMonitor)
	 */
	public InputStream getInputStream(IProgressMonitor monitor)
		throws IOException, CoreException, TooManyOpenConnectionsException {
		if (in == null && url != null) {
			if (connection == null || offset > 0)
				connection = url.openConnection();
			if (offset > 0)
				connection.setRequestProperty("Range", "bytes=" + offset + "-"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

			if (monitor != null) {
				try {
					// Changed by Julian Chen, set 'this.in' as null when connection is failed or terminated.
					InputStream connInputStream = openStreamWithCancel(connection, monitor);
					// if monitor.isCanceled() is set true during openStreamWithCancel(), then the return is null.
					if (connInputStream != null)
						this.in = new MonitoringInputStream(connInputStream);
				} catch (IOException ioe) {
					connection = null;
					throw ioe;
				}
			} else {
				try {
					InputStream connInputStream = connection.getInputStream();
					if (connInputStream != null)
						this.in = new MonitoringInputStream(connInputStream);
				} catch (IOException ioe) {
					connection = null;
					throw ioe;
				}
			}
//			// Added by Julian Chen
//			if (monitor.isCanceled()) {
//				return null;
//			}
			// this can also be run inside a monitoring thread, but it is safe
			// to
			// just call it now, if the input stream has already been obtained
			checkOffset();
			if (in != null) {
				this.lastModified = connection.getLastModified();
			}
		}
		return in;
	}

	public long getContentLength() {
		if (connection != null)
			return connection.getContentLength();
		return 0;
	}

//	public int getStatusCode() {
//		if (connection == null)
//			try {
//				connection = url.openConnection();
//			} catch (IOException e) {
//			}
//		if (connection != null) {
//			try {
//				return ((HttpURLConnection) connection).getResponseCode();
//			} catch (IOException e) {
//				AppPlugin.warn("", e); //$NON-NLS-1$
//			}
//		}
//		return IStatusCodes.HTTP_OK;
//	}

	public int getStatusCode() {
		if (connection == null)
			try {
				connection = url.openConnection();
			} catch (IOException e) {
			}
		if (connection != null) {
			
			//IBM, Eric MF Hsu, Add basic authentication for secure update site provisioning. This requires com.ibm.pvc.sharedbundle bundle to enable this function.
			if(url != null && url.getUserInfo() != null) {	
				
				try {
					Base64Class = Class.forName("com.ibm.rcp.internal.util.BASE64Encoder");
					if( Base64Class != null ) {
					 Method method = Base64Class.getDeclaredMethod("encode", new Class[] { byte[].class });				
					 byte[] o = (byte[]) method.invoke(null, new Object[] {(url.getUserInfo()).getBytes("UTF-8")});
					 connection.setRequestProperty("Authorization","Basic "+ new String(o));
					}
				} catch (Exception e) {
					System.out.println("[eUpdate Debug] Basic Authentication failed"); //$NON-NLS-1$
				}
			}
			
			// Because the "HttpURLConnection.getResponseCode" may be blocked by poor network,
			// we add some time-out detection for workaround.
			// It should be better to change the time-out value of HttpURLConnection if possible.
			isPoorNetwork = false; // reset it before every new try.
			StatusThread st = new StatusThread((HttpURLConnection) connection);
			st.start();
	
			int timeOutValue = 10000; // set 10 seconds by default for getting connection status.
			try {
				String timeOutValueString = System.getProperty("sun.net.client.defaultConnectTimeout"); //$NON-NLS-1$
				if (timeOutValueString != null)
					timeOutValue = Integer.parseInt(timeOutValueString);
				if (timeOutValue < 1000)
					timeOutValue = 1000; // at least 1 second.
			}
			catch (Exception e) {
				// keep default timeOutValue.
			}

			try {
				st.join(timeOutValue);
				if (st.isAlive()) {
					if (UpdateCore.DEBUG) {
						System.out.println("[eUpdate Debug] Internal time out detection for HttpURLConnection."); //$NON-NLS-1$
					}
					st.interrupt();
					isPoorNetwork = true;
				}
			}
			catch (InterruptedException ie) {
			}

			if (isPoorNetwork)
				return IStatusCodes.HTTP_GATEWAY_TIMEOUT;
			else
				return st.getResponseCode();
		}
		return IStatusCodes.HTTP_OK;
	}

	public class StatusThread extends Thread {
		HttpURLConnection conn;
		int result;

		public StatusThread(HttpURLConnection connection) {
			super();
			this.conn = connection;
		}

		public void run() {
			try {
				result = conn.getResponseCode();
			} catch (IOException e) {
				UpdateCore.warn("", e); //$NON-NLS-1$
			}
		}

		public int getResponseCode() {
			return result;
		}
	}

	public String getStatusMessage() {
		if (connection != null) {
			if (isPoorNetwork)
				return ""; //$NON-NLS-1$
			try {
				return ((HttpURLConnection) connection).getResponseMessage();
			} catch (IOException e) {
				UpdateCore.warn("", e); //$NON-NLS-1$
			}
		}
		return ""; //$NON-NLS-1$
	}

	public long getLastModified() {
		if (lastModified == 0) {
			if (connection == null)
				try {
					connection = url.openConnection();
				} catch (IOException e) {
				}
			if (connection != null)
				lastModified = connection.getLastModified();
		}
		return lastModified;
	}

	public void setOffset(long offset) {
		this.offset = offset;
	}
	private void checkOffset() throws IOException {
		if (offset == 0)
			return;
		String range = connection.getHeaderField("Content-Range"); //$NON-NLS-1$
		//System.out.println("Content-Range=" + range);
		if (range == null) {
			//System.err.println("Server does not support ranges");
			throw new IOException(Messages.HttpResponse_rangeExpected); 
		} else if (!range.startsWith("bytes " + offset + "-")) { //$NON-NLS-1$ //$NON-NLS-2$
			//System.err.println("Server returned wrong range");
			throw new IOException(Messages.HttpResponse_wrongRange); 
		}
	}
}
