/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     James D Miles (IBM Corp.) - bug 167687, Need new API when using managed-only update policy
 *******************************************************************************/
package org.eclipse.update.internal.operations;

import java.lang.reflect.*;

import org.eclipse.core.runtime.*;
import org.eclipse.update.core.*;
import org.eclipse.update.core.model.*;
import org.eclipse.update.internal.core.*;
import org.eclipse.update.operations.*;
import org.eclipse.update.internal.operations.UnconfigOperation;

public class BatchUnconfigOperation
	extends Operation
	implements IBatchOperation {
		
	protected IUnconfigFeatureOperation[] operations;
	private boolean bVerifyComplete = false;

	public BatchUnconfigOperation(IUnconfigFeatureOperation[] operations, boolean bVerifyComplete) {
		super();
		this.operations = operations;
		this.bVerifyComplete = bVerifyComplete;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.update.operations.IMultiOperation#getOperations()
	 */
	public IFeatureOperation[] getOperations() {
		return operations;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.update.operations.IOperation#execute(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public boolean execute(IProgressMonitor monitor, IOperationListener listener) throws CoreException, InvocationTargetException {
		int configCount = 0;

		if (operations == null || operations.length == 0)
			return false;
	
		if (bVerifyComplete == false) {
			IStatus status = OperationsManager.getValidator()
					.validatePendingChanges(operations);
			if (status != null && status.getCode() == IStatus.ERROR) {
				throw new CoreException(status);
			}
		}
		
		OperationsManager.setInProgress(true);
		if (monitor == null) 
			monitor = new NullProgressMonitor();
			
		try {			
			if (listener != null)
				listener.beforeExecute(this, null);
			
			monitor.beginTask(
				Messages.OperationsManager_installing,
				operations.length);
			for (int i = 0; i < operations.length; i++) {
				SubProgressMonitor subMonitor =
					new SubProgressMonitor(
						monitor,
						1,
						SubProgressMonitor.PREPEND_MAIN_LABEL_TO_SUBTASK);
				if(operations[i] instanceof UnconfigOperation) {
					UnconfigOperation configOper = (UnconfigOperation)operations[i];
					configOper.setBatchUnconfigOperation();
				}
				operations[i].execute(subMonitor, listener);
				OperationsManager.addPendingOperation(operations[i]);

				operations[i].markProcessed();
				if (listener != null)
					listener.afterExecute(operations[i], null);

				//monitor.worked(1);
				configCount++;
			}
			return SiteManager.getLocalSite().save();
		} catch (InstallAbortedException e) {
			// saves the current configuration
			if (configCount > 0)
				SiteManager.getLocalSite().save();
			throw new InvocationTargetException(e);
		} catch (CoreException e) {
			// saves the current configuration
			if (configCount > 0)
				SiteManager.getLocalSite().save();
			throw new InvocationTargetException(e);
		} finally {
			OperationsManager.setInProgress(false);
			monitor.done();
		}
	}
}
