/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     James D Miles (IBM Corp.) - bug 53858, selective install for multiuser
 *                                 with managed-only policy
*******************************************************************************/
package org.eclipse.update.internal.operations;

import org.eclipse.core.runtime.*;
import org.eclipse.osgi.util.NLS;
import org.eclipse.update.configuration.*;
import org.eclipse.update.core.*;
import org.eclipse.update.internal.core.Messages;
import org.eclipse.update.operations.*;

/**
 * Configure a feature.
 * ConfigOperation
 */
public class ConfigOperation
	extends OptionalFeatureOperation
	implements IConfigFeatureOperation {
	
	boolean bBatchConfigOperation = false;

	public ConfigOperation(
		IConfiguredSite site,
		IFeature feature) {
		this(site, feature, null, null);
	}

	/**
	 * Constructor
	 * @param site
	 * @param feature
	 * @param optionalFeatures optional features to install. If null, the operation will leave optional features unconfigured (if any)
	 * @param unconfiguredOptionalElements optional features unconfigured before the operation. They should remain unconfigured after the install.
	 * @param verifier
	 */
	public ConfigOperation(IConfiguredSite site, IFeature feature,
			IFeatureReference[] optionalFeatures, IFeature[] unconfiguredOptionalElements) {
		super(site, feature);
		IFeature[] installed = UpdateUtils.getInstalledFeatures(feature);
		if (installed.length > 0)
			this.oldFeature = installed[0];
		this.unconfiguredOptionalFeatures = unconfiguredOptionalElements;
		this.optionalFeatures = optionalFeatures;
	}

	public boolean execute(IProgressMonitor pm, IOperationListener listener)
		throws CoreException {

		if(bBatchConfigOperation == false){
			IStatus status =
				OperationsManager.getValidator().validatePendingConfig(feature);
			if (status != null && status.getCode() == IStatus.ERROR) {
				throw new CoreException(status);
			}
		}

		if(isManagedOnly())
			setOptionalFeatures();

		if (oldFeature != null && isManagedOnly()) { //&& isOptionalDelta()) {
			preserveOptionalState();
			boolean oldSuccess = unconfigure(oldFeature, null); // pick any site containing old feature
			if (!oldSuccess) {
				IInstallConfiguration config = SiteManager.getLocalSite().getCurrentConfiguration();
				if (!UpdateUtils.isNestedChild(config, oldFeature)) {
					// "eat" the error if nested child
					String message =
						NLS.bind(Messages.OperationsManager_error_old, (new String[] { oldFeature.getLabel() }));
					IStatus status =
						new Status(
							IStatus.ERROR,
							UpdateUtils.getPluginId(),
							IStatus.OK,
							message,
							null);
					throw new CoreException(status);
				}
			}
		}

		targetSite.configure(feature, optionalFeatures, pm);
//		ensureUnique();

		try {
			// Restart not needed
			boolean restartNeeded = false;

			// Check if this operation is cancelling one that's already pending
			IOperation pendingOperation =
				OperationsManager.findPendingOperation(feature);

			if (pendingOperation instanceof IUnconfigFeatureOperation) {
				// no need to do either pending change
				OperationsManager.removePendingOperation(pendingOperation);
			} else {
				OperationsManager.addPendingOperation(this);
			}

			markProcessed();
			if (listener != null)
				listener.afterExecute(this, null);

			if(bBatchConfigOperation == false){
				restartNeeded = SiteManager.getLocalSite().save() && restartNeeded;
			}
			// notify the model
			OperationsManager.fireObjectChanged(feature, null);

			return restartNeeded;
		} catch (CoreException e) {
			undo();
			UpdateUtils.logException(e);
			throw e;
		}
	}
	
	/*
	 * Use default package visibility. This should only be used within this package.
	 * For managed-only we usually perform verification on [] of operations that will not
	 * pass by verifying 1 at a time.
	 * 
	 */
	void setBatchConfigOperation() {
		bBatchConfigOperation = true;
	}

	public void undo() throws CoreException {
		targetSite.unconfigure(feature);
	}
}
