/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     James D Miles (IBM Corp.) - bug 167687, Need new API when using managed-only update policy
*******************************************************************************/

package org.eclipse.update.operations;

import org.eclipse.update.configuration.*;
import org.eclipse.update.core.*;

/**
 * A factory interface for creating operations.
 * <p>
 * <b>Note:</b> This class/interface is part of an interim API that is still under development and expected to
 * change significantly before reaching stability. It is being made available at this early stage to solicit feedback
 * from pioneering adopters on the understanding that any code that uses this API will almost certainly be broken
 * (repeatedly) as the API evolves.
 * </p>
 * @since 3.0
 */
public interface IOperationFactory {
	/**
	 * Creates an operation for configuring an installed feature in the specified site.
	 * @param targetSite site containing the feature to configure
	 * @param feature feature to be configured
	 * @return the configure operation
	 */
	public IConfigFeatureOperation createConfigOperation(
		IConfiguredSite targetSite,
		IFeature feature);
		
	/**
	 * This config operation must be used if using managed-only update policy.
	 * Creates an operation for configuring an installed feature in the specified site.
	 * @param targetSite site containing the feature to configure
	 * @param feature feature to be configured
	 * @param optionalFeatures optional features to install. If empty array, the operation will leave optional features unconfigured (if any).
	 * If null, the operation will confgure all optional features (if any).
	 * @param unconfiguredOptionalFeatures when configuring optional features, some can be left unconfigured
	 * @return the configure operation
	 * @since 3.3
	 */
	public IConfigFeatureOperation createConfigOperation(
		IConfiguredSite targetSite,
		IFeature feature,
		IFeatureReference[] optionalFeatures,
		IFeature[] unconfiguredOptionalFeatures);

	/**
	 * Creates an operation for unconfiguring a feature
	 * @param targetSite site containing the feature to unconfigure
	 * @param feature feature to be unconfigured
	 * @return the unconfigure operation
	 */
	public IUnconfigFeatureOperation createUnconfigOperation(
		IConfiguredSite targetSite,
		IFeature feature);
	
	/**
	 * Creates an operation for installing a feature.
	 * No configure/unconfigure operations will be performed if using managed-only update policy.
	 * @param targetSite site in which the feature is to be installed
	 * @param feature feature to be installed
	 * @param optionalFeatures optionally included features to be installed (if any)
	 * @param unconfiguredOptionalFeatures when installing optional features, some can be left unconfigured
	 * @param verifier operation verification listener
	 * @return the install operation
	 */
	public IInstallFeatureOperation createInstallOperation(
		IConfiguredSite targetSite,
		IFeature feature,
		IFeatureReference[] optionalFeatures,
		IFeature[] unconfiguredOptionalFeatures,
		IVerificationListener verifier);
		
	/**
	 * Creates an operation to uninstall a feature
	 * @param targetSite site containing the feature to uninstall
	 * @param feature feature to be uninstalled
	 * @return the uninstall operation
	 */
	public IUninstallFeatureOperation createUninstallOperation(
		IConfiguredSite targetSite,
		IFeature feature);
		
	/**
	 * Creates an operation for replacing this feature by a previous version
	 * @param feature current feature
	 * @param anotherFeature the new feature to be swapped in
	 * @return the revert feature version operation
	 */
	public IConfigFeatureOperation createReplaceFeatureVersionOperation(
		IFeature feature,
		IFeature anotherFeature);
		
	/**
	 * Creates an operation for executing a set of feature install operations in batch mode
	 * @param operations operations to execute in batch mode	
	 * @return the batch operation
	 */
	public IBatchOperation createBatchInstallOperation(IInstallFeatureOperation[] operations);
	
	/**
	 * Creates an operation for executing a set of feature config operations in batch mode
	 * @param operations operations to execute in batch mode	
	 * @param if already verified set to true, otherwise set to false so verification will be performed.
	 * @return the batch operation
	 * @since 3.3
	 */
	public IBatchOperation createBatchConfigOperation(IConfigFeatureOperation[] operations, boolean bVerifyComplete);
	
	/**
	 * Creates an operation for executing a set of feature unconfig operations in batch mode
	 * @param operations operations to execute in batch mode	
	 * @param if already verified set to true, otherwise set to false so verification will be performed.
	 * @return the batch operation
	 * @since 3.3
	 */
	public IBatchOperation createBatchUnconfigOperation(IUnconfigFeatureOperation[] operations, boolean bVerifyComplete);

	/**
	 * Creates ann operation to configure/unconfigure an installation site (also known as enable/disable site)
	 * @param site site to configure/unconfigure
	 * @return the toggle site operation
	 */
	public IToggleSiteOperation createToggleSiteOperation(
		IConfiguredSite site);
		

	/**
	 * Creates an operation to revert to a previous installation configuration.
	 * @param config configuration to revert to
	 * @param problemHandler error handler
	 * @return the revert operation
	 */
	public IRevertConfigurationOperation createRevertConfigurationOperation(
		IInstallConfiguration config,
		IProblemHandler problemHandler);
}
