/*******************************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.ercp.update.views;

import org.eclipse.ercp.swt.mobile.Command;
import org.eclipse.ercp.update.UIMessages;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;


public abstract class AbsScreen implements SelectionListener, ControlListener {

	protected final int NONE		= 0;
	protected final int BACK		= 1;
	protected final int CANCEL		= 2;
	protected final int NEXT		= 3;
	protected final int FINISHED	= 4;
	
	NormalView view;
	Composite screen;
	Composite screenPane;
	WrappedLabel titleLabel;
	CommandButton backButton, cancelButton, nextButton;
	Composite buttonPane;

	public AbsScreen() {
	}
	
	public AbsScreen(NormalView view) {
		this.view = view;
	}
	
	public void show(boolean refreshIt) {
		if (screen == null)
			createScreen();

		if (refreshIt) {
			refresh();
			// make sure the screen is ready (refreshed) before we set it visible.
			setVisible(true);
			action();
		}
		else
			setVisible(true);
	}

	public void setVisible(boolean bVis) {
		screen.setVisible(bVis);
		if (bVis)
			setDefaultFocus();
	}
	
	/* Subclasses must implement this method to fill the screen's content.
	 * The implementation may call helper functions init, setButtons,
	 * fillSpace, and spearator to assist creating screen.
	 */
	protected abstract void createScreen(); 

	/*
	 * Based on the layout created by createScreen() method, the refresh() method
	 * can dynamically refresh the screen with variable data when entering the screen.
	 * Subclasses may override this method (the default one does nothing) if necessary.
	 */
	protected abstract void refresh();

	/*
	 * Perform actions after screen is ready.
	 */
	protected abstract void action();

	/*
	 * 
	 */
	protected abstract void setDefaultFocus();

	protected void init(String title) {
		init(title, false);
	}

	protected Composite init(String title, boolean scrollable) {
		if (scrollable) {
			screen = new Composite(view.parent, SWT.V_SCROLL);
			screen.getVerticalBar().setVisible(false); // scrollbar is not showed by default.
			screen.setBounds(view.parent.getClientArea());
			screen.setLayout(new GridLayout(1, false)); // set default layout of screen.
			screen.setLayoutData(new GridData(GridData.FILL_BOTH));
			screen.addControlListener(this);

			screenPane = new Composite(screen, SWT.NONE);
			screenPane.setLayout(new GridLayout(1, false)); // set default layout of screen pane.
			screenPane.setLayoutData(new GridData(GridData.FILL_BOTH));
		}
		else {
			screen = new Composite(view.parent, SWT.NONE);
			screen.setBounds(view.parent.getClientArea());
			screen.setLayout(new GridLayout(1, false)); // set default layout of screen.
			screen.setLayoutData(new GridData(GridData.FILL_BOTH));
			screen.addControlListener(this);

			screenPane = screen;
		}

		if (title != null) {
			titleLabel = new WrappedLabel(screenPane, SWT.NONE, SWT.NONE);
			if (NormalView.OS_ARM_SYMBIAN == view.osType)
				titleLabel.setFont(new Font(Display.getCurrent(), "Arial",  7, SWT.BOLD)); //$NON-NLS-1$
			else
				titleLabel.setFont(new Font(Display.getCurrent(), "Arial", 10, SWT.BOLD)); //$NON-NLS-1$
			titleLabel.setText(title);
		}

		return screenPane;
	}
	
	protected void fillSpace() {
		new Composite(screenPane, SWT.NONE).setLayoutData(new GridData(GridData.FILL_VERTICAL));
	}
	
	protected void separator() {
		if (NormalView.OS_ARM_SYMBIAN == view.osType)
			return;
		new Label(screenPane, SWT.SEPARATOR | SWT.HORIZONTAL).setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
	}
	
	protected void setButtons(int back, int cancel, int next) {

		// Because of SPR JCSY7K7B96, we need a temp workaround when running Sprint Titan on Rafael.
		// Remove it after the bug can be fixed on eSWT.
		if ((NormalView.RUNTIME_TITAN == view.runtimeType)
			&& (view.isDPI192Device)
		){
			setButtons_onSprintRafael(back, cancel, next);
			return;
		}
		// End of temp code.

		buttonPane = new Composite(screenPane, SWT.NONE);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		//gd.heightHint = 27; // too large for limited space on STD
		buttonPane.setLayoutData(gd);
			
		RowLayout paneLayout = new RowLayout(SWT.HORIZONTAL);
		paneLayout.justify = true;
		paneLayout.wrap = false;
		buttonPane.setLayout(paneLayout);

		// Because Command with attribute GENERAL can't be Softkey,
		// we did some trickery here.
		int commandAttribute  = Command.GENERAL;
		if (view.osType == NormalView.OS_ARM_SYMBIAN)
			commandAttribute = Command.STOP;

		if (back == BACK) {
			if (view.inputType == NormalView.INPUT_SHOW_SOFTKEY_MODE)
				backButton = new CommandButton(screen, commandAttribute, 12);
			else
				backButton = new CommandButton(buttonPane, SWT.PUSH);
			if (NormalView.OS_ARM_SYMBIAN == view.osType)
				backButton.setFont(view.fontOnS60);
			backButton.setText(UIMessages.Back);
			backButton.addSelectionListener(this);
		}
	
		if (cancel == CANCEL) {
			if (view.inputType == NormalView.INPUT_SHOW_SOFTKEY_MODE)
				cancelButton = new CommandButton(screen, commandAttribute, 11);
			else
				cancelButton = new CommandButton(buttonPane, SWT.PUSH);
			if (NormalView.OS_ARM_SYMBIAN == view.osType)
				cancelButton.setFont(view.fontOnS60);
			cancelButton.setText(UIMessages.Cancel);
			cancelButton.addSelectionListener(this);
		}
		
		if (next != NONE) {
			if (view.inputType == NormalView.INPUT_SHOW_SOFTKEY_MODE)
				nextButton = new CommandButton(screen, commandAttribute, 13);
			else
				nextButton = new CommandButton(buttonPane, SWT.PUSH);
			if (NormalView.OS_ARM_SYMBIAN == view.osType)
				nextButton.setFont(view.fontOnS60);
			if (next == NEXT)
				nextButton.setText(UIMessages.Next);
			else
				nextButton.setText(UIMessages.Finished);
			nextButton.addSelectionListener(this);
		}
		buttonPane.layout();

		if ((view.inputType != NormalView.INPUT_SHOW_SOFTKEY_MODE) && (nextButton != null)) {
			// in case of buttons are not all showed completely.
			if (nextButton.getLocation().x > screenPane.getSize().x) {
				paneLayout.wrap = true;
				buttonPane.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
				screen.layout();				
			}
		}
	}
	
	/* Subclasses must implement widgetSelected()
	 */
	public abstract void widgetSelected(SelectionEvent se);

	public void widgetDefaultSelected(SelectionEvent arg0) {
		widgetSelected(arg0);
	}	
	
	public void controlMoved(ControlEvent e) {
	}

	public void controlResized(ControlEvent e) {
		if (!screen.isDisposed())
			screen.setBounds(view.parent.getClientArea());
	}
	
	// Because of SPR JCSY7K7B96, we need a temp workaround when running Sprint Titan on Rafael.
	// Remove it after the bug can be fixed on eSWT.
	private void setButtons_onSprintRafael(int back, int cancel, int next) {
		int totalButtonCount = 0;
		int buttonIdx = 0;
		FormData[] fdButtons;

		buttonPane = new Composite(screenPane, SWT.NONE);
		buttonPane.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		buttonPane.setLayout(new FormLayout());

		if (back == BACK)		totalButtonCount++;
		if (cancel == CANCEL)	totalButtonCount++;
		if (next != NONE)		totalButtonCount++;
		fdButtons = new FormData[totalButtonCount];

		if (totalButtonCount == 1) {
			// If there is 1 button,
			fdButtons[0] = new FormData();
			fdButtons[0].top	= new FormAttachment(  0,  0);
			fdButtons[0].bottom	= new FormAttachment(100, -5);
			fdButtons[0].left	= new FormAttachment( 35,  0);
			fdButtons[0].right	= new FormAttachment( 65,  0);
		}
		else if (totalButtonCount == 2) {
			// If there are 2 buttons,
			fdButtons[0] = new FormData();
			fdButtons[0].top	= new FormAttachment(  0,  0);
			fdButtons[0].bottom	= new FormAttachment(100, -5);
			fdButtons[0].left	= new FormAttachment(  5,  0);
			fdButtons[0].right	= new FormAttachment( 35,  0);

			fdButtons[1] = new FormData();
			fdButtons[1].top	= new FormAttachment(  0,  0);
			fdButtons[1].bottom	= new FormAttachment(100, -5);
			fdButtons[1].left	= new FormAttachment( 65,  0);
			fdButtons[1].right	= new FormAttachment( 95,  0);			
		}
		else if (totalButtonCount == 3) {
			// If there are 3 buttons,
			fdButtons[0] = new FormData();
			fdButtons[0].top	= new FormAttachment(  0,  0);
			fdButtons[0].bottom	= new FormAttachment(100, -5);
			fdButtons[0].left	= new FormAttachment(  6,  0);
			fdButtons[0].right	= new FormAttachment( 30,  0);

			fdButtons[1] = new FormData();
			fdButtons[1].top	= new FormAttachment(  0,  0);
			fdButtons[1].bottom	= new FormAttachment(100, -5);
			fdButtons[1].left	= new FormAttachment( 38,  0);
			fdButtons[1].right	= new FormAttachment( 62,  0);

			fdButtons[2] = new FormData();
			fdButtons[2].top	= new FormAttachment(  0,  0);
			fdButtons[2].bottom	= new FormAttachment(100, -5);
			fdButtons[2].left	= new FormAttachment( 70,  0);
			fdButtons[2].right	= new FormAttachment( 94,  0);
		}
		else {
			// it should never happen.
		}

		if (back == BACK) {
			backButton = new CommandButton(buttonPane, SWT.PUSH);
			backButton.setText(UIMessages.Back);
			backButton.addSelectionListener(this);
			backButton.setLayoutData(fdButtons[buttonIdx++]);
		}

		if (cancel == CANCEL) {
			cancelButton = new CommandButton(buttonPane, SWT.PUSH);
			cancelButton.setText(UIMessages.Cancel);
			cancelButton.addSelectionListener(this);
			cancelButton.setLayoutData(fdButtons[buttonIdx++]);
		}
		
		if (next != NONE) {
			nextButton = new CommandButton(buttonPane, SWT.PUSH);
			if (next == NEXT)
				nextButton.setText(UIMessages.Next);
			else
				nextButton.setText(UIMessages.Finished);
			nextButton.addSelectionListener(this);
			nextButton.setLayoutData(fdButtons[buttonIdx++]);
		}

		buttonPane.layout();
	}
	// End of temp code.
}
