/*******************************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.ercp.update.views;

import java.io.File;
import java.net.URL;
import java.util.Vector;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.ercp.swt.mobile.Command;
import org.eclipse.ercp.swt.mobile.TaskTip;
import org.eclipse.ercp.update.AppPlugin;
import org.eclipse.ercp.update.UIImage;
import org.eclipse.ercp.update.UIMessages;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.update.configurator.ConfiguratorUtils;
import org.eclipse.update.core.ICategory;
import org.eclipse.update.core.IFeatureReference;
import org.eclipse.update.core.ISite;
import org.eclipse.update.core.ISiteFeatureReference;
import org.eclipse.update.core.IURLEntry;
import org.eclipse.update.core.SiteManager;
import org.eclipse.update.core.VersionedIdentifier;
import org.eclipse.update.internal.core.SiteFileContentProvider;
import org.eclipse.update.internal.core.SiteURLContentProvider;
import org.eclipse.update.internal.core.UpdateCore;
import org.eclipse.update.internal.ui.model.BookmarkUtil;
import org.eclipse.update.util.BookmarkSite;
import org.eclipse.update.util.CategoryWrap;
import org.eclipse.update.util.FeatureWrap;
import org.eclipse.update.util.SiteWrap;


public class BookmarksScreen extends AbsScreen {

	public static final String BOOKMARK_FILE = "bookmarks.xml"; //$NON-NLS-1$

	private Table siteList;
	private BookmarkSite selectedBookmark;
	private Vector bookmarks = new Vector();
	private Vector tableItems = new Vector();
	private String bookmarkLocation = null;
	private Label  description;
	private TaskTip progress;
	private Command addCommand, editCommand, removeCommand;
	private StringBuffer accessLog;
	private UpdateMonitor accessSiteMonitor;
	private AccessSiteThread accessSiteThread;
	private boolean neverComeHere = true;
//	Remove the following code when duplicated-timerExec() bug is fixed.
	private boolean isDuplicated = false;
//End.
	private Image IMAGE_ICON_SITE = null;

	
	public BookmarksScreen(NormalView view) {
		super(view);

		// Locate the path of bookmark file.
		URL platformXML = ConfiguratorUtils.getCurrentPlatformConfiguration().getConfigurationLocation();
		if (!"file".equals(platformXML.getProtocol())) { //$NON-NLS-1$
			IPath path = AppPlugin.getDefault().getStateLocation();
			path = path.append(BOOKMARK_FILE);
			bookmarkLocation = path.toOSString();
		}
		else
		{
			File f = new File(platformXML.getFile());
			f = new File(f.getParentFile(), BOOKMARK_FILE);
			bookmarkLocation = f.getAbsolutePath();
		}
	}

	protected void createScreen() {
		if (NormalView.OS_ARM_SYMBIAN == view.osType)
			init(UIMessages.SelectSites2);
		else
			init(UIMessages.SelectSites);
		
		IMAGE_ICON_SITE = UIImage.createImage(UIImage.ICON_SITE);

		siteList = new Table(screen, SWT.CHECK | SWT.V_SCROLL);
		siteList.setLayoutData(new GridData(GridData.FILL_BOTH));
		siteList.addSelectionListener(this);
		TableColumn tc = new TableColumn(siteList, SWT.NONE);
		tc.setWidth(view.parent.getSize().x-20);
		
		separator();

		description = new Label(screen, SWT.NONE);
		description.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		setButtons(BACK, CANCEL, NEXT);
		
		addCommand = new Command(screen, Command.GENERAL, 10);
		addCommand.setText(UIMessages.Add);
		addCommand.setLongLabel(UIMessages.AddBookmark);
		addCommand.addSelectionListener(this);

		editCommand = new Command(screen, Command.GENERAL, 9);
		editCommand.setText(UIMessages.Edit);
		editCommand.setLongLabel(UIMessages.EditBookmark);
		editCommand.addSelectionListener(this);

		removeCommand = new Command(screen, Command.GENERAL, 8);
		removeCommand.setText(UIMessages.Remove);
		removeCommand.setLongLabel(UIMessages.RemoveBookmark);
		removeCommand.addSelectionListener(this);

		if (NormalView.OS_ARM_SYMBIAN == view.osType) {
			siteList.setFont(view.fontOnS60);
			description.setFont(view.fontOnS60);
		}
		screen.layout();
	}

	protected void refresh() {
		addCommand.setEnabled(true);
		editCommand.setEnabled(false);
		removeCommand.setEnabled(false);
		backButton.setEnabled(true);
		cancelButton.setEnabled(false);
		nextButton.setEnabled(false);
		description.setText(""); //$NON-NLS-1$

		view.bookmarkFile = bookmarkLocation;
		siteList.removeAll();
		tableItems.removeAllElements();
		BookmarkUtil.load(bookmarkLocation, bookmarks);

		BookmarkSite bkmark;
		for (int i=0; i<bookmarks.size(); i++)
		{
			TableItem ti = new TableItem(siteList, SWT.NONE);
			bkmark = (BookmarkSite) bookmarks.elementAt(i);
			ti.setImage(IMAGE_ICON_SITE);
			ti.setText(bkmark.getName());
			ti.setChecked(bkmark.getSelected());
			tableItems.add(ti);

			if (bkmark.getSelected())
				nextButton.setEnabled(true);
		}
	}

	public void action() {
		if (neverComeHere && (bookmarks.size() == 0)) {
			MessageBox mb = new MessageBox(screen.getShell(), SWT.ICON_INFORMATION | SWT.OK);
			mb.setText(UIMessages.Hints);
			if (NormalView.OS_ARM_SYMBIAN == view.osType)
				mb.setMessage(NLS.bind(UIMessages.HintToNewBookmark3, new String[] {"Options"})); //$NON-NLS-1$
			else {
				if (NormalView.INPUT_SHOW_SOFTKEY_MODE == view.inputType)
					mb.setMessage(UIMessages.HintToNewBookmark2);
				else
					mb.setMessage(UIMessages.HintToNewBookmark);
			}
			mb.open();
		}
		neverComeHere = false;

		setDefaultFocus();
		screen.pack();
	}

	protected void setDefaultFocus() {
		siteList.setFocus();

		if (siteList.getItemCount() > 0) {
			if ((NormalView.OS_ARM_SYMBIAN != view.osType) && (nextButton.getEnabled())) {
				
				//fix bug: SCLL7UELME
				siteList.select(0);
				
				nextButton.setFocus();
			}
			else {
				siteList.select(0);
				Event initEvent = new Event();
				initEvent.widget = siteList;
				siteList.notifyListeners(SWT.Selection, initEvent);
				siteList.setFocus();
			}
		}
	}

	public void widgetSelected(SelectionEvent e) {
		if (e.widget == siteList) {
			//check for null item because win platforms are calling 
			//this when the table gains focus and passing null item!
			if (e.detail == SWT.CHECK && e.item != null) {
				// If any item is checked, then the next button is enable.
				int checkedCount = 0;
				for (int i=0; i< tableItems.size(); i++) {
					if (((TableItem)e.item).equals((TableItem) tableItems.elementAt(i))) {
						((BookmarkSite) bookmarks.elementAt(i)).setSelected(((TableItem)e.item).getChecked());
					}
					if (((TableItem) tableItems.elementAt(i)).getChecked()) {
						checkedCount++;
					}
				}
				nextButton.setEnabled(checkedCount>0);

				// Temporary workaround for SPR #PBPP7HGB69.  There is a bug in eSWT Table on WM device.
				// When selecting a TableItem, the value of SelectionEvent.details should be '0',
				// but it is always '32'(SWT.CHECK) now.  So add the code below.
				int count = siteList.getSelectionCount();
				if (count == 1) {
					editCommand.setEnabled(true);
					removeCommand.setEnabled(true);

					TableItem[] items = siteList.getSelection();
					for (int j=0; j< bookmarks.size(); j++) {
						selectedBookmark = (BookmarkSite) bookmarks.elementAt(j);
						if (items[0].getText().equals(selectedBookmark.getName())) {
							description.setText(selectedBookmark.getDescription());
							view.currentName = selectedBookmark.getName();
							break;
						}
					}
				}
				// End of workaround.  Remove it after the bug is fixed on eSWT.
			}
			else {
				int count = siteList.getSelectionCount();

				editCommand.setEnabled(count == 1);
				removeCommand.setEnabled(count == 1);

				// update description
				if (count == 1) {
					TableItem[] items = siteList.getSelection();
					for (int j=0; j< bookmarks.size(); j++) {
						selectedBookmark = (BookmarkSite) bookmarks.elementAt(j);
						if (items[0].getText().equals(selectedBookmark.getName())) {
							description.setText(selectedBookmark.getDescription());
							view.currentName = selectedBookmark.getName();
							break;
						}
					}
				}
			}
			return;
		}
		
		// save bookmark before leaving this screen.
		BookmarkUtil.store(bookmarkLocation, bookmarks);

		if (e.widget == backButton.widget) {
			setVisible(false);
			view.showScreen(view.ROOT_SCREEN, view.NO_REFRESH);
		}
		else if (e.widget == cancelButton.widget) {
			if (accessSiteMonitor != null) {
				accessSiteMonitor.setCanceled(true);
			}
		}
		else if (e.widget == nextButton.widget) {
			backButton.setEnabled(false);
			cancelButton.setEnabled(true);
			nextButton.setEnabled(false);
			processNextButton();
		}
		else if (e.widget == addCommand) {
			setVisible(false);
			view.showScreen(view.ADD_BOOKMARK_SCREEN, view.REFRESH_IT);
		}
		else if (e.widget == editCommand) {
			setVisible(false);
			view.showScreen(view.EDIT_BOOKMARK_SCREEN, view.REFRESH_IT);
		}
		else if (e.widget == removeCommand) {
			// show remove dialog
			MessageBox mb = new MessageBox(screen.getShell(), SWT.ICON_QUESTION | SWT.YES | SWT.NO);
			mb.setText(UIMessages.RemoveBookmark);
			mb.setMessage(UIMessages.RemoveBookmarkQuery);
			if (mb.open() == SWT.YES) {
				// Delete bookmark, update GUI
				bookmarks.remove(selectedBookmark);
				BookmarkUtil.store(bookmarkLocation, bookmarks);
				refresh();
				setDefaultFocus();
			}
		}
	}

	public void widgetDefaultSelected(SelectionEvent e) {
		if (e.widget == siteList) {
			widgetSelected(e);
		}
	}

	private void processNextButton() {
		// start progress indicator
		progress = new TaskTip(screen.getShell(), SWT.INDETERMINATE);
		progress.setText(UIMessages.SearchingSites);
		progress.setVisible(true);

		// while searching, the default focus is on the cancel button.
		if (view.inputType != NormalView.INPUT_SHOW_SOFTKEY_MODE)
			cancelButton.setFocus();
		else
			screen.setFocus(); // on SoftKey mode, move focus from TaskTip to main screen.

		accessSiteMonitor = new UpdateMonitor();
		accessLog = new StringBuffer();
		accessSiteThread =
			new AccessSiteThread("", bookmarks, //$NON-NLS-1$
									view.viewedTree, view.availableFeatures,
									accessLog, accessSiteMonitor);
		accessSiteThread.start();

		// Currently set this waiting time (for Thread.start()) to 0.7 second.
		// Because device has less resource and Thread may start later than we expect,
		// this time value cannot be too small.
		Display.getCurrent().timerExec(700, new Runnable() {
			public void run() {
				int threadStatus = accessSiteThread.getStatus();
				if ( (threadStatus == AccessSiteThread.RUNNING) || (threadStatus == AccessSiteThread.NONE)) {
					Display.getCurrent().timerExec(500, this);
				}
				else if (!accessSiteMonitor.isCanceled() && (threadStatus == AccessSiteThread.DONE)) {
					// Because the CANCEL status is no more be check from now on, we disable it here.
					cancelButton.setEnabled(false);

					progress.setVisible(false);
					// TODO - Julian: make sure why error occurs if TaskTip.dispose() is performed before MessageBox.open().
					//progress.dispose();
					//progress = null;
//Remove the following code when duplicated-timerExec() bug is fixed.
					if (isDuplicated)
						return;
					else
						isDuplicated = true;
//End.
					// show message if ever error when accessing.
					if (accessLog.length() > 0) {
						MessageBox mb = new MessageBox(screen.getShell(), SWT.ICON_INFORMATION | SWT.OK);
						mb.setText(UIMessages.UpdateSiteStatus);
						mb.setMessage(accessLog.toString());
						mb.open();
					}
//Remove the following code when duplicated-timerExec() bug is fixed.
					isDuplicated = false;
//End.
					// TODO - Julian: make sure why error occurs if TaskTip.dispose() is performed before MessageBox.open().
					progress.dispose();
					progress = null;
					accessSiteThread = null;
					backButton.setEnabled(true);
					//cancelButton.setEnabled(false);
					nextButton.setEnabled(true);
					setVisible(false);
					view.showScreen(view.SELECT_FEATURES_SCREEN, view.REFRESH_IT);				
				}
				
				else { //usually now threadStatus == AccessSiteThread.FAILED
					accessSiteThread = null;
					progress.setVisible(false);
					progress.dispose();
					progress = null;
					backButton.setEnabled(true);
					cancelButton.setEnabled(false);
					nextButton.setEnabled(true);
				}
			}
		} //End of new Runnable() {}
		);
	}

	public class AccessSiteThread extends Thread {

		public static final int NONE		= 0;
		public static final int RUNNING		= 1;
		public static final int DONE		= 2;
		public static final int CANCELED	= 3;

		private Vector bookmarks;
		private Vector viewedTree;
		private Vector availableFeatures;
		private StringBuffer log;
		private IProgressMonitor monitor;
		private int currentStatus;

		public AccessSiteThread(String name,
								Vector bookmarks,			//input
								Vector viewedTree,			//output
								Vector availableFeatures,	//output
								StringBuffer accessLog,		//output
								IProgressMonitor accessMonitor) {
			super(name);
			this.bookmarks = bookmarks;
			this.viewedTree = viewedTree;
			this.availableFeatures = availableFeatures;
			this.log = accessLog;
			this.monitor = accessMonitor;
			currentStatus = NONE;
		}

		public void run() {
			int uniquenessStatus;
			try {
				currentStatus = RUNNING;
				viewedTree.removeAllElements();
				availableFeatures.removeAllElements();

				for (int i=0; i< bookmarks.size(); i++) {
					BookmarkSite bookmarkSite = (BookmarkSite) bookmarks.elementAt(i);
					if (bookmarkSite.getSelected()) {
						ISite remoteSite = null;
						if (!monitor.isCanceled()) {
							try {
								remoteSite = SiteManager.getSite(bookmarkSite.getURL(), monitor);
							}
							catch (CoreException ce) {
								log.append(NLS.bind(
										UIMessages.GotAccessError,
										new String[] {bookmarkSite.getName(), ce.getLocalizedMessage()}
								))
								.append("\n"); //$NON-NLS-1$
								//TODO - Julian: error logging.
								UpdateCore.log(ce);
								if (UpdateCore.DEBUG) {
									ce.printStackTrace();
								}
								// error for this site, try next site.
								continue;
							}
						}
						if (monitor.isCanceled()) {
							currentStatus = CANCELED;
							return;
						}

						// need more check ?
						//if (remoteSite == null)
						//	continue;

// set the site type filter for device-specific updtae site as disable by default.(bug 155420)
						// check site type.
						// if no specific request, only process supported type.
						boolean enableSiteFilter = Boolean.valueOf(System.getProperty("org.eclipse.ercp.update.enableSiteFilter")).booleanValue(); //$NON-NLS-1$
						if (enableSiteFilter
								&& !SiteURLContentProvider.SITE_TYPE.equalsIgnoreCase(remoteSite.getType())
								&& !SiteFileContentProvider.SITE_TYPE.equalsIgnoreCase(remoteSite.getType()) )
						{
							log.append(NLS.bind(
									UIMessages.NotSupportedSiteType,
									new String[] {bookmarkSite.getName(), (remoteSite.getType()==null)?"":remoteSite.getType()} //$NON-NLS-1$
							))
							.append("\n"); //$NON-NLS-1$
							continue;
						}
						SiteWrap sw = new SiteWrap(bookmarkSite.getName());
						sw.setDescription(remoteSite.getDescription().getAnnotation());
						// Also update the site description in the bookmark when accessing.
						bookmarkSite.setDescription(remoteSite.getDescription().getAnnotation());

						// for the features without assigned category info in the current Site.
						CategoryWrap otherCategory = null;

						ISiteFeatureReference[] remoteFeatures = remoteSite.getRawFeatureReferences();
						if (remoteFeatures != null) {
							for (int j=0; j<remoteFeatures.length; j++ ) {
								try {
									// check if new feature.
									uniquenessStatus = checkFeatureUniqueness(remoteFeatures[j]);
									if ((uniquenessStatus == FeatureWrap.FEATURE_UNKNOWN) || (uniquenessStatus == FeatureWrap.FEATURE_DUPLICATE)) {
										// this remote feature was installed and no need to update version.
										continue;
									}
									FeatureWrap fw = new FeatureWrap(remoteFeatures[j]);
									fw.setFeatureUniquenessStatus(uniquenessStatus);
									ICategory[] categories = remoteFeatures[j].getCategories();
									if (categories.length == 0) {
										// this feature has no category info, so we set it to the "other" category.
										if (otherCategory == null) {
											otherCategory = new CategoryWrap(UIMessages.Other);
											sw.addChild(otherCategory);
										}
										otherCategory.addChild(fw);
									}
									else {
										for (int k=0; k<categories.length; k++) {
											CategoryWrap cw = getThisCategoryWrap(sw, categories[k]);
											if (cw != null)
												cw.addChild(fw);
										}
									}

									availableFeatures.add(fw);
								} catch (Exception e) {
									e.printStackTrace();

									// error for this feature, try next feature.
									continue;
								}
							}
						}
						view.viewedTree.add(sw);
					}
				}
				if (monitor.isCanceled())
					currentStatus = CANCELED;
				else
					currentStatus = DONE;
			}
			catch (Exception e) {
				// need this catch() to set the status in case any error occurs but status is not updated.
				currentStatus = CANCELED;
				//TODO - Julian: error logging.
				e.printStackTrace();
			}
		}

		/*
		 * return existing CategoryWrap or create a new one.
		 */
		private CategoryWrap getThisCategoryWrap(SiteWrap sw, ICategory category) {
			if ((sw == null) || (category == null))
				return null;

			CategoryWrap [] cws = sw.getChildren();
			for (int i=0; i< cws.length; i++) {
				if ( category.getLabel().equals(((CategoryWrap) cws[i]).getLabel()) )
					return (CategoryWrap) cws[i];
			}

			// no found, create a new category.
			CategoryWrap result = new CategoryWrap(category.getLabel());
			IURLEntry description = category.getDescription();
			if (description != null)
				result.setDescription(description.getAnnotation());
			sw.addChild(result);
			return result;
		}

		private int checkFeatureUniqueness(ISiteFeatureReference remoteFeature) {
			VersionedIdentifier remoteVID, localVID;
			try {
				remoteVID = remoteFeature.getVersionedIdentifier();
			} catch (CoreException ce1) {
				ce1.printStackTrace();
				// failed to get VID info of this remote feature, give up it.
				return FeatureWrap.FEATURE_UNKNOWN;
			}

			for (int i=0; i<view.installedFeatures.size(); i++) {
				try {
					localVID = ((IFeatureReference)view.installedFeatures.get(i)).getVersionedIdentifier();
				} catch (CoreException ce2) {
					ce2.printStackTrace();
					// failed to get VID info of this local feature, skip it.
					continue;
				}
				if (localVID.getIdentifier().equals(remoteVID.getIdentifier())) {
					//1. local feature and remote feature have the same ID, need to check version.
					if (localVID.getVersion().isGreaterOrEqualTo(remoteVID.getVersion())) {
						//2. remote version is not greater than local version.
						return FeatureWrap.FEATURE_DUPLICATE;
					}
					else {
						//2. remote version is greater than local version, so it may be the updates.
						return FeatureWrap.FEATURE_UPDATES;
					}
				}
			}
			return FeatureWrap.FEATURE_BRAND_NEW;
		}

		/*
		 * 
		 */
		public int getStatus() {
			return currentStatus;
		}
	}
}
