/*******************************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.ercp.update.views;

import java.lang.reflect.InvocationTargetException;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.ercp.swt.mobile.Command;
import org.eclipse.ercp.update.UIMessages;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.ui.PlatformUI;
import org.eclipse.update.configuration.IConfiguredSite;
import org.eclipse.update.configurator.ConfiguratorUtils;
import org.eclipse.update.core.IFeatureReference;
import org.eclipse.update.core.SiteManager;
import org.eclipse.update.core.VersionedIdentifier;
import org.eclipse.update.core.model.InstallAbortedException;
import org.eclipse.update.internal.core.UpdateCore;
import org.eclipse.update.internal.operations.IUnconfigureAndUninstallFeatureOperation;
import org.eclipse.update.internal.operations.OperationFactory;
import org.eclipse.update.operations.OperationsManager;
import org.eclipse.update.util.FeatureWrap;


public class InstallingScreen extends AbsScreen {

	public final static int  PROGRESS_MAX_VALUE = 10;

	public Label featureLabel;
	public Label featureName;
	public ProgressBar featureProgress, overallProgress;
	private int totalInstallFeaturesCount = 0;
	private UpdateMonitor installFeatureMonitor;
	private InstallFeatureThread installFeaturesThread;
	private Composite pane;
	private Command cancelCommand;
	private StringBuffer installLog;
	private int buttonSet = 0;
//Remove the following code when duplicated-timerExec() bug is fixed.
	private boolean isExisted = false;
//End.

	InstallingScreen(NormalView view) {
		super(view);
	}

	protected void createScreen() {
		init(null);

		pane = new Composite(screen, SWT.NONE);
		pane.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
//		RowLayout paneLayout = new RowLayout(SWT.HORIZONTAL);
//		paneLayout.wrap = false;

//		if (NormalView.OS_ARM_SYMBIAN == view.osType)
			pane.setLayout(new GridLayout(1, false));
//		else
//			pane.setLayout(paneLayout);
		
		featureLabel = new Label(pane, SWT.NONE);
		featureLabel.setText(UIMessages.Installing);
		featureName = new Label(pane, SWT.NONE);
		featureName.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		featureProgress = new ProgressBar(screen, SWT.SMOOTH);
		featureProgress.setMaximum(PROGRESS_MAX_VALUE);
		featureProgress.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		Label overallLabel = new Label(screen, SWT.NONE);
		overallLabel.setText(UIMessages.OverallProgress);

		overallProgress = new ProgressBar(screen, SWT.SMOOTH);
		overallProgress.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		fillSpace();
//it should be removed after having soft key mode.
//		if (NormalView.WM_SMARTPHONE == view.wmType) {
//			cancelCommand = new Command(screen, Command.GENERAL, 10);
//			cancelCommand.addSelectionListener(this);
//		}
		setButtons(NONE, CANCEL, NONE);

		if (NormalView.OS_ARM_SYMBIAN == view.osType) {
			featureLabel.setFont(view.fontOnS60);
			featureName.setFont(view.fontOnS60);
		}
		screen.layout();
	}

	protected void refresh() {
//it should be removed after having soft key mode.
//		if (NormalView.WM_SMARTPHONE == view.wmType)
//			cancelCommand.setEnabled(true);
		cancelButton.setEnabled(true);
		featureName.setText(""); //$NON-NLS-1$
		featureProgress.setSelection(0);
		overallProgress.setSelection(0);
		screen.layout(); // force to re-update the screen
	}

	protected void action() {
		try {
			if (view.targetSite == null)
				view.targetSite = SiteManager.getLocalSite().getCurrentConfiguration().getConfiguredSites()[0];
		}
		catch (CoreException ce) {
			ce.printStackTrace();
			//TODO - Julian: error logging.
		}

		if (view.targetSite == null) {
			//TODO - Julian: error logging.
			System.out.println("Error: error to get local site for installation."); //$NON-NLS-1$
			return;
		}

		// get the count of how many features will be installed.
		totalInstallFeaturesCount = 0;
		for (int i=0; i< view.availableFeatures.size(); i++) {
			if (((FeatureWrap) view.availableFeatures.elementAt(i)).isSelected()) {
				totalInstallFeaturesCount++;
			}
		}
		overallProgress.setMaximum(totalInstallFeaturesCount);

		// go installing...
		installFeatureMonitor = new UpdateMonitor();
		installLog = new StringBuffer();
		installFeaturesThread =
			new InstallFeatureThread("", view.targetSite, view.availableFeatures, view.installedFeatures, installLog, installFeatureMonitor); //$NON-NLS-1$
		installFeaturesThread.start();

		setDefaultFocus();

		Display.getCurrent().timerExec(400, new Runnable() {

			public void run() {
				if (installFeaturesThread.isFinished()) {
//Remove the following code when duplicated-timerExec() bug is fixed.
					if (isExisted)
						return;
					else
						isExisted = true;
//End.
					if (installFeaturesThread.getDoneWorksCount() == totalInstallFeaturesCount) {
						// installation is finished completely.
						featureName.setText(NLS.bind(
								UIMessages.FeaturesInstalled, new Object[] { new Integer(totalInstallFeaturesCount).toString() }));
						screen.layout(); // force to re-update the screen
						featureProgress.setSelection(PROGRESS_MAX_VALUE);
						overallProgress.setSelection(totalInstallFeaturesCount);
						view.enableRestartCommand = true;
						// confirm with users for restarting the system.

						if(installFeaturesThread.isJvmInUpdateList()) {
							// New version of JVM runtime or library is installed,
							// and the command and parameters to launch the JVM may also be changed.
							// It's needed for the user to manually close the runtime and restart it,
							// because current auto-restart function will launch the JVM with the original command and parameters.
							if (NormalView.RUNTIME_TITAN == view.runtimeType)
								buttonSet = SWT.OK;
							else
								buttonSet = SWT.YES | SWT.NO;
							MessageBox mb1 = new MessageBox(screen.getShell(), SWT.ICON_INFORMATION | buttonSet);
							mb1.setText(UIMessages.ConfirmRestart);
							mb1.setMessage(NLS.bind(UIMessages.NeedRestartManually, view.RuntimeName));
							if (mb1.open() == SWT.YES) {
								if (NormalView.RUNTIME_TITAN != view.runtimeType) {
									// Sprint Titan doesn't support restart yet.
									System.getProperties().put("eworkbench.returnCode", new Integer(PlatformUI.RETURN_OK)); //$NON-NLS-1$
									PlatformUI.getWorkbench().close();
									return;
								}
							}
						}
						else if (installFeaturesThread.isOldVersionReplaced()){
							// New features were installed and some of them replaced/removed existed older versions.
							// Recommend to restart.
							if (NormalView.RUNTIME_TITAN == view.runtimeType)
								buttonSet = SWT.OK;
							else
								buttonSet = SWT.YES | SWT.NO;
							MessageBox mb2 = new MessageBox(screen.getShell(), SWT.ICON_INFORMATION | buttonSet);
							mb2.setText(UIMessages.ConfirmRestart);
							mb2.setMessage(NLS.bind(UIMessages.RecommendToRestart, view.RuntimeName));
							if (mb2.open() == SWT.YES) {
								if (NormalView.RUNTIME_TITAN != view.runtimeType) {
									// Sprint Titan doesn't support restart yet.
									System.getProperties().put("eworkbench.returnCode", new Integer(PlatformUI.RETURN_RESTART)); //$NON-NLS-1$
									PlatformUI.getWorkbench().close();
									return;
								}
							}
						}
						else {
							// All new installed features are NEW features for current system.
							// No need to restart.
							MessageBox mb4 = new MessageBox(screen.getShell(), SWT.ICON_INFORMATION | SWT.OK);
							mb4.setText(UIMessages.InstallStatus);
							mb4.setMessage(NLS.bind(UIMessages.InstallCompleted, view.RuntimeName));
							mb4.open();
						}
					}
					else {
						// although failed, still update the final status.
						featureProgress.setSelection((int)(PROGRESS_MAX_VALUE * installFeatureMonitor.getDoneWorksCount() / installFeatureMonitor.getTotalWorksCount()));
						overallProgress.setSelection(installFeaturesThread.getDoneWorksCount());

						// installation is not finished, but was terminated, also exit it.
						UpdateCore.log(UIMessages.InstallNotCompleted, null);
						UpdateCore.log(installLog.toString(), null);

						MessageBox mb2 = new MessageBox(screen.getShell(), SWT.ICON_ERROR | SWT.OK);
						mb2.setText(UIMessages.InstallStatus);
						mb2.setMessage(UIMessages.InstallNotCompleted);
						mb2.open();
					}
//Remove the following code when duplicated-timerExec() bug is fixed.
					isExisted = false;
//End.
//it should be removed after having soft key mode.
//					if (NormalView.WM_SMARTPHONE == view.wmType)
//						cancelCommand.setEnabled(false);
					cancelButton.setEnabled(false);
					installFeaturesThread =  null;
					
					setVisible(false);
					view.showScreen(view.ROOT_SCREEN, view.REFRESH_IT);
				}
				else {
					if (installFeaturesThread.isAlive()) {
						// installation is still progressing, update the UI.
						featureName.setText(installFeaturesThread.getCurrentFeatureName());
						screen.layout();// force to re-update the screen
						featureProgress.setSelection((int)(PROGRESS_MAX_VALUE * installFeatureMonitor.getDoneWorksCount() / installFeatureMonitor.getTotalWorksCount()));
						overallProgress.setSelection(installFeaturesThread.getDoneWorksCount());
					}
					else {
						// Sometime the thread on device is late to be started,
						// take more waitings.
					}
					Display.getCurrent().timerExec(300, this);
				}
			}
		});
	}

	protected void setDefaultFocus() {
		if (view.inputType == NormalView.INPUT_SHOW_SOFTKEY_MODE)
			pane.setFocus();
		else
			cancelButton.setFocus();
	}

	public void widgetSelected(SelectionEvent e) {
		if (	(e.widget == cancelButton.widget)
			||	(e.widget == cancelCommand)) {
			if ((installFeaturesThread!=null) && (!installFeaturesThread.isFinished())) {
				// in installing, just terminate the task.
				installFeatureMonitor.setCanceled(true);
			}
		}
	}

	private class InstallFeatureThread extends Thread {

		private IConfiguredSite site;
		private Vector features;
		private Vector allEarlierFeatures;
		private Vector beSuccessfullyInstalleds;
		private StringBuffer log;
		private IProgressMonitor monitor;
		private int count;
		private String currentFeatureName = ""; //$NON-NLS-1$
		private boolean isThreadFinished = false;
		private boolean isJvmInList = false;
		private boolean isAnythingReplaced = false;

		public InstallFeatureThread(String name,
									IConfiguredSite configuredSite,	//input
									Vector availableFeatures,		//input
									Vector earlierInstalledFeatures,//input
									StringBuffer installLog,		//output
									IProgressMonitor installMonitor) {
			super(name);

			this.site = configuredSite;
			this.features = availableFeatures;
			this.allEarlierFeatures = earlierInstalledFeatures;
			this.log = installLog;
			this.monitor = installMonitor;
			this.count = 0;
			this.beSuccessfullyInstalleds = new Vector();
		}

		public void run() {
			isThreadFinished = false;
			isJvmInList = false;
			isAnythingReplaced = false;

			int i=0;
			try {
				FeatureWrap fw = null;
				OperationFactory operationFactory = (OperationFactory)OperationsManager.getOperationFactory();
				for (i=0; i< features.size(); i++) {
					fw = (FeatureWrap) features.elementAt(i);
					if (fw.isSelected()) {
						try {
							if (!monitor.isCanceled()) {
								currentFeatureName = fw.getShowedName();
								operationFactory.createInstallOperation(site, fw.getFeature(), null, null, null)
									.execute(monitor, null);

								// check if new version of JVM library is installed
								if (!isJvmInList && (fw.getVID().getIdentifier().startsWith(ManageScreen.JVM_LIB_FEATURE_NAME))) {
									isJvmInList = true;
								}

								// On device environment, system resource is limited,
								// if lots of features are selected to install at the same time, then the system will easily faint. 
								// So try to do something to release the allocated resource.
								fw.releaseFeature();
								try {
									// leave CPU time with system for handling UI.
									if (features.size() >= 10)
										sleep(500);
									else
										sleep(100);
								}
								catch (InterruptedException ie) {}
								count++;
								beSuccessfullyInstalleds.add(fw.getVID());
							}
							else {
								isThreadFinished = true;
								// Installation is terminated by the user, so exit the thread.
								// It will perform actions in finally{} statement before it really exits.
								return;
							}
						}
						catch (CoreException ce) {
							if (ce instanceof InstallAbortedException) {
								isThreadFinished = true;
								// Installation is terminated by the user, so exit the thread.
								// It will perform actions in finally{} statement before it really exits.
								return;
							}
							else {
								// Failed to install THIS feature.
								// Log it and try next.
								log.append(fw.getShowedName())
								.append("\n"); //$NON-NLS-1$
								//TODO - Julian: error logging.
								ce.printStackTrace();
							}
						}
					}
				}
			}
			catch (Exception e) {
				// Usually it should never come here.
				// One of the possibilities is that
				// the thread tries to install too many features at one time,
				// then the system gets out-of-memory Exception.

				//TODO - Julian: error logging.
				e.printStackTrace();
			}
			finally {
				if (beSuccessfullyInstalleds.size() > 0) {
					// After installing, if there was the same feature with older version, then we need to remove the older one.
					removeOldVersion(beSuccessfullyInstalleds);

					// Update system configuration
					ConfiguratorUtils.getCurrentPlatformConfiguration().refresh();

					if(!isAnythingReplaced) {
						// If all installed features are new brand installed one,
						// then apply changes right now.
						OperationsManager.applyChangesNow();
					}
				}

				// If there is any remained feature that is not installed, log the name.
				// It may be caused by cancelling or error.
				for (int j=i; j<features.size(); j++ ) {
					FeatureWrap fw = (FeatureWrap) features.elementAt(j);
					if (fw.isSelected()) {
						log.append(fw.getShowedName())
						   .append("\n"); //$NON-NLS-1$
					}
				}

				// make sure the status is set.
				isThreadFinished = true;				
			}
		}

		private void removeOldVersion(Vector newInstalledFeatureVIDs) {
			try {
				Object[] allEarlierFeatures_A = allEarlierFeatures.toArray();
				OperationFactory operationFactory = (OperationFactory)OperationsManager.getOperationFactory();
				IUnconfigureAndUninstallFeatureOperation unconfigureAndUninstallOperation = null;
				
				VersionedIdentifier newInstalledVID;
				IFeatureReference earlierFeature;
				for (int i=0; i<newInstalledFeatureVIDs.size(); i++) {
					newInstalledVID = (VersionedIdentifier) newInstalledFeatureVIDs.elementAt(i);
					try {
						// starting to find out if the same feature w/ older version had been installed.
						for (int j=0; j<allEarlierFeatures_A.length; j++) {
							earlierFeature = (IFeatureReference) allEarlierFeatures_A[j];
							if (earlierFeature == null) {
								// it means this feature has been removed earlier in the loop-i.
								continue;
							}

							if (newInstalledVID.getIdentifier().equals(earlierFeature.getVersionedIdentifier().getIdentifier())) {
								// have got the same id, need to also check the version next.
								if (newInstalledVID.getVersion().isGreaterThan(earlierFeature.getVersionedIdentifier().getVersion())) {
									// this one is the older version, need to remove it.
									unconfigureAndUninstallOperation =
										operationFactory.createUnconfigureAndUninstallFeatureOperation(earlierFeature.getSite().getCurrentConfiguredSite(), earlierFeature.getFeature(null));
									unconfigureAndUninstallOperation.execute(null, null);
									isAnythingReplaced = true;

									// Set it as null since it has been removed now.
									// So it won't be compared with in the loop-i again.
									allEarlierFeatures_A[j] = null;

									// assume the system has no more than one feature with the same id, so break the loop-j.
									break;
								}
							}
						}
					} catch (CoreException ce) {
						ce.printStackTrace();
					} catch (InvocationTargetException ite) {
						ite.printStackTrace();
					}
				}
				SiteManager.getLocalSite().save();
			}
			catch (CoreException ce) {
				ce.printStackTrace();
				return;
			}
		}

		public String getCurrentFeatureName() {
			return currentFeatureName;
		}

		public int getDoneWorksCount() {
			return count;
		}

		public boolean isFinished() {
			return isThreadFinished;
		}

		public boolean isJvmInUpdateList() {
			return isJvmInList;
		}

		public boolean isOldVersionReplaced() {
			return isAnythingReplaced;
		}
	}
}
