/*******************************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.ercp.update.views;

import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.ercp.swt.mobile.Command;
import org.eclipse.ercp.update.UIMessages;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.ui.PlatformUI;
import org.eclipse.update.configuration.IConfiguredSite;
import org.eclipse.update.core.IFeature;
import org.eclipse.update.core.IFeatureReference;
import org.eclipse.update.core.IImport;
import org.eclipse.update.core.IPluginEntry;
import org.eclipse.update.core.IURLEntry;
import org.eclipse.update.core.SiteManager;
import org.eclipse.update.core.VersionedIdentifier;


public class ManageScreen extends AbsScreen {
	
	// Temporary workaround for JCSY7K249Q, use List instead of SortedList.
	//private SortedList featureList;
	private List featureList;
	private ScrolledText description;
	private Command propCommand, updateCommand, uninsCommand, restartCommand;
	private Composite pane;
	private FormData fdFeatureList, fdDescription;

	private Vector allImportedPlugins = new Vector();
	private boolean isRequiredFeature;
	private IConfiguredSite[] allSites;
	private IFeature beingSelected;
	public static String JVM_CORE_FEATURE_NAME = "com.ibm.pvc.wece.device."; //$NON-NLS-1$
	public static String JVM_LIB_FEATURE_NAME = "com.ibm.pvc.wece."; //$NON-NLS-1$

	public ManageScreen(NormalView view) {
		super(view);
	}

	protected void createScreen() {
		init(UIMessages.AppsFeatures);
		if (NormalView.INPUT_SHOW_SOFTKEY_MODE != view.inputType) {

		// Temporary workaround for JCSY7K249Q, use List instead of SortedList.
		//featureList = new SortedList(screen, SWT.V_SCROLL | SWT.H_SCROLL);
		featureList = new List(screen, SWT.V_SCROLL | SWT.H_SCROLL);
		featureList.setLayoutData(new GridData(GridData.FILL_BOTH));
		featureList.addSelectionListener(this);

		separator();

		// SortedList on WM6STD has buggy showing when putting a ScrolledText w/ maxLineCount=3 here (SPR #JYJC7GF62C).
		// Not sure why (need someone to look into eSWT code), but change the max line count as '2' on WM6STD first.
		if (NormalView.WM_SMARTPHONE == view.wmType)
			description = new ScrolledText(screen, SWT.NONE, SWT.NONE, 2);
		else
			description = new ScrolledText(screen, SWT.NONE, SWT.NONE, 3);
		description.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		}
		else {
			pane = new Composite(screen, SWT.NONE);
			pane.setLayout(new FormLayout());
			pane.setLayoutData(new GridData(GridData.FILL_BOTH));
			
			fdFeatureList = new FormData();
			fdFeatureList.top = new FormAttachment(0, 0);
			fdFeatureList.bottom = new FormAttachment(75, 0);
			fdFeatureList.left = new FormAttachment(0, 0);
			fdFeatureList.right = new FormAttachment(100, 0);

			fdDescription = new FormData();
			fdDescription.top = new FormAttachment(75, 5);
			fdDescription.bottom = new FormAttachment(100, 0);
			fdDescription.left = new FormAttachment(0, 0);
			fdDescription.right = new FormAttachment(100, 0);

			featureList = new List(pane, SWT.V_SCROLL | SWT.H_SCROLL);
			featureList.setLayoutData(fdFeatureList);
			featureList.addSelectionListener(this);
			
			description = new ScrolledText(pane, SWT.NONE, SWT.NONE, -1);
			description.setLayoutData(fdDescription);
			
			//screen.setTabList(new Control[] {featureList});
		}

		setButtons(BACK, NONE, NONE);
		
		propCommand = new Command(screen, Command.GENERAL, 10);
		propCommand.setText(UIMessages.Properties);
		propCommand.setLongLabel(UIMessages.ShowProperties);
		propCommand.addSelectionListener(this);

		updateCommand = new Command(screen, Command.GENERAL, 9);
		updateCommand.setText(UIMessages.Update);
		updateCommand.setLongLabel(UIMessages.UpdateFeature);
		updateCommand.addSelectionListener(this);
	
		uninsCommand = new Command(screen, Command.GENERAL, 8);
		uninsCommand.setText(UIMessages.Uninstall);
		uninsCommand.setLongLabel(UIMessages.UninstallFeature);
		uninsCommand.addSelectionListener(this);

		restartCommand = new Command(screen, Command.GENERAL, 7);
		restartCommand.setText(NLS.bind(UIMessages.RestartWED, view.RuntimeName));
		restartCommand.addSelectionListener(this);

		if (NormalView.OS_ARM_SYMBIAN == view.osType) {
			featureList.setFont(view.fontOnS60);
			description.setFont(view.fontOnS60);
		}
		screen.layout();
	}

	protected void refresh () {
		featureList.removeAll();
		propCommand.setEnabled(false);
		updateCommand.setEnabled(false);
		uninsCommand.setEnabled(false);
		if (NormalView.RUNTIME_TITAN == view.runtimeType)
			restartCommand.setEnabled(false); // no restart support on Spint now
		else
			restartCommand.setEnabled(view.enableRestartCommand);
		if (NormalView.OS_ARM_SYMBIAN == view.osType)
			restartCommand.setEnabled(true);//TODO - Julian: remove it later.
		description.setText(""); //$NON-NLS-1$

		try {
			allSites = SiteManager.getLocalSite().getCurrentConfiguration().getConfiguredSites();
			if (allSites.length < 1)
				return;

			IFeatureReference[] frs;
			IFeature currentFeature = null;
			for (int i=0; i<allSites.length; i++) {
				frs = allSites[i].getFeatureReferences();
				for (int j=0; j<frs.length; j++) {
					try {
						VersionedIdentifier vid = frs[j].getVersionedIdentifier();
						boolean isItDeleted = false;

						// Check if this feature was deleted (marked as disabled and put in deletion list)
						for (int k=0; k<view.deletedFeatures.size(); k++) {
							if (vid.equals((VersionedIdentifier) view.deletedFeatures.elementAt(k))) {
								// this feature will be removed later, don't show it.
								isItDeleted = true;
								break;
							}
						}

						if (isItDeleted) {
							// No need to process this feature if it was deleted.
							break;
						}
						else {
							// Agree to put on UI
							featureList.add(frs[j].getName() + " (" + vid.getVersion() + ")"); //$NON-NLS-1$ //$NON-NLS-2$
						}

						// If this feature was not deleted and is not disabled,
						// then we need to know what plugins are required/imported by it. (this info will be useful later)
						currentFeature = frs[j].getFeature(null);
						if (SiteManager.getLocalSite().getFeatureStatus(currentFeature).getCode() != IFeature.STATUS_DISABLED) {
							IImport[] plugins = currentFeature.getImports();
							for (int l=0; l<plugins.length; l++) {
								String pluginID = plugins[l].getVersionedIdentifier().getIdentifier();
								if (!allImportedPlugins.contains(pluginID)) {
									// Add the plugin to the list if it's a new one.
									allImportedPlugins.add(pluginID);
								}
							}
						}
					}
					catch (CoreException ce) {
						//TODO - Julian : error logging.
						ce.printStackTrace();
					}
				}
			}
		}
		catch (CoreException ce) {
			ce.printStackTrace();
			return;
		}

	}

	protected void action() {
		setDefaultFocus();
		screen.pack();
		screen.layout(); // it's required on symbian
	}

	protected void setDefaultFocus() {
		featureList.setFocus();

		if (featureList.getItemCount() > 0) {
			if (featureList.getSelectionIndex() < 0)
				featureList.select(0);
			else
				featureList.select(featureList.getSelectionIndex());
			Event initEvent = new Event();
			initEvent.widget = featureList;
			featureList.notifyListeners(SWT.Selection, initEvent);
		}
	}

	public void widgetSelected(SelectionEvent e) {
		if (e.widget == featureList) {
			String[] selectedStrings = featureList.getSelection();
			propCommand.setEnabled(selectedStrings.length == 1);
			updateCommand.setEnabled(selectedStrings.length == 1);
			uninsCommand.setEnabled(selectedStrings.length == 1);
			isRequiredFeature = false;

			if (selectedStrings.length == 1) {
				VersionedIdentifier vid = null;
				beingSelected = null;

				// search the selected feature from all installed ones.
				IFeatureReference[] frs;
				for (int i=0; i<allSites.length; i++) {
					frs = allSites[i].getFeatureReferences();
					for (int j=0; j<frs.length; j++) {
						String versionInfo;
						try {
							vid = frs[j].getVersionedIdentifier();
							versionInfo = " (" + vid.getVersion().toString() + ")"; //$NON-NLS-1$ //$NON-NLS-2$
						}
						catch (Exception nullExceptionOrCoreException) {
							versionInfo = " (?.?.?)"; //$NON-NLS-1$
						}
						if (selectedStrings[0].equals(frs[j].getName()+versionInfo)) {
							try {
								beingSelected = frs[j].getFeature(null);
							} catch (CoreException ce) {ce.printStackTrace();}

							// found it, exit the searching loop
							break;
						}
					}
					//if found, then no need to process next site.
					if (beingSelected != null)
						break;
				}

				if (beingSelected == null) {
					// no this feature installed. should never happen, but be safer.
					propCommand.setEnabled(false);
					updateCommand.setEnabled(false);
					uninsCommand.setEnabled(false);
					return;
				}

				try {
					if (SiteManager.getLocalSite().getFeatureStatus(beingSelected).getCode() != IFeature.STATUS_DISABLED) {
						// If this feature is not disabled,
						// then we need to check if it includes any plugin that is required by another feature.
						// If required, then this feature should not be uninstalled at this time.
						IPluginEntry[] pes = beingSelected.getPluginEntries();
						for (int j=0; j<pes.length; j++) {
							for (int k=0; k<allImportedPlugins.size(); k++) {
								if (((String) allImportedPlugins.elementAt(k)).equals(pes[j].getVersionedIdentifier().getIdentifier()) ) {
//									System.out.println("! Match [" + (String) allImportedPlugins.elementAt(k) + "] and [" + pes[j].getVersionedIdentifier().getIdentifier() + "]");
									isRequiredFeature = true;
									break;
								}
							}
							if (isRequiredFeature)
								break;
						}
					}
				}
				catch (CoreException ce) {
					ce.printStackTrace();
				}

				// check if the beingSelected is system library, system library cannot be uninstall.
				if (vid != null) {
					if (isEssentialFeature(vid)) {
						uninsCommand.setEnabled(false);
					}
					else if (vid.getIdentifier().startsWith("com.ibm.pvc.wed.core.")) { //$NON-NLS-1$
						// if "com.ibm.pvc.wed.core.wm5.feature" or "com.ibm.pvc.wed.core.x86.feature"
						// then it's WED Device Core set.
						uninsCommand.setEnabled(false);
					}
				}
				// also, disabled feature cannot be updated.
				try {
					if (SiteManager.getLocalSite().getFeatureStatus(beingSelected).getCode() == IFeature.STATUS_DISABLED) {
						updateCommand.setEnabled(false);
						// there is one special case here.
						// if one system library has (at least) two versions installed,
						// then the older version (whose status is disabled) should be able to be uninstalled.
						uninsCommand.setEnabled(true);
					}
				} catch (CoreException ce) {ce.printStackTrace();}

				IURLEntry descriptionEntry = beingSelected.getDescription();
				if (descriptionEntry != null) {
					description.setText(descriptionEntry.getAnnotation());
					description.setSelection(0, 0);
				}
				else
					description.setText(""); //$NON-NLS-1$
				screen.layout(); // because 'description' may has more than one line.
				featureList.showSelection();
			}
		}
		else if (e.widget == backButton.widget) {
			setVisible(false);
			view.showScreen(view.ROOT_SCREEN, view.NO_REFRESH);
		}
		else if (e.widget == propCommand) {
			view.availableFeatures.removeAllElements();
			view.availableFeatures.add(beingSelected);
			setVisible(false);
			view.showScreen(view.PROPERTIES_SCREEN, view.REFRESH_IT);
		}
		else if (e.widget == updateCommand) {
			view.availableFeatures.removeAllElements();
			view.availableFeatures.add(beingSelected);
			setVisible(false);
			view.showScreen(view.UPDATING_SCREEN, view.REFRESH_IT);
		}
		else if (e.widget == uninsCommand) {
			if (isRequiredFeature) {
				MessageBox mb3 = new MessageBox(screen.getShell(), SWT.ICON_INFORMATION | SWT.OK);
				mb3.setText(UIMessages.Notice);
				mb3.setMessage(UIMessages.FeatureIsRequired);
				mb3.open();
				return;
			}
			MessageBox mb1 = new MessageBox(screen.getShell(), SWT.ICON_INFORMATION | SWT.YES | SWT.NO);
			mb1.setText(UIMessages.ConfirmUninstall);
			mb1.setMessage(UIMessages.UninstallQuery);
			if (mb1.open() == SWT.YES) {
				view.availableFeatures.removeAllElements();
				view.availableFeatures.add(beingSelected);
				setVisible(false);
				view.showScreen(view.UNINSTALL_SCREEN, view.REFRESH_IT);
			}
		}
		else if (e.widget == restartCommand) {
			MessageBox mb2 = new MessageBox(screen.getShell(), SWT.ICON_INFORMATION | SWT.YES | SWT.NO);
			mb2.setText(UIMessages.ConfirmExit);
			mb2.setMessage(NLS.bind(UIMessages.ExitWEDQuery, view.RuntimeName));
			if (mb2.open() == SWT.YES) {
				System.getProperties().put("eworkbench.returnCode", new Integer(PlatformUI.RETURN_RESTART)); //$NON-NLS-1$
				PlatformUI.getWorkbench().close();
				return;
			}
		}
	}
	
	public void widgetDefaultSelected(SelectionEvent e) {
		// forward the event for device environment.
		if (e.widget == featureList) {
			widgetSelected(e);

			// Just a little convenient shortcut on desktop,
			// to show feature's properties by double-click.
			if (Platform.OS_WIN32.equals(Platform.getOS())
					&& Platform.ARCH_X86.equals(Platform.getOSArch())) {
				view.availableFeatures.removeAllElements();
				view.availableFeatures.add(beingSelected);
				setVisible(false);
				view.showScreen(view.PROPERTIES_SCREEN, view.REFRESH_IT);
			}
		}
	}

	public boolean isEssentialFeature(VersionedIdentifier vid) {
		if (vid.getIdentifier().startsWith("com.ibm.pvc.wece.device.")) {//$NON-NLS-1$
			// if "com.ibm.pvc.wece.device.*.*.feature", (*.* = { win32 }.{ x86, arm, ...})
			// then it's J9 runtime.
			return true;
		}
		if (vid.getIdentifier().equals("com.ibm.pvc.core.ercp.feature")) { //$NON-NLS-1$
			// it's eRCP Core set.
			return true;
		}
		if (vid.getIdentifier().startsWith("com.ibm.pvc.ercp.ui.")) {//$NON-NLS-1$
			// if "com.ibm.pvc.ercp.ui.feature" or "com.ibm.pvc.ercp.ui.*.feature", (* = { win, wm2003, wince, ...})
			// then it's eSWT related set.
			return true;
		}
		if (vid.getIdentifier().equals("org.eclipse.ercp.update.feature")) { //$NON-NLS-1$
			// it's Application Manager feature.
			return true;
		}
		if (vid.getIdentifier().equals("com.ibm.rcp.logredirector.feature")) { //$NON-NLS-1$
			// if "com.ibm.pvc.wed.core.wm5.feature" or "com.ibm.pvc.wed.core.x86.feature"
			// it's Logging Redirector feature.
			return true;
		}
		if (vid.getIdentifier().equals("com.ibm.pvc.wct.deviceagent.feature")) { //$NON-NLS-1$
			// it's Device Agent feature.
			return true;
		}

		return false;
	}
}
