/*******************************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.ercp.update.views;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.ercp.swt.mobile.SortedList;
import org.eclipse.ercp.update.UIMessages;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Text;
import org.eclipse.update.core.IFeature;
import org.eclipse.update.core.IFeatureReference;
import org.eclipse.update.core.IImport;
import org.eclipse.update.core.IPluginEntry;
import org.eclipse.update.core.IURLEntry;
import org.eclipse.update.core.SiteManager;
import org.eclipse.update.core.VersionedIdentifier;


public class PropertiesScreen extends AbsScreen {

	private Composite namePane;
	private Label featureLabel;
	private Label featureName;
	private SortedList menuList;
	private Text propertiesArea;

	IFeature selectedFeature;
	StringBuffer generalContent = null;
	StringBuffer pluginsContent = null;
	StringBuffer requiredByContent = null;

	public PropertiesScreen(NormalView view) {
		super(view);
	}

	public void createScreen() {
		init(null);

		namePane = new Composite(screen, SWT.NONE);
		namePane.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		namePane.setLayout(new RowLayout(SWT.HORIZONTAL));

		featureLabel = new Label(namePane, SWT.NONE);
		featureLabel.setText(UIMessages.PropertiesOf);
		featureName = new Label(namePane, SWT.NONE);

		if (NormalView.OS_ARM_SYMBIAN == view.osType) {
			featureLabel.setFont(view.fontOnS60);
			featureName.setFont(view.fontOnS60);
		}

		Composite subPane = new Composite(screen, SWT.NONE);
		subPane.setLayout(new FormLayout());
		subPane.setLayoutData(new GridData(GridData.FILL_BOTH));

		FormData fdMenu = new FormData();
		FormData fdProperty = new FormData();

		if (view.parent.getClientArea().width > view.parent.getClientArea().height) {

			fdMenu.top = new FormAttachment(0, 0);
			fdMenu.bottom = new FormAttachment(100, 0);
			fdMenu.left = new FormAttachment(0, 0);
			fdMenu.right = new FormAttachment(100/2, -2); //-2 is the reserved margin from menuList to propertiesArea.

			fdProperty.top = new FormAttachment(0, 0);
			fdProperty.bottom = new FormAttachment(100, 0);
			fdProperty.left = new FormAttachment(100/2, 2); //2 is the reserved margin from propertiesArea to menuList.
			fdProperty.right = new FormAttachment(100, 0);		
		}
		else {
			fdMenu.top = new FormAttachment(0, 0);
			fdMenu.bottom = new FormAttachment(100/2, -2); //-2 is the reserved margin from menuList to propertiesArea.
			fdMenu.left = new FormAttachment(0, 0);
			fdMenu.right = new FormAttachment(100, 0);

			fdProperty.top = new FormAttachment(100/2, 2); //2 is the reserved margin from propertiesArea to menuList.
			fdProperty.bottom = new FormAttachment(100, 0);
			fdProperty.left = new FormAttachment(0, 0);
			fdProperty.right = new FormAttachment(100, 0);
		}

		menuList = new SortedList(subPane, SWT.V_SCROLL);
		menuList.setLayoutData(fdMenu);
		menuList.addSelectionListener(this);
		menuList.add(UIMessages.Copyright);
		menuList.add(UIMessages.GeneralInfo);
		menuList.add(UIMessages.License);
		menuList.add(UIMessages.Status);
		menuList.add(UIMessages.IncludedPlugins);
		menuList.add(UIMessages.RequiredBy);

		propertiesArea = new Text(subPane, SWT.MULTI | SWT.WRAP | SWT.V_SCROLL | SWT.BORDER);
		propertiesArea.setLayoutData(fdProperty);
		propertiesArea.setEditable(false);

		separator();
		setButtons(BACK, NONE, NONE);

		if (NormalView.OS_ARM_SYMBIAN == view.osType) {
			menuList.setFont(view.fontOnS60);
			propertiesArea.setFont(view.fontOnS60);
		}
		screen.layout();
	}

	protected void refresh() {
		selectedFeature = (IFeature) view.availableFeatures.elementAt(0);
		generalContent = null;
		pluginsContent = null;
		requiredByContent = null;

		featureName.setText(selectedFeature.getLabel());
		namePane.layout();

		// show copyright as default
		menuList.select(UIMessages.Copyright);
		menuList.showSelection();
		showCopyright();
	}

	protected void action() {
		setDefaultFocus();
		screen.pack();
		screen.layout();
	}

	protected void setDefaultFocus() {
		menuList.setFocus();
	}

	public void widgetSelected(SelectionEvent e) {
		if (e.widget == menuList) {
			String[] selectedStrings = menuList.getSelection();
			if (UIMessages.Copyright.equals(selectedStrings[0])) {
				showCopyright()	;
			}
			else if (UIMessages.GeneralInfo.equals(selectedStrings[0])) {
				showGeneralInfo();
			}
			else if (UIMessages.License.equals(selectedStrings[0])) {
				showLicense();
			}
			else if (UIMessages.Status.equals(selectedStrings[0])) {
				showStatus();
			}
			else if (UIMessages.IncludedPlugins.equals(selectedStrings[0])) {
				showIncludingPlugins();
			}	
			else if (UIMessages.RequiredBy.equals(selectedStrings[0])) {
				showRequiredBy();
			}
			propertiesArea.setSelection(0, 0);
		}

		if (e.widget == backButton.widget) {
			setVisible(false);
			view.showScreen(view.MANAGE_SCREEN, view.NO_REFRESH);
		}
	}

//	public void widgetDefaultSelected(SelectionEvent e) {
//		// forward the event for device environment.
//		if (e.widget == menuList) {
//			widgetSelected(e);
//		}
//	}

	private void showCopyright() {
		IURLEntry copyright = selectedFeature.getCopyright();
		if (copyright != null)
			propertiesArea.setText(copyright.getAnnotation());
		else
			propertiesArea.setText(""); //$NON-NLS-1$
	}

	private void showGeneralInfo() {
		if (generalContent != null) {
			// we don't want to create the same data for the same feature again.
			propertiesArea.setText(generalContent.toString());
			return;
		}

		generalContent = new StringBuffer();

		// feature name
		generalContent.append(UIMessages.Name)
					   .append(selectedFeature.getLabel())
					   .append("\n"); //$NON-NLS-1$

		// feature ID and version
		VersionedIdentifier featureVId = selectedFeature.getVersionedIdentifier();
		if (featureVId != null) {
			generalContent.append(UIMessages.Identifier + featureVId.getIdentifier())
						   .append("\n"); //$NON-NLS-1$
			generalContent.append(UIMessages.Version)
						   .append(featureVId.getVersion())
						   .append("\n"); //$NON-NLS-1$
		}
		else {
			generalContent.append(UIMessages.Identifier)
						   .append("\n"); //$NON-NLS-1$
			generalContent.append(UIMessages.Version)
						   .append("\n"); //$NON-NLS-1$
		}

		// feature provider
		generalContent.append(UIMessages.Provider)
					   .append(selectedFeature.getProvider())
					   .append("\n"); //$NON-NLS-1$

		// feature size
		generalContent.append(UIMessages.Size)
					   .append(selectedFeature.getInstallSize())
					   .append("\n"); //$NON-NLS-1$

		//one blank line for separator
		generalContent.append("\n"); //$NON-NLS-1$

		// feature supported platform
		generalContent.append(UIMessages.SupportedPlatforms)
					   .append("\n"); //$NON-NLS-1$
		generalContent.append(UIMessages.INDENT)
					   .append(UIMessages.OperatingSystem)
					   .append((selectedFeature.getOS()!=null)?selectedFeature.getOS():"") //$NON-NLS-1$
					   .append("\n"); //$NON-NLS-1$
		generalContent.append(UIMessages.INDENT)
					   .append(UIMessages.CPUArchitecture)
					   .append((selectedFeature.getOSArch()!=null)?selectedFeature.getOSArch():"") //$NON-NLS-1$
					   .append("\n"); //$NON-NLS-1$
		generalContent.append(UIMessages.INDENT)
					   .append(UIMessages.WindowingSystem)
					   .append((selectedFeature.getWS()!=null)?selectedFeature.getWS():"") //$NON-NLS-1$
					   .append("\n"); //$NON-NLS-1$
		generalContent.append(UIMessages.INDENT)
					   .append(UIMessages.Languages)
					   .append((selectedFeature.getNL()!=null)?selectedFeature.getNL():"") //$NON-NLS-1$
					   .append("\n"); //$NON-NLS-1$

		//one blank line for separator
		generalContent.append("\n"); //$NON-NLS-1$

		// feature description
		generalContent.append(UIMessages.Description)
					   .append("\n"); //$NON-NLS-1$
		IURLEntry description = selectedFeature.getDescription();
		if (description != null) {
			generalContent.append(UIMessages.INDENT)
						   .append(description.getAnnotation())
						   .append("\n"); //$NON-NLS-1$
		}
		else
			generalContent.append(""); //$NON-NLS-1$

		propertiesArea.setText(generalContent.toString());
	}

	private void showLicense() {
		IURLEntry license = selectedFeature.getLicense();
		if (license != null)
			propertiesArea.setText(license.getAnnotation());
		else
			propertiesArea.setText(""); //$NON-NLS-1$
	}

	private void showStatus() {
		try {
			switch(SiteManager.getLocalSite()
					.getFeatureStatus(selectedFeature)
					.getCode())
			{
			case IFeature.STATUS_HAPPY:
			case IFeature.STATUS_AMBIGUOUS:
				propertiesArea.setText(UIMessages.StatusHappy);
				break;
			case IFeature.STATUS_DISABLED:
				propertiesArea.setText(UIMessages.StatusDisabled);
				break;
			default:
				propertiesArea.setText(UIMessages.StatusUnhappy);
				break;
			}

		}
		catch (CoreException ce) {
			//TODO - Julian: error logging.
			ce.printStackTrace();
		}
	}

	private void showIncludingPlugins() {
		if (pluginsContent != null) {
			// we don't want to create the same data for the same feature again.
			propertiesArea.setText(pluginsContent.toString());
			return;
		}

		pluginsContent = new StringBuffer();
		IPluginEntry[] plugins = selectedFeature.getPluginEntries();

		for (int i=0; i< plugins.length; i++) {
			VersionedIdentifier pluginVId = plugins[i].getVersionedIdentifier();
			if (pluginVId != null) {
				pluginsContent.append(pluginVId.getIdentifier()
						+ " (" + pluginVId.getVersion() + ")"); //$NON-NLS-1$ //$NON-NLS-2$
				pluginsContent.append("\n"); //$NON-NLS-1$
			}
		}
		propertiesArea.setText(pluginsContent.toString());
	}

	private void showRequiredBy() {
		if (requiredByContent != null) {
			// we don't want to create the same data for the same feature again.
			propertiesArea.setText(requiredByContent.toString());
			return;
		}

		requiredByContent = new StringBuffer();
		IPluginEntry[] selectedPlugins = selectedFeature.getPluginEntries();
		IFeature checkedFeature = null;

		boolean sureRequired = false;
		IFeatureReference installedFeature;
		for (int i=0; i<view.installedFeatures.size(); i++) {
			try {
				installedFeature = (IFeatureReference) view.installedFeatures.get(i);
				sureRequired = false;
				if (selectedFeature.getVersionedIdentifier().equals(installedFeature.getVersionedIdentifier())) {
					// skip this checked feature because it's just the selected feature.
					continue;
				}

				checkedFeature = installedFeature.getFeature(null);
				IImport[] importedPlugins = checkedFeature.getImports();
				if (importedPlugins.length == 0) {
					// skip this checked feature because it has no required plugin.
					continue;
				}
				for (int j=0; (j<importedPlugins.length)&&(!sureRequired); j++) {
					for (int k=0; k<selectedPlugins.length; k++) {
						if (importedPlugins[j].getVersionedIdentifier().getIdentifier().equals(selectedPlugins[k].getVersionedIdentifier().getIdentifier())) {
//							System.out.println("[eUpdate Debug] Feature \"" + checkedFeature.getVersionedIdentifier()
//									+ "\" needs Plugin \"" + selectedPlugins[k].getVersionedIdentifier()
//									+ "\" in Feature \"" + selectedFeature.getVersionedIdentifier() + "\"");
							requiredByContent.append(checkedFeature.getLabel()
									+ " (" + checkedFeature.getVersionedIdentifier().getVersion() + ")\n"); //$NON-NLS-1$ //$NON-NLS-2$
							sureRequired = true;
							break;
						}
					}
				}
			}
			catch (CoreException ce) {
				ce.printStackTrace();
			}
		}		
		propertiesArea.setText(requiredByContent.toString());
	}
}
