/*******************************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.ercp.update.views;

import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.ercp.swt.mobile.Command;
import org.eclipse.ercp.swt.mobile.TaskTip;
import org.eclipse.ercp.update.AppPlugin;
import org.eclipse.ercp.update.UIImage;
import org.eclipse.ercp.update.UIMessages;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.TreeEvent;
import org.eclipse.swt.events.TreeListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.update.core.IFeature;
import org.eclipse.update.core.IFeatureReference;
import org.eclipse.update.core.IPluginEntry;
import org.eclipse.update.core.VersionedIdentifier;
import org.eclipse.update.internal.core.InstallRegistry;
import org.eclipse.update.util.CategoryWrap;
import org.eclipse.update.util.FeatureWrap;
import org.eclipse.update.util.INode;
import org.eclipse.update.util.SiteWrap;
import org.osgi.framework.Bundle;


public class SelectFeaturesScreen extends AbsScreen implements TreeListener {

	private Tree featureList;
	private ScrolledText description;
	private Label requiredSpace, freeSpace;
	private Command readDescription;
	private int checkedCount;
	private Vector taggedFeatures = new Vector();
	private UpdateMonitor downloadFeatureJarMonitor;
	private DownloadFeatureJarThread downloadFeatureJarThread;
	private boolean enableEnvFilter = true;
	private Image IMAGE_ICON_SITE		= null;
	private Image IMAGE_ICON_CATEGORY	= null;
	private Image IMAGE_ICON_FEATURE	= null;
	private Image IMAGE_ICON_INCOMPATIBLE_FEATURE	= null;
	private Image IMAGE_ICON_BROKEN_FEATURE			= null;
	private Bundle[] runtime_bundles = null;
	
	public SelectFeaturesScreen(NormalView view) {
		super(view);
	}
	
	protected void createScreen() {	
		init(UIMessages.SelectFeaturesToInstall);
		IMAGE_ICON_SITE		= UIImage.createImage(UIImage.ICON_SITE);
		IMAGE_ICON_CATEGORY	= UIImage.createImage(UIImage.ICON_CATEGORY);
		IMAGE_ICON_FEATURE	= UIImage.createImage(UIImage.ICON_FEATURE);
		IMAGE_ICON_INCOMPATIBLE_FEATURE	= UIImage.createImage(UIImage.ICON_INCOMPATIBLE_FEATURE);
		IMAGE_ICON_BROKEN_FEATURE		= UIImage.createImage(UIImage.ICON_BROKEN_FEATURE);

		Composite subPane = new Composite(screen, SWT.NONE);
		subPane.setLayout(new FormLayout());
		subPane.setLayoutData(new GridData(GridData.FILL_BOTH));

		FormData fdMenu = new FormData();
		FormData fdProperty = new FormData();
		if ((view.parent.getClientArea().height * 2) < view.parent.getClientArea().width) {
			fdMenu.top = new FormAttachment(0, 0);
			fdMenu.bottom = new FormAttachment(100, 0);
			fdMenu.left = new FormAttachment(0, 0);
			fdMenu.right = new FormAttachment(70, -2); //-2 is the reserved margin from menuList to propertiesArea.

			fdProperty.top = new FormAttachment(0, 0);
			fdProperty.bottom = new FormAttachment(100, 0);
			fdProperty.left = new FormAttachment(70, 2); //2 is the reserved margin from propertiesArea to menuList.
			fdProperty.right = new FormAttachment(100, 0);
		}
		else {
			fdMenu.top = new FormAttachment(0, 0);
			fdMenu.bottom = new FormAttachment(70, -2); //-2 is the reserved margin from menuList to propertiesArea.
			fdMenu.left = new FormAttachment(0, 0);
			fdMenu.right = new FormAttachment(100, 0);

			fdProperty.top = new FormAttachment(70, 2); //2 is the reserved margin from propertiesArea to menuList.
			fdProperty.bottom = new FormAttachment(100, 0);
			fdProperty.left = new FormAttachment(0, 0);
			fdProperty.right = new FormAttachment(100, 0);
		}

		featureList = new Tree(subPane, SWT.CHECK ); // add SWT.BORDER on Landscape mode ?
		featureList.addSelectionListener(this);
		featureList.addTreeListener(this);
		featureList.setLayoutData(fdMenu);
	
		description = new ScrolledText(subPane, SWT.NONE, SWT.NONE, -1); // add SWT.BORDER on Landscape mode ?
		description.setLayoutData(fdProperty);

		Composite spaceInfoPane = new Composite(screen, SWT.NONE);
		spaceInfoPane.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		spaceInfoPane.setLayout(new FillLayout(SWT.HORIZONTAL));

		requiredSpace = new Label(spaceInfoPane, SWT.NONE);
		freeSpace = new Label(spaceInfoPane, SWT.NONE);
		spaceInfoPane.layout();

		setButtons(BACK, CANCEL, NEXT);

		if (NormalView.INPUT_SHOW_SOFTKEY_MODE == view.inputType) {
			readDescription = new Command(screen, Command.GENERAL, 10);
			readDescription.setText(UIMessages.ReadDescription);
			readDescription.setLongLabel(UIMessages.ReadDescription);
			readDescription.addSelectionListener(this);
		}

		if (NormalView.OS_ARM_SYMBIAN == view.osType) {
			description.text.setFont(view.fontOnS60);
			requiredSpace.setFont(view.fontOnS60);
			freeSpace.setFont(view.fontOnS60);
		}
//		if (NormalView.INPUT_SHOW_SOFTKEY_MODE == view.inputType) {
//			subPane.setTabList(new Control[] {featureList});
//		}
		screen.layout();

		if (Boolean.valueOf(System.getProperty("org.eclipse.ercp.update.disableEnvFilter")).booleanValue()) { //$NON-NLS-1$
			// the environment filter is enabled by default. (only compatible features can be installed.)
			// disable the filter when specific request. 
			enableEnvFilter = false;
		}
	}

	protected void refresh() {
		checkedCount = 0;
		featureList.removeAll();
		backButton.setEnabled(true);
		nextButton.setEnabled(false);
		cancelButton.setEnabled(false);
		description.setText(""); //$NON-NLS-1$
		for (int i=0; i< view.viewedTree.size(); i++) {
			SiteWrap sw = (SiteWrap) view.viewedTree.elementAt(i);
			CategoryWrap[] cws = sw.getChildren();
			if (cws.length == 0)
				continue;
//			if (sw.getUiObject()!=null) {
//				// It may be LicenseScreen backs to selectFeatureScreen,
//				// so no need to refresh.
//				break;
//			}

			TreeItem siteBranch = new TreeItem(featureList, SWT.NONE);
			if (NormalView.OS_ARM_SYMBIAN == view.osType)
				siteBranch.setFont(view.fontOnS60);
			siteBranch.setText(sw.getName());
			siteBranch.setImage(IMAGE_ICON_SITE);
			sw.setUiObject(siteBranch);

			for (int j=0; j< cws.length; j++) {
				CategoryWrap cw = cws[j];
				TreeItem categoryBranch = new TreeItem((TreeItem) sw.getUiObject(), SWT.NONE);
				if (NormalView.OS_ARM_SYMBIAN == view.osType)
					categoryBranch.setFont(view.fontOnS60);
				categoryBranch.setText(cw.getLabel());
				categoryBranch.setImage(IMAGE_ICON_CATEGORY);
				cw.setUiObject(categoryBranch);

				FeatureWrap[] fws = cw.getChildren();
				for (int k=0; k< fws.length; k++) {
					FeatureWrap fw = fws[k];
					TreeItem leaf = new TreeItem((TreeItem) cw.getUiObject(), SWT.NONE);
					if (NormalView.OS_ARM_SYMBIAN == view.osType)
						leaf.setFont(view.fontOnS60);
					leaf.setText(fw.getShowedName());
					leaf.setImage(IMAGE_ICON_FEATURE);
					fw.addUiObject(leaf);
				}
			}
			if (cws.length == 0) {
				siteBranch.setGrayed(true);
			}
		}
		updateSpaceIndicators();

		if (NormalView.RUNTIME_TITAN == view.runtimeType) {
			// get the list of all current bundles in runtime, will need it later.
			runtime_bundles = AppPlugin.getDefault().getBundle().getBundleContext().getBundles();
		}
	}

	protected void action() {
		setDefaultFocus();
		screen.pack();
	}

	protected void setDefaultFocus() {
		featureList.setFocus();
	}

	private boolean updateSpaceIndicators() {
//		final String KILOBYTE = "KB"; //$NON-NLS-1$
//		final String MEGABYTE = "MB"; //$NON-NLS-1$
		Object[] requiredArgs = new Object[2];
		Object[] freeArgs = new Object[2];

		long requiredSize = 0;
		checkedCount = 0;
		nextButton.setEnabled(false);
		for (int k=0; k< view.availableFeatures.size(); k++) {
			FeatureWrap fw = (FeatureWrap) view.availableFeatures.elementAt(k);
			if (fw.isSelected()) {
				checkedCount++;
				requiredSize += fw.getFeatureSize();
				// next button is enable if at least one feature is checked.
				nextButton.setEnabled(true);
			}
		}

		if (requiredSize > 1024) {
			// space value is greater than 1024(KB), show it in MB.
			requiredArgs[0] = (new Long(requiredSize>>10)).toString();
			requiredArgs[1] = UIMessages.MegaBytes;
			requiredSpace.setText(NLS.bind(UIMessages.RequiredSpace, requiredArgs));
		}
		else {
			// show the space value in KB
			requiredArgs[0] = (new Long(requiredSize)).toString();
			requiredArgs[1] = UIMessages.KiloBytes;
			requiredSpace.setText(NLS.bind(UIMessages.RequiredSpace, requiredArgs));
		}

		if ((NormalView.OS_ARM_SYMBIAN == view.osType) || (NormalView.RUNTIME_TITAN == view.runtimeType)){
			if (requiredSize > 10240) {
				MessageBox mb1 = new MessageBox(screen.getShell(), SWT.ICON_WARNING | SWT.OK);
				//mb1.setText(UIMessages.ExceedLimitedSize);
				mb1.setMessage(UIMessages.ExceedLimitedSize);
				if (mb1.open() == SWT.OK) {
					return false;
				}
			}
		}

		long availableSize = -1;
		try {
			// because no regular way to get disk info on java,
			// here is a little hack to get recent free disk space info from caller via system property.

			availableSize = Long.valueOf(System.getProperty("wed.eUpdate.availableSpace")).longValue() >> 10; //$NON-NLS-1$
			// shifting because need value in KB instead of Byte.

			if (availableSize <= requiredSize)
				nextButton.setEnabled(false);

			if (availableSize > 1024) {
				// space value is greater than 1024(KB), show it in MB.
				freeArgs[0] = (new Long(availableSize>>10)).toString();
				freeArgs[1] = UIMessages.MegaBytes;
				freeSpace.setText(NLS.bind(UIMessages.FreeSpace, freeArgs));
			}
			else {
				// show the space value in KB
				freeArgs[0] = (new Long(availableSize)).toString();
				freeArgs[1] = UIMessages.KiloBytes;
				freeSpace.setText(NLS.bind(UIMessages.FreeSpace, freeArgs));
			}
		}
		catch (NumberFormatException nfe) {
			// no disk info gotten, no showing.
			freeSpace.setText(""); //$NON-NLS-1$
		}catch (Exception eee) { eee.printStackTrace();}
		
		return true;
	}
	
	public void widgetSelected(SelectionEvent e) {
		if (e.widget == featureList) {
			// if thread is running to get data, then skip this event.
			if ((downloadFeatureJarThread != null)
					&& (downloadFeatureJarThread.getStatus() == DownloadFeatureJarThread.RUNNING)) {
				return;
			}

			// identify the selected item
			TreeItem item = (TreeItem)e.item;
			if (item == null) {
				// clean the description field.
				description.setText(""); //$NON-NLS-1$
				return;
			}
			INode selectedNode = null;
			for (int i=0; i< view.viewedTree.size(); i++) {
				selectedNode = ((INode)view.viewedTree.elementAt(i)).findByUiObject(item);
				if (selectedNode != null)
					break;
			}

			if (e.detail == SWT.CHECK) {
				//description.setText(""); //$NON-NLS-1$
				taggedFeatures.removeAllElements();
				if (item.getChecked()) {
					rememberCurrentChecks();
				}

				if (selectedNode instanceof SiteWrap) {
					checkSiteNode((SiteWrap) selectedNode, item.getChecked());
				}
				else if (selectedNode instanceof CategoryWrap) {
					checkCategoryNode((CategoryWrap) selectedNode, item.getChecked());
				}
				else if (selectedNode instanceof FeatureWrap) {
					// if it's grayed, then it's the filtered feature, don't allow it to be checked.
					// Widget "TreeItem" has no disable attribute, so we use gray instead of.
					if (item.getGrayed()) {
						description.setText(UIMessages.FeatureIsIncompatible);
						description.setSelection(0, 0);
						item.setChecked(false);
						return;
					}
					checkFeatureNode((FeatureWrap) selectedNode, item.getChecked());
				}
				downloadNeededFeatureInfo();
			}
			else {
				//Not CHECK, it should be a SELECT event.

//				// check if this item is feature item and is ready to be used.
//				if (selectedNode instanceof FeatureWrap) {
//					FeatureWrap fw = (FeatureWrap) selectedNode;
//					if (!fw.isFeatureReady()) {
//						fw.downloadFeature(null);
//						item.setText(fw.getShowedName());
//					}
//				}

				//if (featureList.getSelectionCount() == 1) {
					description.setText(selectedNode.getDescription());
					screen.layout();
					description.setSelection(0, 0);
				//}
			}
			return;
		}
		else if (e.widget == backButton.widget) {
			setVisible(false);
			view.showScreen(view.BOOKMARKS_SCREEN, view.NO_REFRESH);
		}
		else if (e.widget == cancelButton.widget) {
			if (downloadFeatureJarMonitor != null)
				downloadFeatureJarMonitor.setCanceled(true);
		}
		else if (e.widget == nextButton.widget) {
			setVisible(false);
			view.showScreen(view.LICENSE_SCREEN, view.REFRESH_IT);
		}
		else if (e.widget == readDescription) {
			description.setFocus();
			description.setSelection(0, 0);
		}
	}

	public void widgetDefaultSelected(SelectionEvent e) {
		if (e.widget == featureList) {
			widgetSelected(e);
		}
	}

	/*
	 * 
	 */
	private boolean isIncompatibleFeature(FeatureWrap fw) {
		if (enableEnvFilter) {
			IFeature f = fw.getFeature();
			if ((f.getOS() != null)
					&& isNotInSupportedList(f.getOS(), Platform.getOS())) {
					return true;
			}
			if ((f.getWS() != null)
					&& isNotInSupportedList(f.getWS(), Platform.getWS())) {
					return true;
			}
			if ((f.getOSArch() != null)
					&& isNotInSupportedList(f.getOSArch(), Platform.getOSArch())) {
					return true;
			}
			if ((f.getNL() != null)
					&& isNotInSupportedLanguageList(f.getNL(), Platform.getNL())) {
					return true;
			}
			if ((NormalView.RUNTIME_TITAN == view.runtimeType)
					&& isThereDuplicatePlugins(fw)) {
					return true;
			}
		}
		return false;
	}

	/*
	 * 
	 */
	private boolean isNotInSupportedList(String supportList, String checked) {
		StringTokenizer st = new StringTokenizer(supportList.trim(), ","); //$NON-NLS-1$
		if (st.countTokens() == 0)
			return false; //empty list means it supports all.

		while (st.hasMoreTokens()) {
			if (checked.equalsIgnoreCase(st.nextToken().trim())) {
				return false; // 'false' means 'supported'
			}
		}
		return true; // 'true' means 'insupported'
	}

	/*
	 * 
	 */
	private boolean isThereDuplicatePlugins(FeatureWrap fw) {
		if (fw.getFeatureUniquenessStatus() == FeatureWrap.FEATURE_BRAND_NEW) {
			IPluginEntry[] pes = fw.getFeature().getPluginEntries();
			for (int i=0; i< pes.length; i++) {
				for (int j=80; j< runtime_bundles.length; j++) {
					//if (pes[i].getVersionedIdentifier().getIdentifier().equalsIgnoreCase(runtime_bundles[j].getSymbolicName())) {
					if ((runtime_bundles[j].getSymbolicName() != null)
						&&	(runtime_bundles[j].getSymbolicName().startsWith(pes[i].getVersionedIdentifier().getIdentifier()))) {
						// TODO - Julian: what should we do if ID is not the same as Symbolic Name ?

						if (InstallRegistry.getInstance().get("plugin_"+pes[i].getVersionedIdentifier()) == null) { //$NON-NLS-1$
							return true;
						}
						// 'true' means at least one existed plug-in in runtime has the same ID with some plug-in in this feature,
						// and update.core has no installation record for it.
						// This duplicate plug-in may be installed before via other installation tool.
					}
				}
			}
		}

		return false; // 'false' means no existed plug-in in runtime has the same ID with every plug-ins in this feature.
	}

	/**
	 * 
	 * @param supportList	The list including all supported items
	 * @param checked	The item needed to check. It must be the string in low case
	 * @return	true: it means the checked item is NOT supported.<br>false: it means the checked iterm is supported. 
	 * 
	 */
	private boolean isNotInSupportedLanguageList(String supportList, String checked) {
		StringTokenizer st = new StringTokenizer(supportList.trim(), ","); //$NON-NLS-1$
		if (st.countTokens() == 0)
			return false; //empty list means it supports all.

		while (st.hasMoreTokens()) {
			String token = st.nextToken().trim();
			if (token.indexOf('_') > -1) { //$NON-NLS-1$
				// this token in supportedList includes '_',
				// so 'supported' needs both language and country are matched.
				if (checked.equals(token)) {
					return false; // 'false' means 'supported'
				}				
			}
			else {
				// this token in supportedList does NOT include '_',
				// so 'supported' just needs the language is matched.
				if (checked.startsWith(token)) {
					return false; // 'false' means 'supported'
				}				
			}
		}
		return true; // 'true' means 'insupported'
	}

	/*
	 * 
	 */
	private void checkSiteNode(SiteWrap sw, boolean checked) {
		TreeItem siteyNode = (TreeItem)sw.getUiObject();
		siteyNode.setChecked(checked);
		if (checked)
			siteyNode.setExpanded(true);

		sw.setSelected(checked);

		CategoryWrap[] cws = sw.getChildren();
		for (int i=0; i< cws.length; i++) {
			checkCategoryNode(cws[i], checked);
		}
	}

	/*
	 * 
	 */
	private void checkCategoryNode(CategoryWrap cw, boolean checked) {
		TreeItem categoryNode = (TreeItem)cw.getUiObject();
		categoryNode.setChecked(checked);
		if (checked)
			categoryNode.setExpanded(true);

		cw.setSelected(checked);

		FeatureWrap[] fws = cw.getChildren();
		for (int i=0; i< fws.length; i++) {
			checkFeatureNode(fws[i], checked);
		}
	}

	/*
	 * 
	 */
	private void checkFeatureNode(FeatureWrap fw, boolean checked) {
		if(checked && !fw.isFeatureReady()) {
			// always tag it no matter it's filtered feature or not.
			taggedFeatures.add(fw);
		}

		if (fw.isFeatureReady() && isIncompatibleFeature(fw)) {
			// exit because it's not allowable to check/uncheck the incompatibile feature.
			fw.setSelected(false);
			return;
		}
		// it's possible that the feature is not ready now but it is incompatible. 
		// we leave the check-work with the method downloadNeededFeatureInfo() later.

		fw.setSelected(checked);

		// one feature may be belonged to several categories.
		Vector featureNodes = fw.getUiObjects();
		for (int i=0; i<featureNodes.size(); i++) {
			((TreeItem) featureNodes.elementAt(i)).setChecked(checked);
		}
	}

	/*
	 * 
	 */
	private void finalizeAllCheckeds() {
		for (int i=0; i< view.viewedTree.size(); i++) {
			SiteWrap sw = (SiteWrap) view.viewedTree.elementAt(i);
			boolean siteFinalChecked = false;

			CategoryWrap[] cws = sw.getChildren();
			for (int j=0; j< cws.length; j++) {
				boolean categoryFinalChecked = false;

				FeatureWrap[] fws = cws[j].getChildren();
				for (int k=0; k<fws.length; k++) {
					if (fws[k].isSelected()) {
						categoryFinalChecked = true;
						siteFinalChecked = true;
						break;
					}
				}
				cws[j].setSelected(categoryFinalChecked);
				((TreeItem)cws[j].getUiObject()).setChecked(categoryFinalChecked);
			}
			if (cws.length > 0) {
				// for the site with no available category/feature, it also has no UI Object.
				sw.setSelected(siteFinalChecked);
				((TreeItem)sw.getUiObject()).setChecked(siteFinalChecked);
			}
		}
	}

	/**
	 * Checks whether the required feature with required version limitation has
	 * been installed.
	 * 
	 * @param requiredVID The VersionedIdentifier of the checked feature.
	 * @return <code>true</code> if the required feature is available,
	 * otherwise <code>false</code> if it is unavailable.
	 */
	private boolean isRequiredFeatureInstalled(VersionedIdentifier requiredVID) {
		VersionedIdentifier installedVID;

		for (int i=0; i<view.installedFeatures.size(); i++) {
			try {
				installedVID = ((IFeatureReference)view.installedFeatures.get(i)).getVersionedIdentifier();
			} catch (CoreException ce) {
				// this feature doesn't work, so skip it.
				ce.printStackTrace();
				continue;
			}
			if (installedVID.getIdentifier().equals(requiredVID.getIdentifier())) {
				//1. have the same ID now.
				if (installedVID.getVersion().isGreaterOrEqualTo(requiredVID.getVersion())) {
					//2. have the version greater than or equal to the required version.
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * Searches for the feature from the current available list, and add it into
	 * the selection list to be installed.
	 * 
	 * @param requiredVID The VersionedIdentifier of the required feature.
	 * @return <code>true</code> if the feature is available and has been added
	 * successfuly, otherwise <code>false</code> if this feature is unavailable.
	 */
	private boolean addRequiredFeatureInto(VersionedIdentifier requiredVID) {
		FeatureWrap fw;
		VersionedIdentifier availableVID;

		for (int i=0; i< view.availableFeatures.size(); i++) {
			fw = (FeatureWrap) view.availableFeatures.elementAt(i);
			availableVID = fw.getVID();
			if (availableVID.getIdentifier().equals(requiredVID.getIdentifier())) {
				//1. have the same ID now.
				if (availableVID.getVersion().isGreaterOrEqualTo(requiredVID.getVersion())) {
					//2. have the version greater than or equal to the required version.
					if (!fw.isSelected()) {
						// the required feature is available and not selected yet.
						// go to select it (need to update the screen).
						if(isIncompatibleFeature(fw)) {
							// got one available feature fitting the requirement, but it's not for current platform.
							// try next.
							// it should be a rare case, but check is better.
							continue;
						}
						else {
							// we got it now.
							fw.setSelected(true);
							Vector featureNodes = fw.getUiObjects();
							for (int j=0; j<featureNodes.size(); j++) {
								// one feature node can be belonged to more than one category node.
								((TreeItem) featureNodes.elementAt(j)).setChecked(true);
								((TreeItem) featureNodes.elementAt(j)).setText(fw.getShowedName());
							}
						}
					}
					//else {
						// the required feature is available and has been selected.
						// nothing to do now.
					//}
					return true;
				}
			}
		}
		return false;
	}

	/*
	 * 
	 */
	private void resolveDependency() {
		StringBuffer dependencyLog = new StringBuffer();

		for (int i=0; i<view.availableFeatures.size(); i++) {
			FeatureWrap fw = (FeatureWrap) view.availableFeatures.elementAt(i);
			if (fw.isSelected()) {
				IFeatureReference[] children;
				try {
					children = fw.getFeature().getIncludedFeatureReferences();
				}
				catch (Exception e) {
					e.printStackTrace();
					continue; // try next.
				}

				for (int j=0; j<children.length; j++) {
					try {
						if (!isRequiredFeatureInstalled(children[j].getVersionedIdentifier())) {
							// the required feature is a new one (i.e., not installed yet).
							// going to find it from update site and add it to the installed list.
							if (!addRequiredFeatureInto(children[j].getVersionedIdentifier())) {
								// this required (children) feature is not available, so mark the parent feature as broken.
								Vector featureNodes = fw.getUiObjects();
								for (int k=0; k<featureNodes.size(); k++) {
									// one feature node can be belonged to more than one category.
									((TreeItem) featureNodes.elementAt(k)).setImage(IMAGE_ICON_BROKEN_FEATURE);
								}
								// also, log it.
								dependencyLog.append(NLS.bind(UIMessages.NestedFeatureUnavailable, new String[] {children[j].getName()}))
											 .append("\n"); //$NON-NLS-1$
							}
						}
						//else {
							// the required feature has been installed.
							// nothing to do now.
						//}
					}
					catch (CoreException ce2) {
						ce2.printStackTrace();
						continue; // try next child.
					}
				}
			}
		}
		if (dependencyLog.length() > 0) {
			// some error was logged, show it.
			description.setText(dependencyLog.toString());
			description.setSelection(0, 0);
		}
	}

	/*
	 * 
	 */
	private void rememberCurrentChecks() {
		for (int i=0; i< view.viewedTree.size(); i++) {
			SiteWrap sw = (SiteWrap) view.viewedTree.elementAt(i);
			sw.setPreviousSelected(sw.isSelected());
			CategoryWrap[] cws = sw.getChildren();
			for (int j=0; j< cws.length; j++) {
				CategoryWrap cw = cws[j];
				cw.setPreviousSelected(cw.isSelected());
				FeatureWrap[] fws = cw.getChildren();
				for (int k=0; k< fws.length; k++) {
					FeatureWrap fw = fws[k];
					fw.setPreviousSelected(fw.isSelected());
				}
			}
		}
	}

	/*
	 * 
	 */
	private void recoverPreviousChecks() {
		for (int i=0; i< view.viewedTree.size(); i++) {
			SiteWrap sw = (SiteWrap) view.viewedTree.elementAt(i);
			sw.setSelected(sw.isPreviousSelected());
			((TreeItem) sw.getUiObject()).setChecked(sw.isPreviousSelected());

			CategoryWrap[] cws = sw.getChildren();
			for (int j=0; j< cws.length; j++) {
				CategoryWrap cw = cws[j];
				cw.setSelected(cw.isPreviousSelected());
				((TreeItem) cw.getUiObject()).setChecked(cw.isPreviousSelected());
				((TreeItem) cw.getUiObject()).setExpanded(false);				

				FeatureWrap[] fws = cw.getChildren();
				for (int k=0; k< fws.length; k++) {
					FeatureWrap fw = fws[k];
					fw.setSelected(fw.isPreviousSelected());
					Vector treeItems = fw.getUiObjects();
					for (int l=0; l< treeItems.size(); l++) {
						((TreeItem) treeItems.elementAt(l)).setChecked(fw.isPreviousSelected());
					}
				}
			}
		}
	}

	/*
	 * 
	 */
	private void downloadNeededFeatureInfo() {
		final boolean previousNextBoolean = nextButton.getEnabled();
		if (taggedFeatures.size() < 1) {
			// no need to download, just calculate and show the size value.
			resolveDependency();
			finalizeAllCheckeds();
			if (updateSpaceIndicators() == false) {
				recoverPreviousChecks();
				nextButton.setEnabled(previousNextBoolean);
				updateSpaceIndicators();
			}
			return;
		}

		featureList.setEnabled(false);
		backButton.setEnabled(false);
		cancelButton.setEnabled(true);
		nextButton.setEnabled(false);

		final TaskTip progress = new TaskTip(screen.getShell(), SWT.INDETERMINATE);
		progress.setText(UIMessages.DownloadFeatureInfo);
		progress.setVisible(true);

		// while downloading, the default focus is on the cancel button.
		if (view.inputType != NormalView.INPUT_SHOW_SOFTKEY_MODE)
			cancelButton.setFocus();
		else
			screen.setFocus(); // on SoftKey mode, move focus from TaskTip to main screen.

		downloadFeatureJarMonitor = new UpdateMonitor();
		downloadFeatureJarThread = new DownloadFeatureJarThread("", //$NON-NLS-1$
				taggedFeatures, downloadFeatureJarMonitor);
		downloadFeatureJarThread.start();

		Display.getCurrent().timerExec(400, new Runnable() {
			public void run() {
				if (downloadFeatureJarThread.isAlive()) {
					Display.getCurrent().timerExec(400, this);
				}
				else {
					progress.setVisible(false);
					progress.dispose();
					if (downloadFeatureJarThread.getStatus() == DownloadFeatureJarThread.SUCCESS) {
						resolveDependency();
						finalizeAllCheckeds();
						if (updateSpaceIndicators() == false) {
							recoverPreviousChecks();
							nextButton.setEnabled(previousNextBoolean);
							updateSpaceIndicators();
						}
					}
					else {
						recoverPreviousChecks();
						nextButton.setEnabled(previousNextBoolean);
					}

					featureList.setEnabled(true);
					backButton.setEnabled(true);
					cancelButton.setEnabled(false);
					downloadFeatureJarThread = null;
					downloadFeatureJarMonitor = null;

					// update screen
					boolean isCompatible;
					boolean isAnyoneUnchecked = false;
					FeatureWrap fw;
					TreeItem tItem;
					for (int i=0; i< taggedFeatures.size(); i++) {
						fw = (FeatureWrap) taggedFeatures.elementAt(i);
						if (fw.isFeatureReady()) {
							isCompatible = true;
							if (isIncompatibleFeature(fw)) {
								isCompatible = false;
								fw.setSelected(false); // it may be selected when it was not ready, so we need to unselect it.
							}
							Vector treeItems = fw.getUiObjects();
							for (int j=0; j< treeItems.size(); j++) {
								tItem = (TreeItem) treeItems.elementAt(j);
								tItem.setText(fw.getShowedName());
								if (!isCompatible) {
									tItem.setChecked(false);
									tItem.setGrayed(true);
									tItem.setImage(IMAGE_ICON_INCOMPATIBLE_FEATURE);
									isAnyoneUnchecked = true;
								}
							}
						}
						else {
							// Some error occurred (while downloading ?), so feature is not ready here. Mark it as broken.
							Vector treeItems = fw.getUiObjects();
							for (int j=0; j< treeItems.size(); j++) {
								tItem = (TreeItem) treeItems.elementAt(j);
								tItem.setChecked(false);
								tItem.setGrayed(true);
								tItem.setImage(IMAGE_ICON_BROKEN_FEATURE);
								isAnyoneUnchecked = true;
							}
						}
					}
					// For SPR JCSY7KPC2U, add more comments later.
					if (isAnyoneUnchecked) {
						finalizeAllCheckeds();
						updateSpaceIndicators();
					}

					featureList.setFocus();
				}
			}
		});	
	}

	/*
	 * 
	 */
	public void treeCollapsed(TreeEvent te) {
		// do nothing
	}

	/*
	 * 
	 */
	public void treeExpanded(TreeEvent te) {
		final TreeItem item = (TreeItem)te.item;
		if (item == null) {
			return;
		}

		// identify the expanded item
		INode expandedNode = null;
		for (int i=0; i< view.viewedTree.size(); i++) {
			expandedNode = ((INode)view.viewedTree.elementAt(i)).findByUiObject(item);
			if (expandedNode != null)
				break;
		}

		// only handle the category item
		if (expandedNode instanceof CategoryWrap) {
			taggedFeatures.removeAllElements();
			final FeatureWrap[] fws = ((CategoryWrap)expandedNode).getChildren();
			for (int i=0; i< fws.length; i++) {
				if (!fws[i].isFeatureReady()) {
					//only add it to the list if data is not downloaded yet.
					taggedFeatures.add(fws[i]);
				}
			}

			if (taggedFeatures.size() < 1)
				return;

			// looks like some "feature.xml"s are not downloaded yet.
			// start to download data to show them with all necessary info.
			if (Platform.OS_WIN32.equals(Platform.getOS())
					&& Platform.ARCH_X86.equals(Platform.getOSArch())) {
				featureList.setEnabled(false);
			}
			else {
				// Don't know why the system on device can't receive any UI event if we call the setEnable(false) here.
				// The workaround: call setVisible(false) instead of setVisible(false).
				featureList.setVisible(false);
			}
			backButton.setEnabled(false);
			cancelButton.setEnabled(true);
			final boolean previousNextBoolean = nextButton.getEnabled();
			nextButton.setEnabled(false);

			final TaskTip progress = new TaskTip(screen.getShell(), SWT.INDETERMINATE);
			progress.setText(UIMessages.DownloadFeatureInfo);
			progress.setVisible(true);

			// while downloading, the default focus is on the cancel button.
			if (view.inputType != NormalView.INPUT_SHOW_SOFTKEY_MODE)
				cancelButton.setFocus();
			else
				screen.setFocus(); // on SoftKey mode, move focus from TaskTip to main screen.

			downloadFeatureJarMonitor = new UpdateMonitor();
			downloadFeatureJarThread = new DownloadFeatureJarThread("", //$NON-NLS-1$
					taggedFeatures, downloadFeatureJarMonitor);
			downloadFeatureJarThread.start();

			Display.getCurrent().timerExec(400, new Runnable() {
				public void run() {
					if (downloadFeatureJarThread.isAlive()) {
						Display.getCurrent().timerExec(400, this);
					}
					else {
						progress.setVisible(false);
						progress.dispose();
						if (downloadFeatureJarThread.getStatus() == DownloadFeatureJarThread.SUCCESS) {
						}
						else {
							item.setExpanded(false);
						}

						if (Platform.OS_WIN32.equals(Platform.getOS())
								&& Platform.ARCH_X86.equals(Platform.getOSArch())) {
							featureList.setEnabled(true);
						}
						else {
							// Because calling setEnabled() has problem, we call setVisible() here.
							featureList.setVisible(true);
						}
						backButton.setEnabled(true);
						cancelButton.setEnabled(false);
						nextButton.setEnabled(previousNextBoolean);
						downloadFeatureJarThread = null;
						downloadFeatureJarMonitor = null;

						// update screen
						boolean isCompatible = true;
						FeatureWrap fw;
						TreeItem tItem;
						for (int i=0; i< taggedFeatures.size(); i++) {
							fw = (FeatureWrap) taggedFeatures.elementAt(i);
							if (fw.isFeatureReady()) {
								isCompatible = true; 
								if (isIncompatibleFeature(fw)) {
									isCompatible = false;
									fw.setSelected(false); // should be unnecessary, just be safer.
								}
								Vector treeItems = fw.getUiObjects();
								for (int j=0; j< treeItems.size(); j++) {
									tItem = (TreeItem) treeItems.elementAt(j);
									tItem.setText(fw.getShowedName());
									if (!isCompatible) {
										tItem.setGrayed(true);
										tItem.setImage(IMAGE_ICON_INCOMPATIBLE_FEATURE);
										tItem.setChecked(false); // should be unnecessary, just be safer.
									}
								}
							}
						}
						featureList.setFocus();
					}
				}
			});
		}
	}

	private class DownloadFeatureJarThread extends Thread {

		public static final int NONE		= 0;
		public static final int RUNNING		= 1;
		public static final int SUCCESS		= 2;		// thread is finished and got all necessary data.
		public static final int ENDED		= 3;		// thread is finished because the user terminated it or error occured.

		private Vector tags;
		private IProgressMonitor monitor;
		private int currentStatus;

		public DownloadFeatureJarThread(String name,
									Vector taggs,			//input
									IProgressMonitor downloadMonitor) {
			super(name);
			this.tags = taggs;
			this.monitor = downloadMonitor;
			currentStatus = NONE;
		}

		public void run() {
			try {
				currentStatus = RUNNING;

				FeatureWrap fw;
				for (int i=0; !monitor.isCanceled()&& (i< tags.size()); i++) {
					fw = (FeatureWrap) tags.elementAt(i);
					fw.downloadFeature(monitor);

					// System resource on device is limited,
					// so we need to reservice some CPU time for UI actions. 
					try {
						sleep(200);
					}
					catch (Exception e) {}
				}

				if (monitor.isCanceled())
					currentStatus = ENDED;
				else
					currentStatus = SUCCESS;
			}
			finally {
				// make sure the status is set no matter any error.
				if (currentStatus != SUCCESS)
					currentStatus = ENDED;
			}
		}

		/*
		 * 
		 */
		public int getStatus() {
			return currentStatus;
		}
	}
}
