/*******************************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.ercp.update.views;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.ercp.swt.mobile.Command;
import org.eclipse.ercp.update.UIMessages;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.ui.PlatformUI;
import org.eclipse.update.configuration.IConfiguredSite;
import org.eclipse.update.configuration.IInstallConfiguration;
import org.eclipse.update.configuration.ILocalSite;
import org.eclipse.update.core.IFeature;
import org.eclipse.update.core.IFeatureReference;
import org.eclipse.update.core.SiteManager;
import org.eclipse.update.internal.core.UpdateCore;
import org.eclipse.update.internal.operations.DuplicateConflictsValidator;
import org.eclipse.update.internal.operations.IUnconfigureAndUninstallFeatureOperation;
import org.eclipse.update.internal.operations.OperationFactory;
import org.eclipse.update.internal.operations.UpdateUtils;
import org.eclipse.update.operations.IBatchOperation;
import org.eclipse.update.operations.IInstallFeatureOperation;
import org.eclipse.update.operations.OperationsManager;
import org.eclipse.update.search.IUpdateSearchResultCollector;
import org.eclipse.update.search.UpdateSearchRequest;


public class UpdatingScreen extends AbsScreen {

	private final static int PROGRESS_UNIT = 10; // to get more accurate progress of updating.

	private Composite pane;
	private Label updatingLabel;
	private Label updatingItem;
	private ProgressBar updateProgress;
	private Command cancelCommand;
	private UpdateSearchRequest searchRequest;
	private UpdateMonitor updateFeatureMonitor;
	private UpdateFeaturesThread updateFeaturesThread;
	private int totalSearchedFeaturesCount = 0;
	private boolean isSingleThread = true;
	private int buttonSet = 0;
//Remove the following code when duplicated-timerExec() bug is fixed.
	private boolean isExisted = false;
//End.

	UpdatingScreen(NormalView view) {
		super(view);
	}
	
	protected void createScreen() {
		init(UIMessages.UpdateAllFeatures);
		
		pane = new Composite(screen, SWT.NONE);
		pane.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		pane.setLayout(new RowLayout(SWT.HORIZONTAL));
		
		updatingLabel = new Label(pane, SWT.NONE);
		updatingLabel.setText(UIMessages.Updating);
		updatingItem = new Label(pane, SWT.NONE);
		updateProgress = new ProgressBar(screen, SWT.SMOOTH);
		updateProgress.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		fillSpace();
		if (NormalView.WM_SMARTPHONE == view.wmType) {
			cancelCommand = new Command(screen, Command.GENERAL, 10);
			cancelCommand.setText(UIMessages.Cancel);
			cancelCommand.setLongLabel(UIMessages.Cancel);
			cancelCommand.addSelectionListener(this);
		}
		setButtons(NONE, CANCEL, NONE);

		if (NormalView.OS_ARM_SYMBIAN == view.osType) {
			updatingLabel.setFont(view.fontOnS60);
			updatingItem.setFont(view.fontOnS60);
		}
		screen.layout();
	}

	protected void refresh() {
		if (NormalView.WM_SMARTPHONE == view.wmType)
			cancelCommand.setEnabled(true);
		cancelButton.setEnabled(true);
		updatingItem.setText(""); //$NON-NLS-1$
		updateProgress.setSelection(0);
		pane.layout();// force to re-update the screen
	}

	protected void action() {
		if (view.availableFeatures.size()==0) {
			isSingleThread = false;
			IFeature[] allInstalledFeatures = new IFeature[view.installedFeatures.size()];
			for (int i=0; i< view.installedFeatures.size(); i++) {
				try {
					allInstalledFeatures[i] = ((IFeatureReference)view.installedFeatures.get(i)).getFeature(null);
					if (UpdateCore.DEBUG) {
						System.out.println("[ eUpdate debug ] Feature being assigned to search: " + allInstalledFeatures[i].getLabel());
					}
				} catch (CoreException ce) {
					//TODO - Julian: error logging.
					ce.printStackTrace();
				}
			}
			searchRequest = UpdateUtils.createNewUpdatesRequest(allInstalledFeatures);
		}
		else {
			isSingleThread = true;
			IFeature f = (IFeature) view.availableFeatures.elementAt(0);
			if (UpdateCore.DEBUG) {
				System.out.println("[ eUpdate debug ] Feature being assigned to search: " + f.getLabel());
			}
			searchRequest = UpdateUtils.createNewUpdatesRequest(new IFeature[] {f});
		}
		totalSearchedFeaturesCount = searchRequest.getCategory().getQueries().length;

		if (totalSearchedFeaturesCount == 0) {
			// no installed feature in update list.
			// all disabled or patched features are ignored. 
			MessageBox mb2 = new MessageBox(screen.getShell(), SWT.ICON_INFORMATION | SWT.OK);
			mb2.setText(UIMessages.UpdateStatus);
			mb2.setMessage(UIMessages.NoUpdateAvailable);
			mb2.open();
			setVisible(false);
			view.showScreen(view.ROOT_SCREEN, view.REFRESH_IT);
			return;
		}

//		try { //For Debugging
//			org.eclipse.update.internal.search.UpdatesSearchCategory usc = (org.eclipse.update.internal.search.UpdatesSearchCategory) searchRequest.getCategory();
//			usc.getQueries();
//			org.eclipse.update.search.IUpdateSearchQuery[] searchQueries = usc.getQueries();
//			for (int i=0; i< searchQueries.length; i++) {
//				org.eclipse.update.internal.search.UpdatesSearchCategory.UpdateQuery query = (org.eclipse.update.internal.search.UpdatesSearchCategory.UpdateQuery) searchQueries[i];
//				System.out.println("[ eUpdate debug ] Final search candidate[" + i + "] is " + query.candidate.getLabel());
//				org.eclipse.update.search.IQueryUpdateSiteAdapter searchSite = query.getQuerySearchSite();
//				if (searchSite == null) {
//					System.out.println("[ eUpdate debug ] And the update site Label     is NULL");
//					System.out.println("[ eUpdate debug ] And the update site URL       is NULL");
//					System.out.println("[ eUpdate debug ] And the update site MappingID is NULL");
//				}
//				else {
//					System.out.println("[ eUpdate debug ] And the update site Label     is " + searchSite.getLabel());
//					System.out.println("[ eUpdate debug ] And the update site URL       is " + searchSite.getURL().toString());
//					System.out.println("[ eUpdate debug ] And the update site MappingID is " + searchSite.getMappingId());
//				}
//			}
//		} catch (Exception e) {e.printStackTrace();}		

		updateFeatureMonitor = new UpdateMonitor();

		// check if the config file has been modifed while we were running
		IStatus status = OperationsManager.getValidator().validatePlatformConfigValid();
		if (status != null) {
			//TODO - Julian: error logging.
			System.out.println(status.getMessage());
			return;
		}

		// update data has been collected, going to perform update actions.
		updateProgress.setMaximum((totalSearchedFeaturesCount + 1) * PROGRESS_UNIT);
		updateFeaturesThread =
			new UpdateFeaturesThread("", searchRequest, view.installedFeatures, updateFeatureMonitor); //$NON-NLS-1$
		updateFeaturesThread.start();

		setDefaultFocus();

		Display.getCurrent().timerExec(1000, new Runnable() {
			public void run() {
				if ((updateFeaturesThread.getStatus()==UpdateFeaturesThread.NONE) && updateFeaturesThread.isAlive()) {
					// This is a rare case.
					// After the thread is started, the status is soon changed from NONE to RUNNING generally.
					// However the thread works slowly on device, the thread may be not started really after 1 second waiting.
					// keep waiting.
					Display.getCurrent().timerExec(400, this);
					return;
				}

				if (updateFeaturesThread.getStatus()==UpdateFeaturesThread.RUNNING) {
					// update is still progressing, refresh the UI.
					updatingItem.setText(updateFeaturesThread.getUpdatingItem());
					pane.layout();// force to re-update the screen
					updateProgress.setSelection((int) (updateProgress.getMaximum() * (updateFeatureMonitor.getDoneWorksCount()/(totalSearchedFeaturesCount+1))) );
//					System.out.println("Progress value now is " + updateProgress.getSelection() + " / " + updateProgress.getMaximum());//TODO - Julian: remove debug info later.
					Display.getCurrent().timerExec(400, this);
				}
				else {
//Remove the following code when duplicated-timerExec() bug is fixed.
					if (isExisted)
						return;
					else
						isExisted = true;
//End.
					if (updateFeaturesThread.getStatus() == UpdateFeaturesThread.SUCCESS) {
						// update is finished completely and successfully.
						updatingItem.setText(NLS.bind(
								UIMessages.FeatureUpdated,
								new Object[] { new Integer(updateFeaturesThread.getTotalUpdatedFeaturesCount()).toString() }
						));
						pane.layout();// force to re-update the screen
						updateProgress.setSelection(updateProgress.getMaximum());
						view.enableRestartCommand = true;


						// confirm with users for restarting the system.
						if (NormalView.RUNTIME_TITAN == view.runtimeType)
							buttonSet = SWT.OK;
						else
							buttonSet = SWT.YES | SWT.NO;
						MessageBox mb1 = new MessageBox(screen.getShell(), SWT.ICON_INFORMATION | buttonSet);

						if(updateFeaturesThread.isJvmInUpdateList()) {
							// Some of JVM library have been updated to new version,
							// and the command and parameters to launch the JVM may also be changed.
							// It's needed for the user to manually close the runtime and restart it,
							// because current auto-restart function will launch the JVM with the original command and parameters.
							mb1.setText(UIMessages.ConfirmRestart);
							mb1.setMessage(NLS.bind(UIMessages.NeedRestartManually, view.RuntimeName));
							if (mb1.open() == SWT.YES) {
								if (NormalView.RUNTIME_TITAN != view.runtimeType) {
									// Sprint Titan doesn't support restart yet.
									System.getProperties().put("eworkbench.returnCode", new Integer(PlatformUI.RETURN_OK)); //$NON-NLS-1$
									PlatformUI.getWorkbench().close();
									return;
								}
							}							
						}
						else {
							mb1.setText(UIMessages.ConfirmRestart);
							mb1.setMessage(NLS.bind(UIMessages.RecommendToRestart, view.RuntimeName));
							if (mb1.open() == SWT.YES) {
								if (NormalView.RUNTIME_TITAN != view.runtimeType) {
									// Sprint Titan doesn't support restart yet.
									System.getProperties().put("eworkbench.returnCode", new Integer(PlatformUI.RETURN_RESTART)); //$NON-NLS-1$
									PlatformUI.getWorkbench().close();
									return;
								}
							}
						}
					}
					else if (updateFeaturesThread.getStatus() == UpdateFeaturesThread.NO_UPDATE) {
						// no update available, back to root screen.
						MessageBox mb2 = new MessageBox(screen.getShell(), SWT.ICON_INFORMATION | SWT.OK);
						mb2.setText(UIMessages.UpdateStatus);
						mb2.setMessage(UIMessages.NoUpdateAvailable);
						mb2.open();
					}
					else {
						// installation is not finished, but was terminated, also exit it.
						MessageBox mb3 = new MessageBox(screen.getShell(), SWT.ICON_ERROR | SWT.OK);
						mb3.setText(UIMessages.UpdateStatus);
						if (isSingleThread)
							mb3.setMessage(UIMessages.UpdateNotCompleted);
						else
							mb3.setMessage(UIMessages.UpdateNotAllCompleted);
						mb3.open();
					}
//Remove the following code when duplicated-timerExec() bug is fixed.
					isExisted = false;
//End.
					if (NormalView.WM_SMARTPHONE == view.wmType)
						cancelCommand.setEnabled(false);
					cancelButton.setEnabled(false);
					updateFeaturesThread =  null;
					setVisible(false);
					view.showScreen(view.ROOT_SCREEN, view.REFRESH_IT);
				}
			}
		});
	}

	protected void setDefaultFocus() {
		if (view.inputType == NormalView.INPUT_SHOW_SOFTKEY_MODE)
			pane.setFocus();
		else
			cancelButton.setFocus();
	}

	public void widgetSelected(SelectionEvent e) {
		if (	(e.widget == cancelButton.widget)
			|| 	(e.widget == cancelCommand) ){
			if ((updateFeaturesThread!=null)
					&& (updateFeaturesThread.getStatus()==UpdateFeaturesThread.RUNNING)) {
				// in installing, just terminate the task.
				updateFeatureMonitor.setCanceled(true);
			}
		}
	}

	/**
	 * @return the installation configuration affected by the command
	 */
	public final static IInstallConfiguration getConfiguration() {
		IInstallConfiguration config = null;
		try {
			ILocalSite localSite = SiteManager.getLocalSite();
			config = localSite.getCurrentConfiguration();
		} catch (CoreException ce) {
			//TODO - Julian: error logging.
			ce.printStackTrace();
		}
		return config;
	}


	private class UpdateFeaturesThread extends Thread {

		public static final int NONE		= 0;
		public static final int RUNNING		= 1;
		public static final int SUCCESS		= 2;	// update thread is finished w/o any problem.
		public static final int NO_UPDATE	= 3;	// update thread is finished and did nothing.
		public static final int ENDED		= 4;	// update thread is finished w/ user's canceling, or w/ some error.

		private UpdateSearchRequest request;
		private UpdateMonitor monitor;
		private UpdateSearchResultCollector collector;
		private Vector allFeatures;
		private int currentStatus;
		private int totalSearchedFeaturesCount = 0;
		private int totalUpdatedFeaturesCount = 0;
		private boolean isJvmInList = false;

		public UpdateFeaturesThread(String name, UpdateSearchRequest searchRequest, Vector allInstalledFeatures, UpdateMonitor updateMonitor) {
			super(name);

			this.request = searchRequest;
			this.monitor = updateMonitor;
			this.currentStatus = NONE;
			this.collector = new UpdateSearchResultCollector();
			this.totalSearchedFeaturesCount = searchRequest.getCategory().getQueries().length;
			this.allFeatures = allInstalledFeatures;
		}

		public void run() {
			currentStatus = RUNNING;
			updateFeatureMonitor.beginTask(UIMessages.StartUpdating, (totalSearchedFeaturesCount + 1));
			updateFeatureMonitor.setTaskName(UIMessages.SearchingUpdates);
			isJvmInList = false;

			try {
				request.performSearch(collector, new SubProgressMonitor(monitor, 1));
				if (updateFeatureMonitor.isCanceled()) {
					currentStatus = ENDED;
					return;
				}
			}
			catch (CoreException ce) {
				ce.printStackTrace();
//				System.out.println("### total queries  = " + searchRequest.getCategory().getQueries().length);
//				System.out.println("### got successful = " + collector.getOperations().length);
//				System.out.println("### got failed     = " + ce.getStatus().getChildren().length);

				if (searchRequest.getCategory().getQueries().length
						== ce.getStatus().getChildren().length) {
					// all queries are failed.
					currentStatus = ENDED;
					return;
				}
				// else, we got at least one successful querry, so go on.
			}

			IInstallFeatureOperation[] operations = collector.getOperations();
			if (operations == null || operations.length == 0) {
				currentStatus = NO_UPDATE;
				monitor.done();
				return;
			}
			totalUpdatedFeaturesCount = operations.length;

			// check for duplication conflicts
			ArrayList conflicts = DuplicateConflictsValidator.computeDuplicateConflicts(
					operations,
					UpdatingScreen.getConfiguration());
			if (conflicts != null) {
				//TODO - Julian: error logging.
				currentStatus = ENDED;
				monitor.done();
				return;
			}

			// check if JVM library is inside the update list.
			for (int i=0; i< operations.length; i++) {
				if (operations[i].getFeature().getVersionedIdentifier().getIdentifier().startsWith(ManageScreen.JVM_LIB_FEATURE_NAME)) {
					isJvmInList = true;
					break;
				}
			}

			// going to execute install batch.
			IBatchOperation installOperation =
				OperationsManager.getOperationFactory()
								 .createBatchInstallOperation(operations);
			try {
				boolean isSuccessful = false;
				if (!monitor.isCanceled()) {
					isSuccessful = installOperation.execute(new SubProgressMonitor(monitor, monitor.getTotalWorksCount()-1), null);
				}

				if (monitor.isCanceled() || !isSuccessful) {
					// it was canceled while executing update, or got some error.
					removeOldVersion(operations, true);
					currentStatus = ENDED;
					return;
				}
				else
					removeOldVersion(operations, false);

				// Finish successfully.
				if (UpdateCore.DEBUG) {
					System.out.println("Updating is finished successfully."); //$NON-NLS-1$
				}
				currentStatus = SUCCESS;
			}
			catch (InvocationTargetException ite) {
				//TODO - Julian: error logging.
				currentStatus = ENDED;
				ite.printStackTrace();
			}
			catch (CoreException ce) {
				//TODO - Julian: error logging.
				currentStatus = ENDED;
				ce.printStackTrace();
			}
			finally {
				monitor.done();
			}
		}

		/*
		 * 
		 */
		public String getUpdatingItem() {
			return monitor.getTaskName();
		}

		/*
		 * 
		 */
		public int getStatus() {
			return currentStatus;
		}

		/*
		 * 
		 */
		public boolean isJvmInUpdateList() {
			return isJvmInList;
		}

		/*
		 * 
		 */
		public int getTotalUpdatedFeaturesCount() {
			return totalUpdatedFeaturesCount;
		}

		/*
		 * 
		 */
		private void removeOldVersion(IInstallFeatureOperation[] operations, boolean needCheck) {
			try {
				boolean isNewVersionInstalled = false;
				OperationFactory operationFactory = (OperationFactory)OperationsManager.getOperationFactory();
				IUnconfigureAndUninstallFeatureOperation unconfigureAndUninstallOperation = null;
				for (int i=0; i<operations.length; i++) {
					try	{
						if (needCheck) {
							// Do check here, only remove old version if new one has been installed successfully.
							isNewVersionInstalled = false;
							for (int j=0; j< allFeatures.size(); j++) {
								if (operations[i].getFeature().getVersionedIdentifier().getIdentifier().equals(((IFeatureReference)allFeatures.get(j)).getVersionedIdentifier().getIdentifier())) {
									isNewVersionInstalled = true;
									break;
								}
							}
							if (!isNewVersionInstalled) {
								// new version was not found/installed successfully.(maybe the process was terminated by user)
								// don't remove the old version for this feature.
								return;
							}
						}
						unconfigureAndUninstallOperation =
							operationFactory.createUnconfigureAndUninstallFeatureOperation(operations[i].getOldFeature().getSite().getCurrentConfiguredSite(),//configuredSite,
																						   operations[i].getOldFeature());
						unconfigureAndUninstallOperation.execute(null, null);
					} catch (CoreException ce) {
						ce.printStackTrace();
					} catch (InvocationTargetException ite) {
						ite.printStackTrace();
					}
				}
				SiteManager.getLocalSite().save();
			}
			catch (CoreException ce) {
				ce.printStackTrace();
				return;
			}
		}
	}

	class UpdateSearchResultCollector implements IUpdateSearchResultCollector {
		private ArrayList operations = new ArrayList();

		public void accept(IFeature feature) {
	
			if (UpdateCore.DEBUG) {
				// TODO - Julian: remove debug info later.
				System.out.println("[ eUpdate debug ] searched and got this feature:"); //$NON-NLS-1$
				System.out.println("[ eUpdate debug ] \tLabel   is " + feature.getLabel()); //$NON-NLS-1$
				System.out.println("[ eUpdate debug ] \tID      is " + feature.getVersionedIdentifier().getIdentifier()); //$NON-NLS-1$
				System.out.println("[ eUpdate debug ] \tVersion is " + feature.getVersionedIdentifier().getVersion()); //$NON-NLS-1$
				System.out.println("[ eUpdate debug ] \tURL     is " + feature.getURL().toString()); //$NON-NLS-1$
			}
			
			IInstallFeatureOperation op =
				OperationsManager.getOperationFactory().createInstallOperation(
					null,
					feature,
					null,
					null,
					null);

			IConfiguredSite site =
				UpdateUtils.getDefaultTargetSite(UpdatingScreen.getConfiguration(), op);
			if (site == null)
				site = UpdateUtils.getAffinitySite(UpdatingScreen.getConfiguration(), feature);
			if (site == null)
//				site = view.localSite.getCurrentConfiguredSite();
				site = view.targetSite;

			op.setTargetSite(site);
			operations.add(op);
		}

		public IInstallFeatureOperation[] getOperations() {
			IInstallFeatureOperation[] opsArray =
				new IInstallFeatureOperation[operations.size()];
			operations.toArray(opsArray);
			return opsArray;
		}
	}
}
