/*******************************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.update.util;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.update.core.IFeature;
import org.eclipse.update.core.INonPluginEntry;
import org.eclipse.update.core.IPluginEntry;
import org.eclipse.update.core.ISiteFeatureReference;
import org.eclipse.update.core.IURLEntry;
import org.eclipse.update.core.VersionedIdentifier;

public class FeatureWrap implements INode {

	public final static int UNKNOWN		= 0;
	public final static int RESOLVED	= 1;
	public final static int NO_RESOLVED	= 2;

	public final static int FEATURE_UNKNOWN		= 0x10;	//default or unknown status.
	public final static int FEATURE_BRAND_NEW	= 0x11;	//this feature is a brand new feature.
	public final static int FEATURE_DUPLICATE	= 0x12;	//there has been the same ID of installed feature with equal or greater version. 
	public final static int FEATURE_UPDATES		= 0x13; //there has been the same ID of installed feature, but this has greater version.

	private VersionedIdentifier vid = null;
	private String showedName = ""; //$NON-NLS-1$
	private ISiteFeatureReference featureReference = null;
	private boolean selected = false;
	private IFeature feature = null;//Julian
	private String version = "?.?.?"; //$NON-NLS-1$

	private Vector uiObjects = new Vector();
	private boolean previousSelected = false;
	private int resolvedStatus = UNKNOWN;
	private int featureUniquenessStatus = FEATURE_UNKNOWN;

	public FeatureWrap() {
	}

	public FeatureWrap(ISiteFeatureReference featureReference) {
		this.featureReference = featureReference;
		try {
			vid = featureReference.getVersionedIdentifier();
			version = vid.getVersion().toString();
			this.showedName = "{" + vid.getIdentifier() + "}"
					+ " (" + version + ")"; //$NON-NLS-1$ //$NON-NLS-2$
		}
		catch (CoreException ce) {}

	}

	public boolean isSelected() {
		return selected;
	}

	public void setSelected(boolean selected) {
		this.selected = selected;
	}

	public Vector getUiObjects() {
		return uiObjects;
	}

	public void addUiObject(Object uiObject) {
		this.uiObjects.add(uiObject);
	}

	public String getShowedName() {
		return showedName;
	}

//	public void setShowedName(String showedName) {
//		this.showedName = showedName;
//	}

	public INode findByUiObject(Object uiObject) {
		for (int i=0; i< uiObjects.size(); i++) {
			if (uiObjects.elementAt(i).equals(uiObject))
				return this;
		}
		return null;
	}

	public boolean isFeatureReady() {
		return (feature != null);
	}

	public boolean downloadFeature(IProgressMonitor monitor) {
		if (feature == null) {
			try {
				feature = featureReference.getFeature(monitor);
				showedName = feature.getLabel() + " (" + version + ")"; //$NON-NLS-1$ //$NON-NLS-2$
				return true;
			}
			catch (CoreException ce) {
				feature = null;
			}
		}
		return false;
	}

	/**
	 * Returns the total size in KiloBytes of this feature and its included
	 * feature(s).  The size is the sum of the download size and install size.
	 * 
	 * @return the total size of the feature and the included feature(s)
	 */
	public long getFeatureIncludeSize() {
		if (feature == null)
			downloadFeature(null);

		if (feature != null) {
			long totalSize = feature.getDownloadSize() + feature.getInstallSize();
			if (totalSize > 0) {
				// when the size value cannot be determined, it returns -1(ContentEntryModel.UNKNOWN_SIZE).
				return totalSize;
			}
		}
		return 0;
	}

	/**
	 * Returns the size in KiloBytes of this feature.  Only the value for this
	 * feature self and no the included feature(s).  The size is the sum of
	 * the download size and install size.
	 * 
	 * @return the total size of the feature and the included feature(s)
	 */
	public long getFeatureSize() {
		if (feature == null)
			downloadFeature(null);

		if (feature != null) {
			Set allPluginEntries = new HashSet();
			Set allNonPluginEntries = new HashSet();

			IPluginEntry[] plugins = feature.getPluginEntries();
			allPluginEntries.addAll(Arrays.asList(plugins));
			INonPluginEntry[] nonPlugins = feature.getNonPluginEntries();
			allNonPluginEntries.addAll(Arrays.asList(nonPlugins));

			IPluginEntry[] totalPlugins =
				new IPluginEntry[allPluginEntries.size()];
			INonPluginEntry[] totalNonPlugins =
				new INonPluginEntry[allNonPluginEntries.size()];
			if (allPluginEntries.size() != 0) {
				allPluginEntries.toArray(totalPlugins);
			}
			if (allNonPluginEntries.size() != 0) {
				allNonPluginEntries.toArray(totalNonPlugins);
			}

			long featureSize = 0;
			try {
				featureSize
					= feature.getFeatureContentProvider().getDownloadSizeFor(totalPlugins, totalNonPlugins)
					+ feature.getFeatureContentProvider().getInstallSizeFor(totalPlugins, totalNonPlugins);
			}
			catch (CoreException ce) {
				ce.printStackTrace();
			}
			if (featureSize > 0) {
				// when the size value cannot be determined, it returns -1(ContentEntryModel.UNKNOWN_SIZE).
				return featureSize;
			}
		}
		return 0;
	}

	public String getDescription() {
		if (feature == null)
			downloadFeature(null);

		if (feature != null) {
			IURLEntry description = feature.getDescription();
			if (description != null)
				return description.getAnnotation();
		}
		return ""; //$NON-NLS-1$
	}

	public String getLicense() {
		if (feature == null)
			downloadFeature(null);

		if (feature != null) {
			IURLEntry license = feature.getLicense();
			if (license != null) {
				return license.getAnnotation();
			}
		}
		return ""; //$NON-NLS-1$
	}

	public IFeature getFeature() {
		if (feature == null)
			downloadFeature(null);

		return feature;
	}

	public void releaseFeature() {
		feature = null;
	}

	public boolean isPreviousSelected() {
		return previousSelected;
	}

	public void setPreviousSelected(boolean previousSelected) {
		this.previousSelected = previousSelected;
	}

	public int getResolvedStatus() {
		return resolvedStatus;
	}

	public void setResolvedStatus(int resolvedStatus) {
		this.resolvedStatus = resolvedStatus;
	}

	public VersionedIdentifier getVID() {
		return vid;
	}

//	public void setVID(VersionedIdentifier vid) {
//		this.vid = vid;
//	}

	public int getFeatureUniquenessStatus() {
		return featureUniquenessStatus;
	}

	public void setFeatureUniquenessStatus(int uniquenessStatus) {
		this.featureUniquenessStatus = uniquenessStatus;
	}

}
