/*******************************************************************************
 * Copyright (c) 2004, 2008 Nokia Corporation and/or its subsidiary(-ies).
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Yu You (Nokia Corp.)- initial API specification 
 *     Nokia Corporation - S60 implementation
 *     Nokia Corporation - QT implementation
 *******************************************************************************/
package org.eclipse.ercp.swt.mobile;

import org.eclipse.swt.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.internal.qt.OS;
import org.eclipse.swt.widgets.*;

import java.lang.Math;

/**
 * 
 * A CaptionedControl is used to display a label (caption) in front of a
 * control. An optional trailing text can be used after the control, for
 * example, to indicate units of measurement.
 * <p>
 * The highlighting is implementation dependent. Typically, a focused
 * CaptionedControl will highlight all its elements, i.e. the leading label, the
 * control, and optional trailing label, when it gets focus. The positioning
 * order for the captions is determined by the <code>SWT.LEFT_TO_RIGHT</code>
 * and <code>SWT.RIGHT_TO_LEFT</code> styles hints.
 * </p>
 * 
 * <p>
 * CaptionedControl does not support nested CaptionedControls. An exception will
 * be thrown when an instance of CaptionedControl is given as the constructor's
 * argument. To change the control contained by the CaptionedControl, a new
 * control can be created using the CaptionedControl as its parent control. The
 * old control must be disposed by the application manually. Only one control
 * can be active and visible in a CaptionedControl. Which control is displayed
 * when multiple controls have been added is implementation-dependent.
 * </p>
 * <p>
 * <em>Example code:</em>
 * </p>
 * <p>
 * <code><pre>
 * CaptionedControl control = new CaptionedControl(shell, SWT.NONE);
 * ConstraintedText currency = new ConstraintedText(control, SWT.NONE,
 *      ConstraintedText.NUMERIC);
 * control.setText(&quot;Velocity&quot;);
 * control.getTrailingText(&quot;cm&quot;);
 * </pre></code>
 * </p>
 * 
 * <p>
 * <dl>
 * <dt><b>Styles: </b></dt>
 * <dd>SWT.LEFT_TO_RIGHT (default)</dd>
 * <dd>SWT.RIGHT_TO_LEFT</dd>
 * <dt><b>Events: </b></dt>
 * <dd>(none)</dd>
 * </dl>
 * <p>
 * Note: Even this class is a subclass of Composite, it does not make sense to
 * set a layout on it.
 * </p>
 * <p>
 * IMPORTANT: This class is not intended to be subclassed.
 * </p>
 * 
 */
public final class CaptionedControl extends Composite {

class CaptionedControlLayout extends Layout {

/**
 * @see Layout.computeSize
 */
protected Point computeSize(Composite composite, int hint, int hint2, boolean flushCache) {
    Point res = new Point(0, 0);
    CaptionedControlLayoutData data = getData();
    if ((getStyle() & SWT.HORIZONTAL) != 0) {
        if (hint == SWT.DEFAULT) {
            res.x = data.row1.x;
        }
        else {
            res.x = data.row1.x - data.child.x + hint;
        }
        if (hint2 == SWT.DEFAULT) {
            res.y = data.row1.y;
        }
        else {
            res.y = hint2;
        }
    }
    else {
        if (hint == SWT.DEFAULT) {
            res.x = Math.max(data.row1.x, data.row2.x);
        }
        else {
            res.x = data.trail.x + Math.max( data.row1.x, hint);
        }
        if (hint2 == SWT.DEFAULT) {
            res.y = data.row1.y + data.row2.y;
        }
        else {
            res.y = data.row1.y + Math.max(hint2, data.trail.y);
        }
    }
    
    // Since the margins are big enough, adding the border is pointless.
    res.x += 2 * margin;
    res.y += 2 * margin;

    return res;
}

/**
 * @see Layout.layout
 */
protected void layout(Composite composite, boolean flushCache) {
    CaptionedControlLayoutData data = getData();

    Point size = composite.getSize();    
    if (size.x <= 0 || size.y <= 0) {
        return;
    }
    
    int x = margin;
    int y = margin;
    int w = size.x - 2 * margin;
    int h = 0;
    
    if ((getStyle() & SWT.HORIZONTAL) != 0) {
        h = size.y - 2 * margin;
    }
    else {
        h = data.row1.y;
    }
    
    if (h < 0) h = 0;
    
    imageLabel.setBounds(x, y, data.img.x, h);
    x += data.img.x;
    
    titleLabel.setBounds(x, y, data.title.x, h);
    x += data.title.x;

    if ((getStyle() & SWT.VERTICAL) != 0) {
        y += h;
        h = size.y - 2 * margin - h;
        if (h < 0) h = 0;
        x = margin;
    }
    
    int childW = 0;
    if (child != null) {
        if ((getStyle() & SWT.HORIZONTAL) != 0) {
            childW = w - data.img.x - data.title.x - data.trail.x;
        }
        else {
            childW = w - data.trail.x;
        }
        if (childW < 0) {
            childW = 0;
        }

        child.setBounds(x, y, childW, h);
        x += childW;
    }
    
    trailLabel.setBounds(x, y, data.trail.x, h);
}

protected CaptionedControlLayoutData getData() {
    CaptionedControlLayoutData res = new CaptionedControlLayoutData();

    int extra = spacing * 2;
    
    if (imageLabel.getImage() != null) {
        res.img = imageLabel.computeSize(SWT.DEFAULT, SWT.DEFAULT);
    }
    String caption = titleLabel.getText();
    if (caption != null && caption.length() > 0) {
        res.title = titleLabel.computeSize(SWT.DEFAULT, SWT.DEFAULT);
    }
    String trail = trailLabel.getText();
    if (trail != null && trail.length() > 0) {
        res.trail = trailLabel.computeSize(SWT.DEFAULT, SWT.DEFAULT);
    }
    
    if (child != null) {
        res.child = child.computeSize(SWT.DEFAULT, SWT.DEFAULT);
    }
    
    if (res.img.x > 0 && res.img.y > 0) {
        res.img.x += extra;
        res.img.y += extra;
    }
    if (res.title.x > 0 && res.title.y > 0) {
        res.title.x += extra;
        res.title.y += extra;
    }
    if (res.trail.x > 0 && res.trail.y > 0) {
        res.trail.x += extra;
        res.trail.y += extra;
    }
    
    if ((getStyle() & SWT.HORIZONTAL) != 0) {
        res.row1.x = res.img.x + res.title.x + res.child.x + res.trail.x;
        res.row1.y = Math.max(res.row1.y, res.img.y);
        res.row1.y = Math.max(res.row1.y, res.title.y);
        res.row1.y = Math.max(res.row1.y, res.trail.y);
        res.row1.y = Math.max(res.row1.y, res.child.y);
    }
    else {
        res.row1.x = res.img.x + res.title.x;
        res.row1.y = Math.max(res.row1.y, res.img.y);
        res.row1.y = Math.max(res.row1.y, res.title.y);
        res.row2.x = res.child.x + res.trail.x;
        res.row2.y = Math.max(res.row2.y, res.trail.y);
        res.row2.y = Math.max(res.row2.y, res.child.y);
    }
    
    return res;
}
}

class CaptionedControlLayoutData {
    Point img = new Point(0, 0);
    Point title = new Point(0, 0);
    Point trail = new Point(0, 0);
    Point child = new Point(0, 0);
    Point row1 = new Point(0, 0);
    Point row2 = new Point(0, 0);
}

private Label imageLabel;
private Label titleLabel;
private Label trailLabel;
private Control child;
private final int margin = 5;
private final int spacing = 1;
private CaptionedControlLayout layout;

/**
 * Constructs a new instance of this class given its parent and a style
 * value describing its behavior and appearance.
 * <p>
 * The style value is either one of the style constants defined in class
 * <code>SWT</code> which is applicable to instances of this class, or must
 * be built by <em>bitwise OR</em> 'ing together (that is, using the
 * <code>int</code> "|" operator) two or more of those <code>SWT</code>
 * style constants. The class description lists the style constants that are
 * applicable to the class. Style bits are also inherited from superclasses.
 * </p>
 * 
 * @param parent
 *            a widget which will be the parent of the new instance (cannot
 *            be null)
 * @param style
 *            the style of widget to construct
 * 
 * @exception IllegalArgumentException
 *                <ul>
 *                <li>ERROR_NULL_ARGUMENT - if the parent is null</li>
 *                </ul>
 * @exception SWTException
 *                <ul>
 *                <li>ERROR_THREAD_INVALID_ACCESS - if not called from the
 *                thread that created the parent</li>
 *                <li>ERROR_INVALID_SUBCLASS - if this class is not an
 *                allowed subclass</li>
 *                </ul>
 * 
 * @see SWT#LEFT_TO_RIGHT
 * @see SWT#RIGHT_TO_LEFT
 */
public CaptionedControl (Composite parent, int style) {
    super(parent, checkBits(style, SWT.HORIZONTAL, SWT.VERTICAL, 0, 0, 0, 0));
    if (parent instanceof CaptionedControl) {
        SWT.error(SWT.ERROR_INVALID_PARENT);
    }
    scrollAreaHandle = 0;
    imageLabel = new Label(this, SWT.CENTER);
    titleLabel = new Label(this, SWT.CENTER);
    trailLabel = new Label(this, SWT.CENTER);
    layout = new CaptionedControlLayout();
    super.setLayout(layout);
    layout();
}

/**
 * @see Composite.addControl
 */
protected void addControl (Control control) {
    child = control;
}

/**
 * @see Scrollable.computeSize
 */
public Point computeSize(int wHint, int hHint, boolean changed) {
    checkWidget();
    if (changed) {
        OS.QWidget_updateGeometry(internal_topHandle());
    }
    if (layout != null) {
        return layout.computeSize(this, wHint, hHint, changed);
    }
    else {
        return new Point(DEFAULT_WIDTH, DEFAULT_HEIGHT);
    }
}

/**
 * @see Scrollable.computeTrim
 */
public Rectangle computeTrim(int x, int y, int width, int height) {
    checkWidget();
    
    Rectangle res = new Rectangle(0, 0, 0, 0);
    
    if (layout != null) {
        CaptionedControlLayoutData data = layout.getData();
        if ((getStyle() & SWT.HORIZONTAL) != 0) { 
            res.width = data.img.x + data.title.x + data.trail.x + width;
            res.height = height;
        }
        else {
            res.width = data.trail.x + Math.max(data.row1.x, width);
            res.height = data.row1.y + height;
        }
    }
    else {
        res.width = width;
        res.height = height;
    }
    res.x = x; 
    res.y = y;
    res.width += 2 * margin;
    res.height += 2 * margin;
    
    return res;
}

/**
 * @see Composite.createHandle
 */
protected void createHandle (int index) {
    handle = OS.QCaptionedWidget_new();
    int policy = OS.QWidget_focusPolicy(handle) & ~OS.QT_FOCUSPOLICY_CLICKFOCUS;
    policy |= OS.QT_FOCUSPOLICY_TABFOCUS;
    OS.QWidget_setFocusPolicy(handle, policy);
    state |= HANDLE;
    state &= ~CANVAS;
}

protected void doRedraw() {
    OS.QWidget_update(internal_topHandle());
}

protected void doRedraw(int x, int y, int width, int height) {
    OS.QWidget_update(internal_topHandle(), x, y, width, height);
}

/**
 * @see Composite.getChildren
 */
public Control[] getChildren() {
    checkWidget();
    Control[] children = super.getChildren();
    Control[] res = new Control[children.length - 3];
    for (int i = 0, j = 0; i < children.length; i++) {
        if (children[i] != imageLabel 
            && children[i] != titleLabel
            && children[i] != trailLabel ) {
            res[j] = children[i];
            j++;
        }
    }
    return res;
}

/**
 * @see Scrollable.getClientArea
 */
public Rectangle getClientArea() {
    checkWidget();
    
    Rectangle res = new Rectangle(0, 0, 0, 0);
    
    Point size = getSize();
    size.x -= 2 * margin;
    size.y -= 2 * margin;    
    
    if (layout != null) {
        CaptionedControlLayoutData data = layout.getData();
        if ((getStyle() & SWT.HORIZONTAL) != 0) {
            res.width = size.x - data.img.x - data.title.x - data.trail.x;
            res.height = size.y;
        }
        else {
            res.width = size.x - data.trail.x; 
            res.height = size.y - data.row1.y;
        }
    }
    else {
        res.width = size.x;
        res.height = size.y;
    }
    
    res.x = 0;
    res.y = 0;
    
    return res;
}

/**
 * Returns the CaptionedControl's icon image, or null if it has never been
 * set.
 * 
 * @return the icon image or null.
 * 
 * @exception SWTException
 *                <ul>
 *                <li>ERROR_WIDGET_DISPOSED - if the receiver has been
 *                disposed</li>
 *                <li>ERROR_THREAD_INVALID_ACCESS - if not called from the
 *                thread that created the receiver</li>
 *                </ul>
 * @see #setImage(Image)
 */
public Image getImage () {
    checkWidget();
    return imageLabel.getImage();
}

/**
 * Gets the caption text, which will be an empty string if it has never been
 * set.
 * 
 * @return The label text.
 * @exception SWTException
 *                <ul>
 *                <li>ERROR_WIDGET_DISPOSED - if the receiver has been
 *                disposed</li>
 *                <li>ERROR_THREAD_INVALID_ACCESS - if not called from the
 *                thread that created the parent</li>
 *                </ul>
 * 
 * @see #setText(java.lang.String)
 */
public String getText () {
    checkWidget();
    return titleLabel.getText();
}

/**
 * Gets the trailing text, which will be an empty string if it has never
 * been set.
 * 
 * @return The trailing text.
 * 
 * @exception SWTException
 *                <ul>
 *                <li>ERROR_WIDGET_DISPOSED - if the receiver has been
 *                disposed</li>
 *                <li>ERROR_THREAD_INVALID_ACCESS - if not called from the
 *                thread that created the parent</li>
 *                </ul>
 * 
 * @see #setTrailingText(java.lang.String)
 */
public String getTrailingText () {
    checkWidget();
    return trailLabel.getText();
}

/**
 * @see Scrollable.internal_frameHandle
 */
protected int internal_frameHandle() {
    return handle;
}

/**
 * @see Composite.internal_topHandle
 */
public int internal_topHandle () {
    return handle;
}

public void pack(boolean changed) {
		layout(changed);
		super.pack(changed);
}

/**
 * @see Composite.removeControl
 */
protected void removeControl (Control control) {
    if (control != null && control == child) {
        child = null;
    }
}

/**
 * Sets the image as an icon to the CaptionedControl. The icon can co-exist
 * with caption text. The icon position is platform-dependent.
 * 
 * <p>
 * The parameter can be null indicating that no image should be displayed.
 * The implementation will adjust the image size to make it best fit the
 * CaptionedControl.
 * </p>
 * 
 * @param image
 *            the image to display on the receiver
 * 
 * @exception IllegalArgumentException
 *                <ul>
 *                <li>ERROR_NULL_ARGUMENT - if the image is null</li>
 *                <li>ERROR_INVALID_ARGUMENT - if the image has been
 *                disposed</li>
 *                </ul>
 * @exception SWTException
 *                <ul>
 *                <li>ERROR_WIDGET_DISPOSED - if the receiver has been
 *                disposed</li>
 *                <li>ERROR_THREAD_INVALID_ACCESS - if not called from the
 *                thread that created the receiver</li>
 *                </ul>
 * 
 */
public void setImage (Image image) {
    checkWidget();
    if (image == null) {
        SWT.error(SWT.ERROR_NULL_ARGUMENT);
    }
    if (image != null && image.isDisposed()) {
        SWT.error(SWT.ERROR_INVALID_ARGUMENT);
    }
    imageLabel.setImage(image);
    layout();
}

/**
 * @see Composite.setLayout
 */
public void setLayout (Layout layout) {
    // Even though this class is a subclass of Composite, setting a layout
    // on it is not supported.
    checkWidget();
}

/**
 * Sets the caption label
 * 
 * @param string
 *            the new caption label
 * 
 * @throws java.lang.IllegalArgumentException
 *             <code>ERROR_NULL_ARGUMENT</code> if the text is null
 * @exception SWTException
 *                <ul>
 *                <li>ERROR_WIDGET_DISPOSED - if the receiver has been
 *                disposed</li>
 *                <li>ERROR_THREAD_INVALID_ACCESS - if not called from the
 *                thread that created the parent</li>
 *                </ul>
 * @see #getText
 */
public void setText (java.lang.String string) {
    checkWidget();
    if (string == null) {
        SWT.error(SWT.ERROR_NULL_ARGUMENT);
    }
    titleLabel.setText(string);
    layout();
}

/**
 * Sets the trailing label
 * 
 * @param string
 *            the new trailing label
 * 
 * @exception IllegalArgumentException
 *                <ul>
 *                <li>ERROR_NULL_ARGUMENT - if the string is null</li>
 *                </ul>
 * @exception SWTException
 *                <ul>
 *                <li>ERROR_WIDGET_DISPOSED - if the receiver has been
 *                disposed</li>
 *                <li>ERROR_THREAD_INVALID_ACCESS - if not called from the
 *                thread that created the parent</li>
 *                </ul>
 * 
 * @see #getTrailingText
 */
public void setTrailingText (java.lang.String string) {
    checkWidget();
    if (string == null) {
        SWT.error(SWT.ERROR_NULL_ARGUMENT);
    }
    trailLabel.setText(string);
    layout();
}

}
