/*******************************************************************************
 * Copyright (c) 2008 Nokia Corporation and/or its subsidiary(-ies).
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Nokia Corporation - initial implementation 
 *******************************************************************************/
package org.eclipse.swt.internal.qt;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Scrollable;
import org.eclipse.swt.widgets.TypedListener;

public abstract class ListBase extends Scrollable {

    private static final int LISTBASE_SHOW_CHECKBOX = 0x80;

    protected static int LISTTYPE_STRINGLIST = 1;

    protected static int LISTTYPE_SORTEDLIST = 2;

    protected static int LISTTYPE_LISTVIEW = 4;

    protected static int LISTTYPE_LISTBOX = 8;

    protected int layoutStyle;

    protected int dataModelHandle;

    protected int selectionModelHandle;

    protected int listType;

    boolean undoSelectionChangePhase = false;

    public ListBase(Composite parent, int style) {
        super(parent, style);
        initList();
    }

    public ListBase(Composite parent, int style, int layoutStyle) {
        super(parent, style);
        this.layoutStyle = layoutStyle;
        // show check box only for S60
        if (OS.windowServer == OS.WS_SYMBIAN_S60) {
            this.layoutStyle |= LISTBASE_SHOW_CHECKBOX;
        }
        initList();
    }

    void initList() {
        /**
         * QListView owns created data model
         */
        dataModelHandle = createDataModel();
        OS.QAbstractItemView_setModel(internal_topHandle(), dataModelHandle);

        OS.QAbstractItemView_setSelectionBehavior(internal_topHandle(),
                OS.QT_ABSTRACTITEMVIEW_SELECTIONBEHAVIOR_ROWS);

        int selectionMode = OS.QT_ABSTRACTITEMVIEW_SELECTIONMODE_SINGLE;
        if ((style & SWT.MULTI) != 0) {
            selectionMode = OS.QT_ABSTRACTITEMVIEW_SELECTIONMODE_MULTI;
        }
        OS.QAbstractItemView_setSelectionMode(internal_topHandle(),
                selectionMode);

        selectionModelHandle = OS
                .QAbstractItemView_selectionModel(internal_topHandle());

        hookEvents();
    }

    protected abstract int createDataModel();

    protected void createHandle(int index) {
        scrollAreaHandle = OS.QListView_new();
        handle = OS.QAbstractScrollArea_viewPort(scrollAreaHandle);
        state |= HANDLE;
    }

    protected void createWidget(int index) {
        super.createWidget(index);
    }

    protected void hookEvents() {
        if (selectionModelHandle != 0) {
            int signalProxy = OS.SignalHandler_new(internal_topHandle(),
                    display, OS.QSIGNAL_SELECTIONCHANGED);
            OS.QObject_connectOrThrow(
                    selectionModelHandle,
                    "selectionChanged( const QItemSelection&, const QItemSelection& )",
                    signalProxy,
                    "widgetSignal( const QItemSelection&, const QItemSelection& )",
                    OS.QT_AUTOCONNECTION);
        
            int itemActivatedSignalProxy = OS.SignalHandler_new(internal_topHandle(),
                    display, OS.QSIGNAL_ABSTRACTITEMVIEW_ACTIVATED);
            OS.QObject_connectOrThrow(internal_topHandle(),
                    "activated(const QModelIndex& )", itemActivatedSignalProxy,
                    "widgetSignal(const QModelIndex&)", OS.QT_AUTOCONNECTION);


        }
    }

    protected static int checkStyle(int style) {
        return checkBits(style, SWT.SINGLE, SWT.MULTI, 0, 0, 0, 0);
    }

    protected void listBase_addSelectionListener(SelectionListener listener) {
        checkWidget();
        if (listener == null)
            SWT.error(SWT.ERROR_NULL_ARGUMENT);
        TypedListener typedListener = new TypedListener(listener);
        addListener(SWT.Selection, typedListener);
        addListener(SWT.DefaultSelection, typedListener);
    }

    protected void listBase_deselect(int index) {
        checkWidget();
        if (index < 0 || index > listBase_getItemCount() - 1)
            return;
        internalSelect(index, OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_DESELECT,
                false, false);
    }

    protected void listBase_deselect(int start, int end) {
        checkWidget();

        int count = listBase_getItemCount();
        if (start > count - 1 || end < 0 || start > end)
            return;

        start = Math.max(0, start);
        end = Math.min(end, count - 1);

        internalSelect(start, end,
                OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_DESELECT, false, false);
    }

    protected void listBase_deselect(int[] indices) {
        checkWidget();
        if (indices == null)
            SWT.error(SWT.ERROR_NULL_ARGUMENT);
        int length = indices.length;
        if (length == 0)
            return;
        int index;
        int count = listBase_getItemCount();
        for (int i = 0; i < length; i++) {
            index = indices[i];
            if (index < 0 || index > count - 1)
                continue;
            internalSelect(indices[i],
                    OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_DESELECT, false,
                    false);
        }
    }

    protected void listBase_deselectAll() {
        checkWidget();
        OS.QItemSelectionModel_clearSelection(selectionModelHandle);
    }

    protected int listBase_getFocusIndex() {
        checkWidget();
        int indexHandle = OS
                .QItemSelectionModel_currentIndex(selectionModelHandle);
        int row = OS.QModelIndex_row(indexHandle);
        OS.QModelIndex_delete(indexHandle);
        return row;
    }

    protected String[] listBase_getItems() {
        checkWidget();
        int count = listBase_getItemCount();
        if (count < 1)
            return new String[] {};
        String[] strings = OS.ListModel_itemStrings(dataModelHandle);
        /**
         * It seems that 'String[0]' != 'String{} when String[0] is returned
         * from JNI call.
         */
        if (strings == null || strings == new String[0]) {
            strings = new String[] {};
        }
        return strings;
    }

    protected int listBase_getSelectionCount() {
        checkWidget();
        int[] indices = OS.QItemSelectionModel_selectedRows(
                selectionModelHandle, 0);
        if (indices != null) {
            return indices.length;
        } else {
            return 0;
        }
    }

    protected String[] listBase_getSelection() {
        checkWidget();
        int[] indices = listBase_getSelectionIndices();
        if (indices.length < 1)
            return new String[] {};
        String[] result = new String[indices.length];
        for (int i = 0; i < indices.length; i++) {
            result[i] = listBase_getItem(indices[i]);
        }
        return result;
    }

    protected int[] listBase_getSelectionIndices() {
        checkWidget();
        int[] indices = OS.QItemSelectionModel_selectedRows(
                selectionModelHandle, 0);
        if (indices == null /* || indices == new int[0] */) {
            indices = new int[] {};
        }
        sort(indices);
        return indices;
    }

    protected void listBase_refreshItem(int index) {
        checkWidget();
        if (index < 0 || index >= listBase_getItemCount())
            SWT.error(SWT.ERROR_INVALID_RANGE);
        int indexHandle = OS.QAbstractItemModel_index(dataModelHandle, index,
                0, 0);
        OS.QAbstractItemView_update(internal_topHandle(), indexHandle);
        OS.QModelIndex_delete(indexHandle);
    }

    protected void listBase_refreshList() {
        checkWidget();
        for (int i = 0; i < listBase_getItemCount(); i++) {
            listBase_refreshItem(i);
        }
    }

    protected void listBase_removeSelectionListener(SelectionListener listener) {
        removeListener(SWT.Selection, listener);
        removeListener(SWT.DefaultSelection, listener);
    }

    protected void listBase_select(int index) {
        checkWidget();
        if (index < 0 || index > listBase_getItemCount() - 1)
            return;
        int command = OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_SELECT;
        if ((style & SWT.SINGLE) != 0) {
            command |= OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_CLEAR;
        }
        internalSelect(index, command, false, false);
    }

    protected void listBase_select(int start, int end) {
        checkWidget();
        if (start == end) {
            listBase_select(start);
            return;
        }

        int count = listBase_getItemCount();
        if (start > count - 1 || end < 0 || start > end)
            return;

        start = Math.max(0, start);
        end = Math.min(end, count - 1);

        if ((style & SWT.MULTI) == 0)
            return;

        internalSelect(start, end,
                OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_SELECT, false, false);
    }

    protected void listBase_select(int[] indices) {
        checkWidget();
        if (indices == null)
            SWT.error(SWT.ERROR_NULL_ARGUMENT);
        int length = indices.length;
        if (length == 1) {
            listBase_select(indices[0]);
            return;
        }
        if (length == 0 || (style & SWT.MULTI) == 0)
            return;
        int index;
        int count = listBase_getItemCount();
        for (int i = 0; i < length; i++) {
            index = indices[i];
            if (index < 0 || index > count - 1)
                continue;
            internalSelect(index,
                    OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_SELECT, false,
                    false);
        }
    }

    protected void listBase_selectAll() {
        checkWidget();
        if ((style & SWT.MULTI) == 0 && listBase_getItemCount() != 1)
            return;
        internalSelect(0, listBase_getItemCount() - 1,
                OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_SELECT, false, false);
    }

    protected boolean listBase_isSelected(int index) {
        checkWidget();
        int indexHandle = OS.QAbstractItemModel_index(dataModelHandle, index,
                0, 0);
        if (indexHandle == 0)
            return false;
        boolean selected = OS.QItemSelectionModel_isSelected(
                selectionModelHandle, indexHandle);
        OS.QModelIndex_delete(indexHandle);
        return selected;
    }

    protected void listBase_setSelection(int index) {
        checkWidget();
        if (index < 0 || index > listBase_getItemCount() - 1) {
            listBase_deselectAll();
            return;
        }
        int command = OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_CLEAR
                | OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_SELECT
                | OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_CURRENT;
        internalSelect(index, command, true, true);
    }

    protected void listBase_setSelection(int start, int end) {
        checkWidget();
        if (start == end) {
            listBase_setSelection(start);
            return;
        }

        listBase_deselectAll();

        int count = listBase_getItemCount();
        if (start > count - 1 || end < 0 | start > end)
            return;

        start = Math.max(0, start);
        end = Math.min(end, count - 1);

        if ((style & SWT.MULTI) == 0)
            return;

        internalSelect(start, end,
                OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_SELECT, true, true);
    }

    protected void listBase_setSelection(int[] indices) {
        checkWidget();
        if (indices == null)
            SWT.error(SWT.ERROR_NULL_ARGUMENT);
        listBase_deselectAll();
        int length = indices.length;
        if (length == 0 || ((style & SWT.MULTI) == 0 && length != 1))
            return;

        int[] newIndices = new int[indices.length];
        System.arraycopy(indices, 0, newIndices, 0, indices.length);
        sort(newIndices);

        int index;
        int count = listBase_getItemCount();
        for (int i = 0; i < length; i++) {
            index = newIndices[i];
            if (index < 0 || index > count - 1)
                continue;
            if (i == 0) {
                internalSelect(index,
                        OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_SELECT, true,
                        true);
            } else {
                internalSelect(index,
                        OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_SELECT, false,
                        false);
            }
        }
    }

    protected void listBase_setSelection(String[] items) {
        checkWidget();
        if (items == null)
            SWT.error(SWT.ERROR_NULL_ARGUMENT);

        listBase_deselectAll();

        int length = items.length;
        if (length == 0 || ((style & SWT.MULTI) == 0 && length != 1))
            return;
        int count = listBase_getItemCount();
        if (count < 1)
            return;
        int index;
        int minIndex = count;
        for (int i = 0; i < length; i++) {
            if (items[i] == null)
                SWT.error(SWT.ERROR_INVALID_ARGUMENT);
            index = 0;
            while (index < count) {
                index = listBase_indexOf(items[i], index);
                if (index < 0)
                    break;
                if (index > minIndex) {
                    internalSelect(index,
                            OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_SELECT,
                            false, false);
                } else {
                    if (minIndex < count) {
                        internalSelect(minIndex,
                                OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_SELECT,
                                false, false);
                    }
                    minIndex = index;
                }
                index++;
            }
        }
        if (minIndex < count) {
            internalSelect(minIndex,
                    OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_SELECT, true, true);
        }
    }

    protected void listBase_showSelection() {
        checkWidget();
        int[] indexes = listBase_getSelectionIndices();
        if (indexes == null || indexes.length < 1)
            return;
        int indexHandle = OS.QAbstractItemModel_index(dataModelHandle,
                indexes[0], 0, 0);
        OS.QAbstractItemView_scrollTo(internal_topHandle(), indexHandle,
                OS.QT_ABSTRACTITEMVIEW_SCROLLHINT_ENSUREVISIBLE);
        OS.QModelIndex_delete(indexHandle);
    }

    private void internalSelect(int index, int command, boolean ensureVisible,
            boolean setFocus) {
        int indexHandle = OS.QAbstractItemModel_index(dataModelHandle, index,
                0, 0);
        if (setFocus) {
            OS.QItemSelectionModel_setCurrentIndex(selectionModelHandle,
                    indexHandle, command);
        } else {
            OS.QItemSelectionModel_select(selectionModelHandle, indexHandle,
                    command);
        }
        if (ensureVisible) {
            OS.QAbstractItemView_scrollTo(internal_topHandle(), indexHandle,
                    OS.QT_ABSTRACTITEMVIEW_SCROLLHINT_ENSUREVISIBLE);
        }
        OS.QModelIndex_delete(indexHandle);
    }

    private void internalSelect(int start, int end, int command,
            boolean ensureVisible, boolean setFocus) {
        int startIndexHandle = OS.QAbstractItemModel_index(dataModelHandle,
                start, 0, 0);
        int endIndexHandle = OS.QAbstractItemModel_index(dataModelHandle, end,
                0, 0);
        OS.QItemSelectionModel_select(selectionModelHandle, startIndexHandle,
                endIndexHandle, command);
        if (ensureVisible) {
            OS.QAbstractItemView_scrollTo(internal_topHandle(),
                    startIndexHandle,
                    OS.QT_ABSTRACTITEMVIEW_SCROLLHINT_POSITIONATTOP);
        }
        if (setFocus) {
            OS.QItemSelectionModel_setCurrentIndex(selectionModelHandle,
                    startIndexHandle,
                    OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_CURRENT);
        }

        OS.QModelIndex_delete(startIndexHandle);
        OS.QModelIndex_delete(endIndexHandle);
    }

    private boolean getTraversalDoIt(int type, int key, int modifier,
            int character) {
        int currentIndex = listBase_getFocusIndex();
        boolean firstHasFoucs = (currentIndex == 0);
        boolean lastHasFoucs = (currentIndex == listBase_getItemCount() - 1);

        boolean doIt = (firstHasFoucs && (type & (SWT.TRAVERSE_ARROW_PREVIOUS | SWT.TRAVERSE_TAB_PREVIOUS)) != 0)
                || (lastHasFoucs
                        && (type & (SWT.TRAVERSE_ARROW_NEXT | SWT.TRAVERSE_TAB_NEXT)) != 0 || (type & SWT.TRAVERSE_MNEMONIC) != 0);
        return doIt;
    }

    protected void setTraversalFlags(int type, int key, int modifier,
            int character) {
        traverseDoit = getTraversalDoIt(type, key, modifier, character);
        if ((type & SWT.TRAVERSE_MNEMONIC) != 0) {
            traverseCancel = true;
        } else {
            traverseCancel = traverseDoit;
        }
    }

    protected void qt_signal_selectionChanged(int selectionHandle,
            int deSelectionHandle) {
        if (undoSelectionChangePhase == false) {
            Event event = new Event();
            sendEvent(SWT.Selection, event);
            if (event.doit == false) {
                // undo change
                undoSelectionChangePhase = true;
                OS.QItemSelectionModel_select__set(selectionModelHandle,
                        selectionHandle,
                        OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_DESELECT);
                OS.QItemSelectionModel_select__set(selectionModelHandle,
                        deSelectionHandle,
                        OS.QT_ITEMSELECTIONMODEL_SELECTIONFLAGS_SELECT);
                undoSelectionChangePhase = false;
            } else {
                // update check states
                if (OS.windowServer == OS.WS_SYMBIAN_S60) {
                    OS.ListModel_setCheckState(dataModelHandle,
                            selectionHandle, OS.QT_CHECKED);
                    OS.ListModel_setCheckState(dataModelHandle,
                            deSelectionHandle, OS.QT_UNCHECKED);
                }
            }
        }
    }

    protected void qt_signal_abstractitemview_activated(int row, int column) {
        super.qt_signal_abstractitemview_activated(row, column);
        sendEvent(SWT.DefaultSelection);
    }

    protected int listBase_getItemCount() {
        checkWidget();
        return OS.QAbstractItemModel_rowCount(dataModelHandle);
    }

    protected int listBase_getItemHeight() {
        checkWidget();
        return OS.ListModel_itemHeight(dataModelHandle);
    }

    protected void listBase_add(String string) {
        checkWidget();
        if (string == null)
            SWT.error(SWT.ERROR_NULL_ARGUMENT);
        int count = listBase_getItemCount();
        OS.ListModel_beginInsertRows(dataModelHandle, 0, count, count);
        OS.ListModel_append(dataModelHandle, string);
        OS.ListModel_endInsertRows(dataModelHandle);
    }

    protected String listBase_getItem(int index) {
        checkWidget();
        int count = listBase_getItemCount();
        if (index < 0 || index > count - 1)
            SWT.error(SWT.ERROR_INVALID_RANGE);
        return OS.ListModel_itemString(dataModelHandle, index);
    }

    protected int listBase_indexOf(String string, int start) {
        checkWidget();
        if (string == null)
            SWT.error(SWT.ERROR_NULL_ARGUMENT);
        if (start < 0 || start > listBase_getItemCount() - 1)
            return -1;
        return OS.ListModel_indexOf(dataModelHandle, string, start);
    }

    protected void listBase_remove(int index) {
        checkWidget();
        int count = listBase_getItemCount();
        if (index < 0 || index > count - 1)
            SWT.error(SWT.ERROR_INVALID_RANGE);
        int indexHandle = OS.QAbstractItemModel_index(dataModelHandle, index,
                0, 0);
        OS.ListModel_beginRemoveRows(dataModelHandle, 0, index, index);
        OS.ListModel_remove(dataModelHandle, index);
        OS.ListModel_endRemoveRows(dataModelHandle);
        OS.QModelIndex_delete(indexHandle);
    }

    protected void listBase_remove(String string) {
        checkWidget();
        if (string == null)
            SWT.error(SWT.ERROR_NULL_ARGUMENT);
        int index = listBase_indexOf(string, 0);
        if (index < 0)
            SWT.error(SWT.ERROR_INVALID_ARGUMENT);
        listBase_remove(index);
    }

    protected void listBase_remove(int start, int end) {
        checkWidget();
        if (start > end)
            return;
        int count = listBase_getItemCount();
        if (start < 0 || end > count - 1)
            SWT.error(SWT.ERROR_INVALID_RANGE);
        OS.ListModel_beginRemoveRows(dataModelHandle, 0, start, end);
        for (int index = end; index >= start; index--) {
            int indexHandle = OS.QAbstractItemModel_index(dataModelHandle,
                    index, 0, 0);
            OS.ListModel_remove(dataModelHandle, index);
            OS.QModelIndex_delete(indexHandle);
        }
        OS.ListModel_endRemoveRows(dataModelHandle);
    }

    protected void listBase_remove(int[] indices) {
        checkWidget();
        if (indices == null)
            SWT.error(SWT.ERROR_NULL_ARGUMENT);
        int removeCount = indices.length;
        if (removeCount < 1)
            return;

        int[] newIndices = new int[indices.length];
        System.arraycopy(indices, 0, newIndices, 0, indices.length);
        sort(newIndices);
        int count = listBase_getItemCount();
        int start = newIndices[0], end = newIndices[newIndices.length - 1];
        if (start < 0 || end > count - 1) {
            SWT.error(SWT.ERROR_INVALID_RANGE);
        }
        int previousIndex = -1;
        for (int i = removeCount - 1; i > -1; i--) {
            if (newIndices[i] == previousIndex)
                continue;
            listBase_remove(newIndices[i]);
            previousIndex = newIndices[i];
        }
    }

    protected void listBase_removeAll() {
        checkWidget();
        int count = listBase_getItemCount();
        if (count < 1)
            return;
        OS.ListModel_beginRemoveRows(dataModelHandle, 0, 0, count - 1);
        OS.ListModel_clearList(dataModelHandle);
        OS.ListModel_endRemoveRows(dataModelHandle);
    }

    protected void listBase_setItems(String[] items) {
        checkWidget();
        if (items == null)
            SWT.error(SWT.ERROR_NULL_ARGUMENT);

        listBase_removeAll();
        int count = items.length;
        if (count < 1)
            return;
        OS.ListModel_beginInsertRows(dataModelHandle, 0, 0, count - 1);
        for (int i = 0; i < count; i++) {
            if (items[i] == null)
                SWT.error(SWT.ERROR_INVALID_ARGUMENT);
            OS.ListModel_append(dataModelHandle, items[i]);
        }
        OS.ListModel_endInsertRows(dataModelHandle);
    }

    protected int listBase_getTopIndex() {
        // TODO: consider the situation on listview icon mode, rigt to left
        checkWidget();
        if (listBase_getItemCount() < 1)
            return 0;

        return OS.QAbstractItemView_swt_indexAt(internal_topHandle(), 1, 1);
    }

    protected void listBase_setTopIndex(int index) {
        checkWidget();
        if (index < 0 || index > listBase_getItemCount() - 1)
            return;
        int indexHandle = OS.QAbstractItemModel_index(dataModelHandle, index,
                0, 0);
        OS.QAbstractItemView_scrollTo(internal_topHandle(), indexHandle,
                OS.QT_ABSTRACTITEMVIEW_SCROLLHINT_POSITIONATTOP);
        OS.QModelIndex_delete(indexHandle);
    }

    protected void releaseHandle() {
        super.releaseHandle();
        dataModelHandle = 0;
        selectionModelHandle = 0;
    }
}
