/*******************************************************************************
 * Copyright (c) 2000, 2007 IBM Corporation and others.
 * Portion Copyright (c) 2008 Nokia Corporation and/or its subsidiary(-ies).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Nokia Corporation - Qt implementation
 *******************************************************************************/
package org.eclipse.swt.widgets;

import org.eclipse.swt.SWT;
import org.eclipse.swt.SWTException;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.internal.qt.OS;

/**
 * Instances of this class allow the user to navigate the file system and select
 * a directory.
 * <dl>
 * <dt><b>Styles:</b></dt>
 * <dd>(none)</dd>
 * <dt><b>Events:</b></dt>
 * <dd>(none)</dd>
 * </dl>
 * <p>
 * IMPORTANT: This class is intended to be subclassed <em>only</em> within the
 * SWT implementation.
 * </p>
 */
public class DirectoryDialog extends Dialog {
	String message = "", filterPath = "";
	static final char SEPARATOR = System.getProperty("file.separator")
			.charAt(0);

	String dialogID;
	static int dialogCount;

	/**
	 * Constructs a new instance of this class given only its parent.
	 * 
	 * @param parent
	 *            a shell which will be the parent of the new instance
	 * 
	 * @exception IllegalArgumentException
	 *                <ul>
	 *                <li>ERROR_NULL_ARGUMENT - if the parent is null</li>
	 *                </ul>
	 * @exception SWTException
	 *                <ul>
	 *                <li>ERROR_THREAD_INVALID_ACCESS - if not called from the
	 *                thread that created the parent</li>
	 *                <li>ERROR_INVALID_SUBCLASS - if this class is not an
	 *                allowed subclass</li>
	 *                </ul>
	 */
	public DirectoryDialog(Shell parent) {
		this(parent, SWT.NULL);
	}

	/**
	 * Constructs a new instance of this class given its parent and a style
	 * value describing its behavior and appearance.
	 * <p>
	 * The style value is either one of the style constants defined in class
	 * <code>SWT</code> which is applicable to instances of this class, or must
	 * be built by <em>bitwise OR</em>'ing together (that is, using the
	 * <code>int</code> "|" operator) two or more of those <code>SWT</code>
	 * style constants. The class description lists the style constants that are
	 * applicable to the class. Style bits are also inherited from superclasses.
	 * </p>
	 * 
	 * @param parent
	 *            a shell which will be the parent of the new instance
	 * @param style
	 *            the style of dialog to construct
	 * 
	 * @exception IllegalArgumentException
	 *                <ul>
	 *                <li>ERROR_NULL_ARGUMENT - if the parent is null</li>
	 *                </ul>
	 * @exception SWTException
	 *                <ul>
	 *                <li>ERROR_THREAD_INVALID_ACCESS - if not called from the
	 *                thread that created the parent</li>
	 *                <li>ERROR_INVALID_SUBCLASS - if this class is not an
	 *                allowed subclass</li>
	 *                </ul>
	 */
	public DirectoryDialog(Shell parent, int style) {
		super(parent, checkStyle(parent, style));
		checkSubclass();
		dialogCount++;
		dialogID = toString() + String.valueOf(dialogCount);
	}

	static protected int checkStyle(Shell parent, int style) {

		style &= ~SWT.MIRRORED;
		if ((style & SWT.RIGHT_TO_LEFT) == 0) {
			if (parent != null) {
				if ((parent.style & SWT.RIGHT_TO_LEFT) != 0)
					style |= SWT.RIGHT_TO_LEFT;
			}
		}

		if ((style & SWT.LEFT_TO_RIGHT) != 0
				&& (style & SWT.RIGHT_TO_LEFT) != 0) {
			style &= ~SWT.LEFT_TO_RIGHT;
		}
		return style;
	}

	/**
	 * Returns the path which the dialog will use to filter the directories it
	 * shows.
	 * 
	 * @return the filter path
	 * 
	 * @see #setFilterPath
	 */
	public String getFilterPath() {
		return filterPath;
	}

	/**
	 * Returns the dialog's message, which is a description of the purpose for
	 * which it was opened. This message will be visible on the dialog while it
	 * is open.
	 * 
	 * @return the message
	 */
	public String getMessage() {
		return message;
	}

	/**
	 * Makes the dialog visible and brings it to the front of the display.
	 * 
	 * @return a string describing the absolute path of the selected directory,
	 *         or null if the dialog was cancelled or an error occurred
	 * 
	 * @exception SWTException
	 *                <ul>
	 *                <li>ERROR_WIDGET_DISPOSED - if the dialog has been
	 *                disposed</li>
	 *                <li>ERROR_THREAD_INVALID_ACCESS - if not called from the
	 *                thread that created the dialog</li>
	 *                </ul>
	 */
	public String open() {
		checkDialog();

		DisposeListener listener = new DisposeListener() {
			public void widgetDisposed(DisposeEvent e) {
				if (e.widget == parent) {
					/*
					 * close dialogs which opened on top of parent
					 */
					OS.QDialog_swt_closeDialogs(parent.handle, dialogID);
				}

			}
		};

		/*
		 * listen to parent dispose event
		 */
		parent.addDisposeListener(listener);

		int qtLayoutDirection = (style & SWT.RIGHT_TO_LEFT) != 0 ? OS.QT_RIGHTTOLEFT
				: OS.QT_LEFTTORIGHT;
		String resultString = OS.QFileDialog_swt_getOpenDirectoryName(parent.handle,
				title, filterPath, dialogID, qtLayoutDirection);

		if (parent != null && !parent.isDisposed()) {
			parent.removeDisposeListener(listener);
		}

		if (resultString != null) {
			filterPath = resultString;
			int len = resultString.length();
			if (len > 0) {
				if (resultString.charAt(len - 1) == SEPARATOR) {
					filterPath = resultString.substring(0, len - 1);
				}
			}
		} else {
			filterPath = null;
		}

		return filterPath;
	}

	/**
	 * Sets the path that the dialog will use to filter the directories it shows
	 * to the argument, which may be null. If the string is null, then the
	 * operating system's default filter path will be used.
	 * <p>
	 * Note that the path string is platform dependent. For convenience, either
	 * '/' or '\' can be used as a path separator.
	 * </p>
	 * 
	 * @param string
	 *            the filter path
	 */
	public void setFilterPath(String string) {
		filterPath = string;
	}

	/**
	 * Sets the dialog's message, which is a description of the purpose for
	 * which it was opened. This message will be visible on the dialog while it
	 * is open.
	 * 
	 * @param string
	 *            the message
	 * 
	 * @exception IllegalArgumentException
	 *                <ul>
	 *                <li>ERROR_NULL_ARGUMENT - if the string is null</li>
	 *                </ul>
	 */
	public void setMessage(String string) {
		if (string == null)
			error(SWT.ERROR_NULL_ARGUMENT);
		message = string;
	}

	void checkDialog() {
		if (parent == null)
			error(SWT.ERROR_WIDGET_DISPOSED);
		if (parent.isDisposed())
			error(SWT.ERROR_WIDGET_DISPOSED);
		Display display = parent.display;
		if (display == null)
			error(SWT.ERROR_WIDGET_DISPOSED);
		if (display.thread != Thread.currentThread())
			error(SWT.ERROR_THREAD_INVALID_ACCESS);
	}

}
