/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.ui.internal;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.commands.CommandManager;
import org.eclipse.core.commands.contexts.ContextManager;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionDelta;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IRegistryChangeEvent;
import org.eclipse.core.runtime.IRegistryChangeListener;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SafeRunner;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.dynamichelpers.IExtensionTracker;
import org.eclipse.jface.operation.ModalContext;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferenceDialog;
import org.eclipse.jface.preference.PreferenceManager;
import org.eclipse.jface.util.ListenerList;
import org.eclipse.jface.util.OpenStrategy;
import org.eclipse.jface.util.SafeRunnable;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IElementFactory;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IPerspectiveDescriptor;
import org.eclipse.ui.IPerspectiveRegistry;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IWindowListener;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPreferenceConstants;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.WorkbenchException;
import org.eclipse.ui.XMLMemento;
import org.eclipse.ui.application.IWorkbenchConfigurer;
import org.eclipse.ui.application.WorkbenchAdvisor;
import org.eclipse.ui.commands.ICommandService;
import org.eclipse.ui.commands.IWorkbenchCommandSupport;
import org.eclipse.ui.handlers.IHandlerService;
import org.eclipse.ui.internal.commands.CommandService;
import org.eclipse.ui.internal.commands.WorkbenchCommandSupport;
import org.eclipse.ui.internal.handlers.HandlerService;
import org.eclipse.ui.internal.misc.Assert;
import org.eclipse.ui.internal.misc.ShowMessage;
import org.eclipse.ui.internal.misc.UIStats;
import org.eclipse.ui.internal.registry.UIExtensionTracker;
import org.eclipse.ui.internal.util.PrefUtil;
import org.eclipse.ui.views.IViewRegistry;
//import org.eclipse.ui.wizards.IWizardRegistry;

/**
 * The workbench class represents the top of the Eclipse user interface. Its
 * primary responsability is the management of workbench windows, org.eclipse.jface.dialogs,
 * wizards, and other workbench-related windows.
 * <p>
 * Note that any code that is run during the creation of a workbench instance
 * should not required access to the display.
 * </p>
 * <p>
 * Note that this internal class changed significantly between 2.1 and 3.0.
 * Applications that used to define subclasses of this internal class need to be
 * rewritten to use the new workbench advisor API.
 * </p>
 */
public final class Workbench implements IWorkbench {
	private static final String VERSION_STRING[] = { "0.046", "2.0" }; //$NON-NLS-1$ //$NON-NLS-2$

	private static final String DEFAULT_WORKBENCH_STATE_FILENAME = "workbench.xml"; //$NON-NLS-1$

	/**
	 * Holds onto the only instance of Workbench.
	 */
	private static Workbench instance;

	/**
	 * The testable object facade.
	 * 
	 * @since 3.0
	 */
//	private static WorkbenchTestable testableObject;

	/**
	 * The display used for all UI interactions with this workbench.
	 * 
	 * @since 3.0
	 */
	private Display display;

//	private WindowManager windowManager;

	//ELIU: only support one org.eclipse.jface.window for now.
	private WorkbenchWindow activatedWindow;

//	private EditorHistory editorHistory;

	private boolean runEventLoop = true;

	private boolean isStarting = true;

	private boolean isClosing = false;

	/**
	 * PlatformUI return code (as opposed to IPlatformRunnable return code).
	 */
	private int returnCode = PlatformUI.RETURN_UNSTARTABLE;

	private ListenerList windowListeners = new ListenerList();

	/**
	 * Advisor providing application-specific configuration and customization of
	 * the workbench.
	 * 
	 * @since 3.0
	 */
	private WorkbenchAdvisor advisor;

	/**
	 * Object for configuring the workbench. Lazily initialized to an instance
	 * unique to the workbench instance.
	 * 
	 * @since 3.0
	 */
	private WorkbenchConfigurer workbenchConfigurer;

	// for dynamic UI
	/**
	 * ExtensionEventHandler handles extension life-cycle events.
	 */
	private ExtensionEventHandler extensionEventHandler;

	/**
	 * A count of how many large updates are going on. This tracks nesting of
	 * requests to disable services during a large update -- similar to the
	 * <code>setRedraw</code> functionality on <code>Control</code>. When
	 * this value becomes greater than zero, services are disabled. When this
	 * value becomes zero, services are enabled. Please see
	 * <code>largeUpdateStart()</code> and <code>largeUpdateEnd()</code>.
	 */
	private int largeUpdates = 0;
	
	/**
	 * The map of services maintained by the workbench. These services are
	 * initialized during workbench during the <code>init</code> method.
	 */
	final Map services = new HashMap();
	
	/**
	 * This flag specifies whether the shells should be opened
	 */
	private boolean openShellEnabled = !Boolean.getBoolean("container.ercp.hidefirstapp");	

	/**
	 * Creates a new workbench.
	 * 
	 * @param display
	 *            the display to be used for all UI interactions with the
	 *            workbench
	 * @param advisor
	 *            the application-specific advisor that configures and
	 *            specializes this workbench instance
	 * @since 3.0
	 */
	private Workbench(Display display, WorkbenchAdvisor advisor) {
		super();

		if (instance != null && instance.isRunning()) {
			throw new IllegalStateException(
					WorkbenchMessages.Workbench_CreatingWorkbenchTwice);
		}
		Assert.isNotNull(display);
		Assert.isNotNull(advisor);
		this.advisor = advisor;
		this.display = display;
		Workbench.instance = this;

		// for dynamic UI [This seems to be for everything that isn't handled by
		// some
		// subclass of RegistryManager. I think that when an extension is moved
		// to the
		// RegistryManager implementation, then it should be removed from the
		// list in
		// ExtensionEventHandler#appear.
		// I've found that the new wizard extension in particular is a poor
		// choice to
		// use as an example, since the result of reading the registry is not
		// cached
		// -- so it is re-read each time. The only real contribution of this
		// dialog is
		// to show the user a nice dialog describing the addition.]
		extensionEventHandler = new ExtensionEventHandler(this);
		Platform.getExtensionRegistry().addRegistryChangeListener(
				extensionEventHandler);
	}

	/**
	 * Returns the one and only instance of the workbench, if there is one.
	 * 
	 * @return the workbench, or <code>null</code> if the workbench has not
	 *         been created, or has been created and already completed
	 */
	public static final Workbench getInstance() {
		return instance;
	}

	/**
	 * Creates the workbench and associates it with the the given display and
	 * workbench advisor, and runs the workbench UI. This entails processing and
	 * dispatching events until the workbench is closed or restarted.
	 * <p>
	 * This method is intended to be called by <code>PlatformUI</code>. Fails
	 * if the workbench UI has already been created.
	 * </p>
	 * <p>
	 * The display passed in must be the default display.
	 * </p>
	 * 
	 * @param display
	 *            the display to be used for all UI interactions with the
	 *            workbench
	 * @param advisor
	 *            the application-specific advisor that configures and
	 *            specializes the workbench
	 * @return return code {@link PlatformUI#RETURN_OK RETURN_OK}for normal
	 *         exit; {@link PlatformUI#RETURN_RESTART RETURN_RESTART}if the
	 *         workbench was terminated with a call to
	 *         {@link IWorkbench#restart IWorkbench.restart}; other values
	 *         reserved for future use
	 */
	public static final int createAndRunWorkbench(Display display,
			WorkbenchAdvisor advisor) {
		// create the workbench instance
		Workbench workbench = new Workbench(display, advisor);
		// run the workbench event loop
		int returnCode = workbench.runUI();
		return returnCode;
	}

	/**
	 * Creates the <code>Display</code> to be used by the workbench.
	 * 
	 * @return the display
	 */
	public static Display createDisplay() {
		// setup the application name used by SWT to lookup resources on some
		// platforms
		String applicationName = "eWorkbench"; //$NON-NLS-1$ //mdr WorkbenchPlugin.getDefault().getAppName();
		if (applicationName != null) {
			Display.setAppName(applicationName);
		}

		// create the display
//		Display newDisplay = null;
//		if (Policy.DEBUG_SWT_GRAPHICS || Policy.DEBUG_SWT_DEBUG) {
//			DeviceData data = new DeviceData();
//			if (Policy.DEBUG_SWT_GRAPHICS)
//				data.tracking = true;
//			if (Policy.DEBUG_SWT_DEBUG)
//				data.debug = true;
//			newDisplay = new Display(data);
//		} else {
//			newDisplay = new Display();
//		}
		Display newDisplay = new Display();
		// workaround for 1GEZ9UR and 1GF07HN
		newDisplay.setWarnings(false);

		// Set the priority higher than normal so as to be higher
		// than the JobManager.
//		Thread.currentThread().setPriority(
//				Math.min(Thread.MAX_PRIORITY, Thread.NORM_PRIORITY + 1));

		return newDisplay;
	}

	/**
	 * Returns the testable object facade, for use by the test harness.
	 * 
	 * @return the testable object facade
	 * @since 3.0
	 */
/*	public static WorkbenchTestable getWorkbenchTestable() {
		if (testableObject == null) {
			testableObject = new WorkbenchTestable();
		}
		return testableObject;
	}
*/
	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public void addWindowListener(IWindowListener l) {
		windowListeners.add(l);
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public void removeWindowListener(IWindowListener l) {
		windowListeners.remove(l);
	}

	/**
	 * Fire org.eclipse.jface.window opened event.
	 * 
	 * @param org.eclipse.jface.window
	 *            The org.eclipse.jface.window which just opened; should not be <code>null</code>.
	 */
	protected void fireWindowOpened(final IWorkbenchWindow window) {
		Object list[] = windowListeners.getListeners();
		for (int i = 0; i < list.length; i++) {
			final IWindowListener l = (IWindowListener) list[i];
			Platform.run(new SafeRunnable() {
				public void run() {
					l.windowOpened(window);
				}
			});
		}
	}

	/**
	 * Fire org.eclipse.jface.window closed event.
	 * 
	 * @param org.eclipse.jface.window
	 *            The org.eclipse.jface.window which just closed; should not be <code>null</code>.
	 */
	protected void fireWindowClosed(final IWorkbenchWindow window) {
		if (activatedWindow == window) {
			// Do not hang onto it so it can be GC'ed
			activatedWindow = null;
		}

		Object list[] = windowListeners.getListeners();
		for (int i = 0; i < list.length; i++) {
			final IWindowListener l = (IWindowListener) list[i];
			Platform.run(new SafeRunnable() {
				public void run() {
					l.windowClosed(window);
				}
			});
		}
	}

	/**
	 * Fire org.eclipse.jface.window activated event.
	 * 
	 * @param org.eclipse.jface.window
	 *            The org.eclipse.jface.window which was just activated; should not be
	 *            <code>null</code>.
	 */
	protected void fireWindowActivated(final IWorkbenchWindow window) {
		Object list[] = windowListeners.getListeners();
		for (int i = 0; i < list.length; i++) {
			final IWindowListener l = (IWindowListener) list[i];
			Platform.run(new SafeRunnable() {
				public void run() {
					l.windowActivated(window);
				}
			});
		}
	}

	/**
	 * Fire org.eclipse.jface.window deactivated event.
	 * 
	 * @param org.eclipse.jface.window
	 *            The org.eclipse.jface.window which was just deactivated; should not be
	 *            <code>null</code>.
	 */
	protected void fireWindowDeactivated(final IWorkbenchWindow window) {
		Object list[] = windowListeners.getListeners();
		for (int i = 0; i < list.length; i++) {
			final IWindowListener l = (IWindowListener) list[i];
			Platform.run(new SafeRunnable() {
				public void run() {
					l.windowDeactivated(window);
				}
			});
		}
	}

	/**
	 * Closes the workbench. Assumes that the busy cursor is active.
	 * 
	 * @param force
	 *            true if the close is mandatory, and false if the close is
	 *            allowed to fail
	 * @return true if the close succeeded, and false otherwise
	 */
	private boolean busyClose(final boolean force) {

		// notify the advisor of preShutdown and allow it to veto if not forced
		isClosing = advisor.preShutdown();
		if (!force && !isClosing) {
			return false;
		}

		// save any open editors if they are dirty
//		isClosing = saveAllEditors(!force);
//		if (!force && !isClosing) {
//			return false;
//		}

/*		IPreferenceStore store = getPreferenceStore();
		boolean closeEditors = store
				.getBoolean(IWorkbenchPreferenceConstants.CLOSE_EDITORS_ON_EXIT);
		if (closeEditors) {
			Platform.run(new SafeRunnable() {
				public void run() {
					IWorkbenchWindow windows[] = getWorkbenchWindows();
					for (int i = 0; i < windows.length; i++) {
						IWorkbenchPage pages[] = windows[i].getPages();
						for (int j = 0; j < pages.length; j++) {
							isClosing = isClosing
									&& pages[j].closeAllEditors(false);
						}
					}
				}
			});
			if (!force && !isClosing) {
				return false;
			}
		}
*/
		if (getWorkbenchConfigurer().getSaveAndRestore()) {
			Platform.run(new SafeRunnable() {
				public void run() {
					XMLMemento mem = recordWorkbenchState();
					// Save the IMemento to a file.
					saveMementoToFile(mem);
				}

				public void handleException(Throwable e) {
					String message;
					if (e.getMessage() == null) {
						message = WorkbenchMessages.ErrorClosingNoArg;
					} else {
						message = NLS.bind(
								WorkbenchMessages.ErrorClosingOneArg, e
										.getMessage());
					}

					if (!ShowMessage.openQuestion(null,
							WorkbenchMessages.Error, message)) {
						isClosing = false;
					}
				}
			});
		}
		if (!force && !isClosing) {
			return false;
		}

		// Simulate the behaviour of JFace's WindowManager
		SafeRunner.run(new SafeRunnable(WorkbenchMessages.ErrorClosing) {
			public void run() {
				if (isClosing || force) {
					IWorkbenchWindow[] windows = getWorkbenchWindows();
					for (int i = 0; i < windows.length; i++) {
						IWorkbenchWindow window = windows[i];
						isClosing = window.close();
					// stop closing windows after the first which refuse to close
						if (!isClosing)
							break;
					}
				}
			}
		});
		if (!force && !isClosing) {
			return false;
		}
		
		shutdown();

		runEventLoop = false;
		return true;
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public boolean saveAllEditors(boolean confirm) {
/*		final boolean finalConfirm = confirm;
		final boolean[] result = new boolean[1];
		result[0] = true;

		Platform.run(new SafeRunnable(WorkbenchMessages.ErrorClosing) {
			public void run() {
				// Collect dirtyEditors
				ArrayList dirtyEditors = new ArrayList();
				ArrayList dirtyEditorsInput = new ArrayList();
				IWorkbenchWindow windows[] = getWorkbenchWindows();
				for (int i = 0; i < windows.length; i++) {
					IWorkbenchPage pages[] = windows[i].getPages();
					for (int j = 0; j < pages.length; j++) {
						WorkbenchPage page = (WorkbenchPage) pages[j];
						IEditorPart editors[] = page.getDirtyEditors();
						for (int k = 0; k < editors.length; k++) {
							IEditorPart editor = editors[k];
							if (editor.isSaveOnCloseNeeded()) {
								if (!dirtyEditorsInput.contains(editor
										.getEditorInput())) {
									dirtyEditors.add(editor);
									dirtyEditorsInput.add(editor
											.getEditorInput());
								}
							}
						}
					}
				}
				if (dirtyEditors.size() > 0) {
					IWorkbenchWindow w = getActiveWorkbenchWindow();
					if (w == null)
						w = windows[0];
					result[0] = EditorManager.saveAll(dirtyEditors,
							finalConfirm, w);
				}
			}
		});
		return result[0];
*/
		return true;
	}
	
	/**
	 * Opens a new workbench org.eclipse.jface.window and page with a specific perspective.
	 * 
	 * Assumes that busy cursor is active.
	 */
	private IWorkbenchWindow busyOpenWorkbenchWindow(String perspID,
			IAdaptable input) throws WorkbenchException {
		// Create a workbench org.eclipse.jface.window (becomes active org.eclipse.jface.window)
		WorkbenchWindow newWindow = newWorkbenchWindow();
		newWindow.create(); // must be created before adding to org.eclipse.jface.window manager
//		windowManager.add(newWindow);

		// Create the initial page.
		if (perspID != null) {
			try {
				newWindow.busyOpenPage(perspID, input);
			} catch (WorkbenchException e) {
//				windowManager.remove(newWindow);
				throw e;
			}
		}

		// Open org.eclipse.jface.window after opening page, to avoid flicker.
		newWindow.open();

		return newWindow;
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public boolean close() {
		Integer assignedReturnCode = (Integer) System.getProperties().get("eworkbench.returnCode");
		if(assignedReturnCode != null )
			return close(assignedReturnCode.intValue(), true);
		return close(PlatformUI.RETURN_OK, false);
	}

	/**
	 * Closes the workbench, returning the given return code from the run
	 * method. If forced, the workbench is closed no matter what.
	 * 
	 * @param returnCode
	 *            {@link PlatformUI#RETURN_OK RETURN_OK}for normal exit;
	 *            {@link PlatformUI#RETURN_RESTART RETURN_RESTART}if the
	 *            workbench was terminated with a call to
	 *            {@link IWorkbench#restart IWorkbench.restart};
	 *            {@link PlatformUI#RETURN_UNSTARTABLE RETURN_UNSTARTABLE}if
	 *            the workbench could not be started; other values reserved for
	 *            future use
	 * @param force
	 *            true to force the workbench close, and false for a "soft"
	 *            close that can be canceled
	 * @return true if the close was successful, and false if the close was
	 *         canceled
	 */
	/* package */
	boolean close(int returnCode, final boolean force) {
		this.returnCode = returnCode;
		final boolean[] ret = new boolean[1];
/*
		// fixed Bugzilla Bug 191426 begin
		Shell shells[] = display.getShells();
		for (int i = 0; i < shells.length; i++) {
			Object data = shells[i].getData();
			if (!(data instanceof IWorkbenchWindow))
				continue;
			IWorkbenchPage[] pages = ((IWorkbenchWindow)data).getPages();
			for(int j=0; j<pages.length; j++) {
				if(pages[j] == null)
					continue;
				IViewReference[] views = pages[j].getViewReferences();
				for(int k=0; k<views.length; k++) {
					if(views[k] == null)
						continue;
					IViewPart view = views[k].getView(false);
					if(view != null) {
						view.dispose();
					}
				}
			}			
		}
		// fixed Bugzilla Bug 191426 end
	*/
//ELIU Do not support busyindicator		
//		BusyIndicator.showWhile(null, new Runnable() {
//			public void run() {
				ret[0] = busyClose(force);
//			}
//		});
		return ret[0];
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public IWorkbenchWindow getActiveWorkbenchWindow() {
		// Return null if called from a non-UI thread.
		// This is not spec'ed behaviour and is misleading, however this is how
		// it
		// worked in 2.1 and we cannot change it now.
		// For more details, see [Bug 57384] [RCP] Main org.eclipse.jface.window not active on
		// startup
		if (Display.getCurrent() == null) {
			return null;
		}

		// Look at the current shell and up its parent
		// hierarchy for a workbench org.eclipse.jface.window.
		Control shell = display.getActiveShell();
		//Add shell disposed filer to prevent incorrect native event. 
		while (shell != null && !shell.isDisposed()) {			
			Object data = shell.getData();
			if (data instanceof IWorkbenchWindow)
				return (IWorkbenchWindow) data;
			shell = shell.getParent();
		}

		// Look for the org.eclipse.jface.window that was last known being
		// the active one
		WorkbenchWindow win = getActivatedWindow();
		if (win != null) {
			return win;
		}

		// Look at all the shells and pick the first one
		// that is a workbench org.eclipse.jface.window.
		Shell shells[] = display.getShells();
		for (int i = 0; i < shells.length; i++) {
			Object data = shells[i].getData();
			if (data instanceof IWorkbenchWindow)
				return (IWorkbenchWindow) data;
		}

		// Can't find anything!
		return null;
	}

	/*
	 * Returns the editor history.
	 */
//	protected EditorHistory getEditorHistory() {
//		if (editorHistory == null) {
//			editorHistory = new EditorHistory();
//		}
//		return editorHistory;
//	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
//	public IEditorRegistry getEditorRegistry() {
//		return WorkbenchPlugin.getDefault().getEditorRegistry();
//	}

	/*
	 * Returns the number for a new org.eclipse.jface.window. This will be the first number > 0
	 * which is not used to identify another org.eclipse.jface.window in the workbench.
	 */
//	private int getNewWindowNumber() {
//		// Get org.eclipse.jface.window list.
//		Window[] windows = windowManager.getWindows();
//		int count = windows.length;
//
//		// Create an array of booleans (size = org.eclipse.jface.window count).
//		// Cross off every number found in the org.eclipse.jface.window list.
//		boolean checkArray[] = new boolean[count];
//		for (int nX = 0; nX < count; nX++) {
//			if (windows[nX] instanceof WorkbenchWindow) {
//				WorkbenchWindow ww = (WorkbenchWindow) windows[nX];
//				int index = ww.getNumber() - 1;
//				if (index >= 0 && index < count)
//					checkArray[index] = true;
//			}
//		}
//
//		// Return first index which is not used.
//		// If no empty index was found then every slot is full.
//		// Return next index.
//		for (int index = 0; index < count; index++) {
//			if (!checkArray[index])
//				return index + 1;
//		}
//		return count + 1;
//	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
//	public IWorkbenchOperationSupport getOperationSupport() {
//		return WorkbenchPlugin.getDefault().getOperationSupport();
//		return null;
//	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public IPerspectiveRegistry getPerspectiveRegistry() {
		return WorkbenchPlugin.getDefault().getPerspectiveRegistry();
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public PreferenceManager getPreferenceManager() {
		return WorkbenchPlugin.getDefault().getPreferenceManager();
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public IPreferenceStore getPreferenceStore() {
		return WorkbenchPlugin.getDefault().getPreferenceStore();
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public ISharedImages getSharedImages() {
		return WorkbenchPlugin.getDefault().getSharedImages();
	}

	/**
	 * Returns the org.eclipse.jface.window manager for this workbench.
	 * 
	 * @return the org.eclipse.jface.window manager
	 */
	/* package */
//	WindowManager getWindowManager() {
//		return windowManager;
//	}

	/*
	 * Answer the workbench state file.
	 */
	private File getWorkbenchStateFile() {
		IPath path = WorkbenchPlugin.getDefault().getDataLocation();
		if (path == null)
			return null;
		path = path.append(DEFAULT_WORKBENCH_STATE_FILENAME);
		return path.toFile();
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public int getWorkbenchWindowCount() {
		IWorkbenchWindow[] windows = getWorkbenchWindows();
		return windows != null ? windows.length : 0;
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public IWorkbenchWindow[] getWorkbenchWindows() {
		if (Display.getCurrent() == null) return null;
		
		Shell[] shells = Display.getCurrent().getShells();
		List windows = new ArrayList(shells.length);
		for (int i = 0; i < shells.length; i++) {
			if (shells[i].getData() instanceof WorkbenchWindow) {
				windows.add(shells[i].getData());
			}
		}
		return (IWorkbenchWindow[]) windows.toArray(new IWorkbenchWindow[windows.size()]);
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
//	public IWorkingSetManager getWorkingSetManager() {
//		return WorkbenchPlugin.getDefault().getWorkingSetManager();
//	}

	/**
	 * {@inheritDoc}
	 */
//	public ILocalWorkingSetManager createLocalWorkingSetManager() {
//		return new LocalWorkingSetManager(WorkbenchPlugin.getDefault()
//				.getBundleContext());
//	}

	/**
	 * Initializes the workbench now that the display is created.
	 * 
	 * @return true if init succeeded.
	 */
	private boolean init(Display display) {
		// setup debug mode if required.
		if (WorkbenchPlugin.getDefault().isDebugging()) {
			WorkbenchPlugin.DEBUG = true;
			ModalContext.setDebugMode(true);
		}

		// create workbench org.eclipse.jface.window manager
//		windowManager = new WindowManager();

/*		IIntroRegistry introRegistry = WorkbenchPlugin.getDefault()
				.getIntroRegistry();
		if (introRegistry.getIntroCount() > 0) {
			IProduct product = Platform.getProduct();
			if (product != null) {
				introDescriptor = (IntroDescriptor) introRegistry
						.getIntroForProduct(product.getId());
			}
		}
*/
		// Initialize the activity support.
//		workbenchActivitySupport = new WorkbenchActivitySupport();
//		activityHelper = ActivityPersistanceHelper.getInstance();

		/*
		 * Phase 1 of the initialization of commands. When this phase completes,
		 * all the services and managers will exist, and be accessible via the
		 * getService(Object) method.
		 */  
		commandManager = new CommandManager();
		final CommandService commandService = new CommandService(
				commandManager);
		services.put(ICommandService.class, commandService);
//		ContextManager.DEBUG = Policy.DEBUG_CONTEXTS;
//		contextManager = new ContextManager();
//		final IContextService contextService = new ContextService(
//				contextManager);
//		services.put(IContextService.class, contextService);
		final IHandlerService handlerService = new HandlerService(
				commandManager);
		services.put(IHandlerService.class, handlerService);
//		BindingManager.DEBUG = Policy.DEBUG_KEY_BINDINGS;
//		bindingManager = new BindingManager(contextManager, commandManager);
//		final IBindingService bindingService = new BindingService(
//				bindingManager, this);
//		services.put(IBindingService.class, bindingService);

		/*
		 * Phase 2 of the initialization of commands. The registry and
		 * org.eclipse.jface.preference store are parsed into memory. When this phase completes,
		 * all persisted state should be available to the application.
		 */
		commandService.readRegistry();
//		handlerService.readRegistry();
//		contextService.readRegistry();
//		bindingService.readRegistryAndPreferences(commandService);
		
		/*
		 * Phase 3 of the initialization of commands. The source providers that
		 * the workbench provides are creating and registered with the above
		 * services. These source providers notify the services when particular
		 * pieces of workbench state change.
		 */
//		final ActiveShellSourceProvider activeShellSourceProvider = new ActiveShellSourceProvider(
//				this);
//		handlerService.addSourceProvider(activeShellSourceProvider);
//		contextService.addSourceProvider(activeShellSourceProvider);
//		final ActivePartSourceProvider activePartSourceProvider = new ActivePartSourceProvider(
//				this);
//		handlerService.addSourceProvider(activePartSourceProvider);
//		contextService.addSourceProvider(activePartSourceProvider);

		/*
		 * TODO Putting this here is a like a sword in my side. But alas, the
		 * handler support from XML just isn't up to the task yet.
		 */
//m2		final Command showViewCommand = commandService
//m2				.getCommand("org.eclipse.ui.views.showView"); //$NON-NLS-1$
//m2		showViewCommand.setHandler(new ShowViewHandler());
//m2		final Command showPerspectiveCommand = commandService
//m2				.getCommand("org.eclipse.ui.perspectives.showPerspective"); //$NON-NLS-1$
//m2		showPerspectiveCommand.setHandler(new ShowPerspectiveHandler());

		/*
		 * Phase 4 of the initialization of commands.  This handles the creation
		 * of wrappers for legacy APIs.  By the time this phase completes, any
		 * code trying to access commands through legacy APIs should work.
		 *  
		 * TODO This is the deprecated support. It would be nice to pull out all
		 * of this except for the Workbench*Support constructors.
		 */
//		workbenchContextSupport = new WorkbenchContextSupport(this,
//				contextManager);
//		workbenchCommandSupport = new WorkbenchCommandSupport(bindingManager,
//				commandManager, contextManager, handlerService);
//		workbenchCommandSupport = new WorkbenchCommandSupport(bindingManager,
//				commandManager, null, null);
//m		initializeCommandResolver();

		addWindowListener(windowListener);

		initializeImages();
		initializeFonts();
		initializeColors();
		initializeApplicationColors();

		// now that the workbench is sufficiently initialized, let the advisor
		// have a turn.
		advisor.internalBasicInitialize(getWorkbenchConfigurer());

		// configure use of color icons in toolbars
		boolean useColorIcons = getPreferenceStore().getBoolean(
				IPreferenceConstants.COLOR_ICONS);
//ELIU		ActionContributionItem.setUseColorIconsInToolbars(useColorIcons);

		// initialize workbench single-click vs double-click behavior
		initializeSingleClickOption();

		// deadlock code
		boolean avoidDeadlock = true;

		String[] commandLineArgs = Platform.getCommandLineArgs();
		for (int i = 0; i < commandLineArgs.length; i++) {
			if (commandLineArgs[i].equalsIgnoreCase("-allowDeadlock")) //$NON-NLS-1$
				avoidDeadlock = false;
		}

		if (avoidDeadlock) {
			UILockListener uiLockListener = new UILockListener(display);
			Platform.getJobManager().setLockListener(uiLockListener);
			display.setSynchronizer(new UISynchronizer(display, uiLockListener));
		}

		// attempt to restore a previous workbench state
		try {
			UIStats.start(UIStats.RESTORE_WORKBENCH, "Workbench"); //$NON-NLS-1$

			advisor.preStartup();

			if (!advisor.openWindows()) {
				return false;
			}

		} finally {
			UIStats.end(UIStats.RESTORE_WORKBENCH, this, "Workbench"); //$NON-NLS-1$
		}

		forceOpenPerspective();

		isStarting = false;

		return true;
	}

	/**
	 * Establishes the relationship between JFace actions and the command
	 * manager.
	 */
/*	private void initializeCommandResolver() {
		ExternalActionManager.getInstance().setCallback(
				new CommandCallback(bindingManager, commandManager,
						new IActiveChecker() {
							public final boolean isActive(final String commandId) {
								return workbenchActivitySupport
										.getActivityManager().getIdentifier(
												commandId).isEnabled();
							}
						}));
	}
*/
	
	/**
	 * Initialize colors defined by the new colorDefinitions extension point.
	 * Note this will be rolled into initializeColors() at some point.
	 * 
	 * @since 3.0
	 */
	private void initializeApplicationColors() {
//		ColorDefinition[] colorDefinitions = WorkbenchPlugin.getDefault()
//				.getThemeRegistry().getColors();
//		ThemeElementHelper.populateRegistry(getThemeManager().getTheme(
//				IThemeManager.DEFAULT_THEME), colorDefinitions,
//				getPreferenceStore());
	}

	private void initializeSingleClickOption() {
		IPreferenceStore store = WorkbenchPlugin.getDefault()
				.getPreferenceStore();
		boolean openOnSingleClick = store
				.getBoolean(IPreferenceConstants.OPEN_ON_SINGLE_CLICK);
		boolean selectOnHover = store
				.getBoolean(IPreferenceConstants.SELECT_ON_HOVER);
		boolean openAfterDelay = store
				.getBoolean(IPreferenceConstants.OPEN_AFTER_DELAY);
		int singleClickMethod = openOnSingleClick ? OpenStrategy.SINGLE_CLICK
				: OpenStrategy.DOUBLE_CLICK;
		if (openOnSingleClick) {
			if (selectOnHover)
				singleClickMethod |= OpenStrategy.SELECT_ON_HOVER;
			if (openAfterDelay)
				singleClickMethod |= OpenStrategy.ARROW_KEYS_OPEN;
		}
		OpenStrategy.setOpenMethod(singleClickMethod);
	}

	/*
	 * Initializes the workbench fonts with the stored values.
	 */
	private void initializeFonts() {
//		FontDefinition[] fontDefinitions = WorkbenchPlugin.getDefault()
//				.getThemeRegistry().getFonts();
//		ThemeElementHelper.populateRegistry(getThemeManager().getTheme(
//				IThemeManager.DEFAULT_THEME), fontDefinitions,
//				getPreferenceStore());
	}

	/*
	 * Initialize the workbench images.
	 * 
	 * @param windowImages An array of the descriptors of the images to be used
	 * in the corner of each org.eclipse.jface.window, or <code>null</code> if none. It is
	 * expected that the array will contain the same icon, rendered at different
	 * sizes.
	 * 
	 * @since 3.0
	 */
	private void initializeImages() {
/*		ImageDescriptor[] windowImages = WorkbenchPlugin.getDefault()
				.getWindowImages();
		if (windowImages == null)
			return;

		Image[] images = new Image[windowImages.length];
		for (int i = 0; i < windowImages.length; ++i)
			images[i] = windowImages[i].createImage();
		Window.setDefaultImages(images);
*/
	}

	/*
	 * Take the workbenches' images out of the shared registry.
	 * 
	 * @since 3.0
	 */
	private void uninitializeImages() {
		WorkbenchImages.dispose();
//		Window.setDefaultImage(null);
	}

	/*
	 * Initialize the workbench colors.
	 * 
	 * @since 3.0
	 */
	private void initializeColors() {
		// @issue some colors are generic; some are app-specific
		WorkbenchColors.startup();
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public boolean isClosing() {
		return isClosing;
	}

	/*
	 * Returns true if the workbench is in the process of starting
	 */
	/* package */
	boolean isStarting() {
		return isStarting;
	}

	/*
	 * Creates a new workbench org.eclipse.jface.window.
	 * 
	 * @return the new workbench org.eclipse.jface.window
	 */
	private WorkbenchWindow newWorkbenchWindow() {
//ELIU		return new WorkbenchWindow(getNewWindowNumber());
		return new WorkbenchWindow();

	}

	/*
	 * If a perspective was specified on the command line (-perspective) then
	 * force that perspective to open in the active org.eclipse.jface.window.
	 */
	private void forceOpenPerspective() {
//		if (getWorkbenchWindowCount() == 0) {
//			// there should be an open org.eclipse.jface.window by now, bail out.
//			return;
//		}

		String perspId = null;
		String[] commandLineArgs = Platform.getCommandLineArgs();
		for (int i = 0; i < commandLineArgs.length - 1; i++) {
			if (commandLineArgs[i].equalsIgnoreCase("-perspective")) { //$NON-NLS-1$
				perspId = commandLineArgs[i + 1];
				break;
			}
		}
		if (perspId == null) {
			return;
		}
		IPerspectiveDescriptor desc = getPerspectiveRegistry()
				.findPerspectiveWithId(perspId);
		if (desc == null) {
			return;
		}

		IWorkbenchWindow win = getActiveWorkbenchWindow();
		if (win == null) {
			if(getWorkbenchWindowCount() ==0){
				win = null;
			} else {
				win = getWorkbenchWindows()[0];
			}
		}
		try {
			showPerspective(perspId, win);
		} catch (WorkbenchException e) {
			String msg = "Workbench exception showing specified command line perspective on startup."; //$NON-NLS-1$
			WorkbenchPlugin.log(msg, new Status(IStatus.ERROR,
					PlatformUI.PLUGIN_ID, 0, msg, e));
		}
	}

	/**
	 * Opens the initial workbench org.eclipse.jface.window.
	 */
	/* package */void openFirstTimeWindow() {
		try {
			busyOpenWorkbenchWindow(getPerspectiveRegistry()
					.getDefaultPerspective(), getDefaultPageInput());
		} catch (WorkbenchException e) {
			// Don't use the org.eclipse.jface.window's shell as the dialog parent,
			// as the org.eclipse.jface.window is not open yet (bug 76724).
//ELIU			ErrorDialog.openError(null,
//					WorkbenchMessages.Problems_Opening_Page, e.getMessage(), e
//							.getStatus());
			ShowMessage.printError(e.getMessage(), e
					.getStatus());
		}
	}

	/*
	 * Restores the workbench UI from the workbench state file (workbench.xml).
	 * 
	 * @return a status object indicating OK if a org.eclipse.jface.window was opened,
	 * RESTORE_CODE_RESET if no org.eclipse.jface.window was opened but one should be, and
	 * RESTORE_CODE_EXIT if the workbench should close immediately
	 */
	/* package */IStatus restoreState() {

		if (!getWorkbenchConfigurer().getSaveAndRestore()) {
			String msg = WorkbenchMessages.Workbench_restoreDisabled;
			return new Status(IStatus.WARNING, WorkbenchPlugin.PI_WORKBENCH,
					IWorkbenchConfigurer.RESTORE_CODE_RESET, msg, null);
		}
		// Read the workbench state file.
		final File stateFile = getWorkbenchStateFile();
		// If there is no state file cause one to open.
		if (stateFile == null || !stateFile.exists()) {
			String msg = WorkbenchMessages.Workbench_noStateToRestore;
			return new Status(IStatus.WARNING, WorkbenchPlugin.PI_WORKBENCH,
					IWorkbenchConfigurer.RESTORE_CODE_RESET, msg, null);
		}

		final IStatus result[] = { new Status(IStatus.OK,
				WorkbenchPlugin.PI_WORKBENCH, IStatus.OK, "", null) }; //$NON-NLS-1$
		Platform.run(new SafeRunnable(WorkbenchMessages.ErrorReadingState) {
			public void run() throws Exception {
				FileInputStream input = new FileInputStream(stateFile);
				BufferedReader reader = new BufferedReader(
						new InputStreamReader(input, "utf-8")); //$NON-NLS-1$
				IMemento memento = XMLMemento.createReadRoot(reader);

				// Validate known version format
				String version = memento
						.getString(IWorkbenchConstants.TAG_VERSION);
				boolean valid = false;
				for (int i = 0; i < VERSION_STRING.length; i++) {
					if (VERSION_STRING[i].equals(version)) {
						valid = true;
						break;
					}
				}
				if (!valid) {
					reader.close();
					String msg = WorkbenchMessages.Invalid_workbench_state_ve;
					ShowMessage.openError((Shell) null,
							WorkbenchMessages.Restoring_Problems, msg); //$NON-NLS-1$
					stateFile.delete();
					result[0] = new Status(IStatus.ERROR,
							WorkbenchPlugin.PI_WORKBENCH,
							IWorkbenchConfigurer.RESTORE_CODE_RESET, msg, null);
					return;
				}

				// Validate compatible version format
				// We no longer support the release 1.0 format
				if (VERSION_STRING[0].equals(version)) {
					reader.close();
					String msg = WorkbenchMessages.Workbench_incompatibleSavedStateVersion;
//					boolean ignoreSavedState = new MessageDialog(null,
//							WorkbenchMessages.Workbench_incompatibleUIState,
//							null, msg, ShowMessage.WARNING, new String[] {
//									IDialogConstants.OK_LABEL,
//									IDialogConstants.CANCEL_LABEL }, 0).open() == 0;
					boolean ignoreSavedState = ShowMessage.openBox(null,
							WorkbenchMessages.Workbench_incompatibleUIState,
							msg,SWT.ICON_WARNING | SWT.OK | SWT.CANCEL)==SWT.OK;
					// OK is the default
					if (ignoreSavedState) {
						stateFile.delete();
						result[0] = new Status(IStatus.WARNING,
								WorkbenchPlugin.PI_WORKBENCH,
								IWorkbenchConfigurer.RESTORE_CODE_RESET, msg,
								null);
					} else {
						result[0] = new Status(IStatus.WARNING,
								WorkbenchPlugin.PI_WORKBENCH,
								IWorkbenchConfigurer.RESTORE_CODE_EXIT, msg,
								null);
					}
					return;
				}

				// Restore the saved state
				IStatus restoreResult = restoreState(memento);
				reader.close();
				if (restoreResult.getSeverity() == IStatus.ERROR) {
//ELIU					ErrorDialog.openError(null,
//							WorkbenchMessages.Workspace_problemsTitle,
//							WorkbenchMessages.Workbench_problemsRestoringMsg,
//							restoreResult);
					ShowMessage.printError(WorkbenchMessages.Workbench_problemsRestoringMsg,
							restoreResult);
				}
			}

			public void handleException(Throwable e) {
				super.handleException(e);
				String msg = e.getMessage() == null ? "" : e.getMessage(); //$NON-NLS-1$
				result[0] = new Status(IStatus.ERROR,
						WorkbenchPlugin.PI_WORKBENCH,
						IWorkbenchConfigurer.RESTORE_CODE_RESET, msg, e);
				stateFile.delete();
			}

		});
		// ensure at least one org.eclipse.jface.window was opened
//		if (result[0].isOK() && windowManager.getWindows().length == 0) {
//			String msg = WorkbenchMessages.Workbench_noWindowsRestored;
//			result[0] = new Status(IStatus.ERROR, WorkbenchPlugin.PI_WORKBENCH,
//					IWorkbenchConfigurer.RESTORE_CODE_RESET, msg, null);
//		}
		return result[0];
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public IWorkbenchWindow openWorkbenchWindow(IAdaptable input)
			throws WorkbenchException {
		return openWorkbenchWindow(getPerspectiveRegistry()
				.getDefaultPerspective(), input);
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public IWorkbenchWindow openWorkbenchWindow(final String perspID,
			final IAdaptable input) throws WorkbenchException {
		// Run op in busy cursor.
		final Object[] result = new Object[1];
//ELIU do not support busyindicator		
//		BusyIndicator.showWhile(null, new Runnable() {
//			public void run() {
				try {
					result[0] = busyOpenWorkbenchWindow(perspID, input);
				} catch (WorkbenchException e) {
					result[0] = e;
				}
//			}
//		});
		if (result[0] instanceof IWorkbenchWindow) {
			return (IWorkbenchWindow) result[0];
		} else if (result[0] instanceof WorkbenchException) {
			throw (WorkbenchException) result[0];
		} else {
			throw new WorkbenchException(
					WorkbenchMessages.Abnormal_Workbench_Conditi);
		}
	}

	/*
	 * Record the workbench UI in a document
	 */
	private XMLMemento recordWorkbenchState() {
		XMLMemento memento = XMLMemento
				.createWriteRoot(IWorkbenchConstants.TAG_WORKBENCH);
		IStatus status = saveState(memento);
		if (status.getSeverity() != IStatus.OK) {
			// don't use newWindow as parent because it has not yet been opened
			// (bug 76724)
//			ErrorDialog.openError(null,
//					WorkbenchMessages.Workbench_problemsSaving,
//					WorkbenchMessages.Workbench_problemsSavingMsg, status);
			ShowMessage.printError(WorkbenchMessages.Workbench_problemsSavingMsg, status);
		}
		return memento;
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public boolean restart() {
		// this is the return code from run() to trigger a restart
		return close(PlatformUI.RETURN_RESTART, false);
	}

	/*
	 * Restores the state of the previously saved workbench
	 */
	private IStatus restoreState(IMemento memento) {

		MultiStatus result = new MultiStatus(PlatformUI.PLUGIN_ID, IStatus.OK,
				WorkbenchMessages.Workbench_problemsRestoring, null);
		IMemento childMem;
		try {
			UIStats.start(UIStats.RESTORE_WORKBENCH, "MRUList"); //$NON-NLS-1$
			IMemento mruMemento = memento
					.getChild(IWorkbenchConstants.TAG_MRU_LIST); //$NON-NLS-1$
//			if (mruMemento != null) {
//				result.add(getEditorHistory().restoreState(mruMemento));
//			}
		} finally {
			UIStats.end(UIStats.RESTORE_WORKBENCH, this, "MRUList"); //$NON-NLS-1$
		}
		// Get the child windows.
		IMemento[] children = memento
				.getChildren(IWorkbenchConstants.TAG_WINDOW);

		// Read the workbench windows.
		for (int x = 0; x < children.length; x++) {
			childMem = children[x];
			WorkbenchWindow newWindow = newWorkbenchWindow();
			newWindow.create();

			// allow the application to specify an initial perspective to open
			// @issue temporary workaround for ignoring initial perspective
			// String initialPerspectiveId =
			// getAdvisor().getInitialWindowPerspectiveId();
			// if (initialPerspectiveId != null) {
			// IPerspectiveDescriptor desc =
			// getPerspectiveRegistry().findPerspectiveWithId(initialPerspectiveId);
			// result.merge(newWindow.restoreState(childMem, desc));
			// }
			// add the org.eclipse.jface.window so that any work done in newWindow.restoreState
			// that relies on Workbench methods has windows to work with
//			windowManager.add(newWindow);
			// whether the org.eclipse.jface.window was opened
			boolean opened = false;
			// now that we've added it to the org.eclipse.jface.window manager we need to listen
			// for any exception that might hose us before we get a chance to
			// open it. If one occurs, remove the new org.eclipse.jface.window from the manager.
			try {
				result.merge(newWindow.restoreState(childMem, null));
				try {
					newWindow.fireWindowRestored();
				} catch (WorkbenchException e) {
					result.add(e.getStatus());
				}
				newWindow.open();
				opened = true;
			} finally {
				if (!opened)
					newWindow.close();
			}
		}
		return result;
	}

	/**
	 * Returns an array with the ids of all plugins that extend the
	 * <code>org.eclipse.ui.startup</code> extension point.
	 * 
	 * @return String[]
	 */
	public String[] getEarlyActivatedPlugins() {
		IExtensionPoint point = Platform.getExtensionRegistry()
				.getExtensionPoint(PlatformUI.PLUGIN_ID,
						IWorkbenchConstants.PL_STARTUP);
		IExtension[] extensions = point.getExtensions();
		String[] result = new String[extensions.length];
		for (int i = 0; i < extensions.length; i++) {
			result[i] = extensions[i].getNamespace();
		}
		return result;
	}

	/*
	 * Starts all plugins that extend the <code> org.eclipse.ui.startup </code>
	 * extension point, and that the user has not disabled via the org.eclipse.jface.preference
	 * page.
	 */
	private void startPlugins() {
		Runnable work = new Runnable() {
			final String disabledPlugins = getPreferenceStore().getString(
					IPreferenceConstants.PLUGINS_NOT_ACTIVATED_ON_STARTUP);

			public void run() {
				IExtensionRegistry registry = Platform.getExtensionRegistry();

				// bug 55901: don't use getConfigElements directly, for pre-3.0
				// compat, make sure to allow both missing class
				// attribute and a missing startup element
				IExtensionPoint point = registry.getExtensionPoint(
						PlatformUI.PLUGIN_ID, IWorkbenchConstants.PL_STARTUP);

				IExtension[] extensions = point.getExtensions();
				for (int i = 0; i < extensions.length; ++i) {
					IExtension extension = extensions[i];

					// if the plugin is not in the set of disabled plugins, then
					// execute the code to start it
					if (disabledPlugins.indexOf(extension.getNamespace()) == -1)
						Platform.run(new EarlyStartupRunnable(extension));
				}
			}
		};

		Thread thread = new Thread(work);
		thread.start();
	}

	/**
	 * Internal method for running the workbench UI. This entails processing and
	 * dispatching events until the workbench is closed or restarted.
	 * 
	 * @return return code {@link PlatformUI#RETURN_OK RETURN_OK}for normal
	 *         exit; {@link PlatformUI#RETURN_RESTART RETURN_RESTART}if the
	 *         workbench was terminated with a call to
	 *         {@link IWorkbench#restart IWorkbench.restart};
	 *         {@link PlatformUI#RETURN_UNSTARTABLE RETURN_UNSTARTABLE}if the
	 *         workbench could not be started; other values reserved for future
	 *         use
	 * @since 3.0
	 */
	private int runUI() {
		UIStats.start(UIStats.START_WORKBENCH, "Workbench"); //$NON-NLS-1$

		Listener closeListener = new Listener() {
			public void handleEvent(Event event) {
				event.doit = close();
			}
		};

		// Initialize an exception handler.
		IExceptionHandler handler = ExceptionHandler.getInstance();

		try {
			// react to display close event by closing workbench nicely
			display.addListener(SWT.Close, closeListener);

			// install backstop to catch exceptions thrown out of event loop
			Workbench.setExceptionHandler(handler);

			// initialize workbench and restore or open one org.eclipse.jface.window
			boolean initOK = init(display);
			// always allow opening shells after this point
			openShellEnabled = true;			

			// drop the splash screen now that a workbench org.eclipse.jface.window is up
			Platform.endSplash();

			// let the advisor run its start up code
			if (initOK) {
				advisor.postStartup(); // may trigger a close/restart
			}

			if (initOK && runEventLoop) {
				// start eager plug-ins
				startPlugins();
//				addStartupRegistryListener();

				display.asyncExec(new Runnable() {
					public void run() {
						UIStats.end(UIStats.START_WORKBENCH, this, "Workbench"); //$NON-NLS-1$
					}
				});

//				getWorkbenchTestable().init(display, this);

				// the event loop
				runEventLoop(handler, display);
			}

		} catch (final Exception e) {
			if (!display.isDisposed()) {
				handler.handleException(e);
			} else {
				String msg = "Exception in Workbench.runUI after display was disposed"; //$NON-NLS-1$
				WorkbenchPlugin.log(msg, new Status(IStatus.ERROR,
						WorkbenchPlugin.PI_WORKBENCH, 1, msg, e));
			}
		} finally {
			// mandatory clean up

			// The runEventLoop flag may not have been cleared if an exception
			// occurred
			// Needs to be false to ensure PlatformUI.isWorkbenchRunning()
			// returns false.
			runEventLoop = false;

			if (!display.isDisposed()) {
				display.removeListener(SWT.Close, closeListener);
			}
		}

		// restart or exit based on returnCode
		return returnCode;
	}

	/*
	 * Runs an event loop for the workbench.
	 */
	private void runEventLoop(IExceptionHandler handler, Display display) {
		runEventLoop = true;
		while (runEventLoop) {
			try {
				if (!display.readAndDispatch()) {
					getAdvisor().eventLoopIdle(display);
				}
			} catch (Throwable t) {
				t.printStackTrace();
				handler.handleException(t);
			}
		}
	}

	/*
	 * Saves the current state of the workbench so it can be restored later on
	 */
	private IStatus saveState(IMemento memento) {
		MultiStatus result = new MultiStatus(PlatformUI.PLUGIN_ID, IStatus.OK,
				WorkbenchMessages.Workbench_problemsSaving, null);

		// Save the version number.
		memento.putString(IWorkbenchConstants.TAG_VERSION, VERSION_STRING[1]);

		// Save the workbench windows.
		IWorkbenchWindow[] windows = getWorkbenchWindows();
		for (int nX = 0; nX < windows.length; nX++) {
			WorkbenchWindow window = (WorkbenchWindow) windows[nX];
			IMemento childMem = memento
					.createChild(IWorkbenchConstants.TAG_WINDOW);
			result.merge(window.saveState(childMem));
		}
//		result.add(getEditorHistory().saveState(
//				memento.createChild(IWorkbenchConstants.TAG_MRU_LIST))); //$NON-NLS-1$
		return result;
	}

	/*
	 * Save the workbench UI in a persistence file.
	 */
	private boolean saveMementoToFile(XMLMemento memento) {
		// Save it to a file.
		// XXX: nobody currently checks the return value of this method.
		File stateFile = getWorkbenchStateFile();
		if (stateFile == null)
			return false;
		try {
			FileOutputStream stream = new FileOutputStream(stateFile);
			OutputStreamWriter writer = new OutputStreamWriter(stream, "utf-8"); //$NON-NLS-1$
			memento.save(writer);
			writer.close();
		} catch (IOException e) {
			stateFile.delete();
			ShowMessage.openError((Shell) null,
					WorkbenchMessages.SavingProblem,
					WorkbenchMessages.ProblemSavingState);
			return false;
		}

		// Success !
		return true;
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public IWorkbenchPage showPerspective(String perspectiveId,
			IWorkbenchWindow window) throws WorkbenchException {
		Assert.isNotNull(perspectiveId);

		// If the specified org.eclipse.jface.window has the requested perspective open, then the
		// org.eclipse.jface.window
		// is given focus and the perspective is shown. The page's input is
		// ignored.
		WorkbenchWindow win = (WorkbenchWindow) window;
		if (win != null) {
			WorkbenchPage page = win.getActiveWorkbenchPage();
			if (page != null) {
				IPerspectiveDescriptor perspectives[] = page
						.getOpenPerspectives();
				for (int i = 0; i < perspectives.length; i++) {
					IPerspectiveDescriptor persp = perspectives[i];
					if (perspectiveId.equals(persp.getId())) {
						win.getShell().open();
						page.setPerspective(persp);
						return page;
					}
				}
			}
		}

		// If another org.eclipse.jface.window that has the workspace root as input and the
		// requested
		// perpective open and active, then the org.eclipse.jface.window is given focus.
		IAdaptable input = getDefaultPageInput();
		IWorkbenchWindow[] windows = getWorkbenchWindows();
		for (int i = 0; i < windows.length; i++) {
			win = (WorkbenchWindow) windows[i];
			if (window != win) {
				WorkbenchPage page = win.getActiveWorkbenchPage();
				if (page != null) {
					boolean inputSame = false;
					if (input == null)
						inputSame = (page.getInput() == null);
					else
						inputSame = input.equals(page.getInput());
					if (inputSame) {
						Perspective persp = page.getActivePerspective();
						if (perspectiveId.equals(persp.getDesc().getId())) {
							Shell shell = win.getShell();
							shell.open();
							if (shell.getMinimized())
								shell.setMinimized(false);
							return page;
						}
					}
				}
			}
		}

		// Otherwise the requested perspective is opened and shown in the
		// specified
		// org.eclipse.jface.window or in a new org.eclipse.jface.window depending on the current user org.eclipse.jface.preference
		// for opening
		// perspectives, and that org.eclipse.jface.window is given focus.
		win = (WorkbenchWindow) window;
		if (win != null) {
			IPreferenceStore store = WorkbenchPlugin.getDefault()
					.getPreferenceStore();
			int mode = store.getInt(IPreferenceConstants.OPEN_PERSP_MODE);
			IWorkbenchPage page = win.getActiveWorkbenchPage();
			IPerspectiveDescriptor persp = null;
			if (page != null)
				persp = page.getPerspective();

			// Only open a new org.eclipse.jface.window if user org.eclipse.jface.preference is set and the org.eclipse.jface.window
			// has an active perspective.
			if (IPreferenceConstants.OPM_NEW_WINDOW == mode && persp != null) {
				IWorkbenchWindow newWindow = openWorkbenchWindow(perspectiveId,
						input);
				return newWindow.getActivePage();
			}

			IPerspectiveDescriptor desc = getPerspectiveRegistry()
					.findPerspectiveWithId(perspectiveId);
			if (desc == null)
				throw new WorkbenchException(
						NLS
								.bind(
										WorkbenchMessages.WorkbenchPage_ErrorCreatingPerspective,
										perspectiveId));
			win.getShell().open();
			if (page == null)
				page = win.openPage(perspectiveId, input);
			else
				page.setPerspective(desc);
			return page;
		}

		// Just throw an exception....
		throw new WorkbenchException(NLS
				.bind(WorkbenchMessages.Workbench_showPerspectiveError,
						perspectiveId));
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public IWorkbenchPage showPerspective(String perspectiveId,
			IWorkbenchWindow window, IAdaptable input)
			throws WorkbenchException {
		Assert.isNotNull(perspectiveId);

		// If the specified org.eclipse.jface.window has the requested perspective open and the
		// same requested
		// input, then the org.eclipse.jface.window is given focus and the perspective is shown.
		boolean inputSameAsWindow = false;
		WorkbenchWindow win = (WorkbenchWindow) window;
		if (win != null) {
			WorkbenchPage page = win.getActiveWorkbenchPage();
			if (page != null) {
				boolean inputSame = false;
				if (input == null)
					inputSame = (page.getInput() == null);
				else
					inputSame = input.equals(page.getInput());
				if (inputSame) {
					inputSameAsWindow = true;
					IPerspectiveDescriptor perspectives[] = page
							.getOpenPerspectives();
					for (int i = 0; i < perspectives.length; i++) {
						IPerspectiveDescriptor persp = perspectives[i];
						if (perspectiveId.equals(persp.getId())) {
							win.getShell().open();
							page.setPerspective(persp);
							return page;
						}
					}
				}
			}
		}

		// If another org.eclipse.jface.window has the requested input and the requested
		// perpective open and active, then that org.eclipse.jface.window is given focus.
		IWorkbenchWindow[] windows = getWorkbenchWindows();
		for (int i = 0; i < windows.length; i++) {
			win = (WorkbenchWindow) windows[i];
			if (window != win) {
				WorkbenchPage page = win.getActiveWorkbenchPage();
				if (page != null) {
					boolean inputSame = false;
					if (input == null)
						inputSame = (page.getInput() == null);
					else
						inputSame = input.equals(page.getInput());
					if (inputSame) {
						Perspective persp = page.getActivePerspective();
						if (perspectiveId.equals(persp.getDesc().getId())) {
							win.getShell().open();
							return page;
						}
					}
				}
			}
		}

		// If the specified org.eclipse.jface.window has the same requested input but not the
		// requested
		// perspective, then the org.eclipse.jface.window is given focus and the perspective is
		// opened and shown
		// on condition that the user org.eclipse.jface.preference is not to open perspectives in
		// a new org.eclipse.jface.window.
		win = (WorkbenchWindow) window;
		if (inputSameAsWindow && win != null) {
			IPreferenceStore store = WorkbenchPlugin.getDefault()
					.getPreferenceStore();
			int mode = store.getInt(IPreferenceConstants.OPEN_PERSP_MODE);

			if (IPreferenceConstants.OPM_NEW_WINDOW != mode) {
				IWorkbenchPage page = win.getActiveWorkbenchPage();
				IPerspectiveDescriptor desc = getPerspectiveRegistry()
						.findPerspectiveWithId(perspectiveId);
				if (desc == null)
					throw new WorkbenchException(
							NLS
									.bind(
											WorkbenchMessages.WorkbenchPage_ErrorCreatingPerspective,
											perspectiveId));
				win.getShell().open();
				if (page == null)
					page = win.openPage(perspectiveId, input);
				else
					page.setPerspective(desc);
				return page;
			}
		}

		// If the specified org.eclipse.jface.window has no active perspective, then open the
		// requested perspective and show the specified org.eclipse.jface.window.
		if (win != null) {
			IWorkbenchPage page = win.getActiveWorkbenchPage();
			IPerspectiveDescriptor persp = null;
			if (page != null)
				persp = page.getPerspective();
			if (persp == null) {
				IPerspectiveDescriptor desc = getPerspectiveRegistry()
						.findPerspectiveWithId(perspectiveId);
				if (desc == null)
					throw new WorkbenchException(
							NLS
									.bind(
											WorkbenchMessages.WorkbenchPage_ErrorCreatingPerspective,
											perspectiveId));
				win.getShell().open();
				if (page == null)
					page = win.openPage(perspectiveId, input);
				else
					page.setPerspective(desc);
				return page;
			}
		}

		// Otherwise the requested perspective is opened and shown in a new
		// org.eclipse.jface.window, and the
		// org.eclipse.jface.window is given focus.
		IWorkbenchWindow newWindow = openWorkbenchWindow(perspectiveId, input);
		return newWindow.getActivePage();
	}

	/*
	 * Shuts down the application.
	 */
	private void shutdown() {
		// shutdown application-specific portions first
		advisor.postShutdown();

		// for dynamic UI
		Platform.getExtensionRegistry().removeRegistryChangeListener(
				extensionEventHandler);
		Platform.getExtensionRegistry().removeRegistryChangeListener(
				startupRegistryListener);
//		workbenchActivitySupport.dispose();
//		WorkbenchHelpSystem.disposeIfNecessary();
		// shutdown the rest of the workbench
		WorkbenchColors.shutdown();
//		activityHelper.shutdown();
		uninitializeImages();
		if (WorkbenchPlugin.getDefault() != null) {
			WorkbenchPlugin.getDefault().reset();
		}
//		WorkbenchThemeManager.getInstance().dispose();
//		PropertyPageContributorManager.getManager().dispose();
//		ObjectActionContributorManager.getManager().dispose();
		if (tracker != null)
			tracker.close();
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
//	public IDecoratorManager getDecoratorManager() {
//		return WorkbenchPlugin.getDefault().getDecoratorManager();
//	}

	/*
	 * Returns the workbench org.eclipse.jface.window which was last known being the active one,
	 * or <code> null </code> .
	 */
	private WorkbenchWindow getActivatedWindow() {
		if (activatedWindow != null) {
			Shell shell = activatedWindow.getShell();
			if (shell != null && !shell.isDisposed()) {
				return activatedWindow;
			}
		}

		return null;
	}

	/*
	 * Sets the workbench org.eclipse.jface.window which was last known being the active one, or
	 * <code> null </code> .
	 */
	/* package */
	void setActivatedWindow(WorkbenchWindow window) {
		activatedWindow = window;
	}

	/**
	 * Returns the unique object that applications use to configure the
	 * workbench.
	 * <p>
	 * IMPORTANT This method is declared package-private to prevent regular
	 * plug-ins from downcasting IWorkbench to Workbench and getting hold of the
	 * workbench configurer that would allow them to tamper with the workbench.
	 * The workbench configurer is available only to the application.
	 * </p>
	 */
	/* package */
	WorkbenchConfigurer getWorkbenchConfigurer() {
		if (workbenchConfigurer == null) {
			workbenchConfigurer = new WorkbenchConfigurer();
		}
		return workbenchConfigurer;
	}

	/**
	 * Returns the workbench advisor that created this workbench.
	 * <p>
	 * IMPORTANT This method is declared package-private to prevent regular
	 * plug-ins from downcasting IWorkbench to Workbench and getting hold of the
	 * workbench advisor that would allow them to tamper with the workbench. The
	 * workbench advisor is internal to the application.
	 * </p>
	 */
	/* package */
	WorkbenchAdvisor getAdvisor() {
		return advisor;
	}

	/*
	 * (non-Javadoc) Method declared on IWorkbench.
	 */
	public Display getDisplay() {
		return display;
	}

	/**
	 * Returns the default perspective id, which may be <code>null</code>.
	 * 
	 * @return the default perspective id, or <code>null</code>
	 */
	public String getDefaultPerspectiveId() {
		return getAdvisor().getInitialWindowPerspectiveId();
	}

	/**
	 * Returns the default workbench org.eclipse.jface.window page input.
	 * 
	 * @return the default org.eclipse.jface.window page input or <code>null</code> if none
	 */
	public IAdaptable getDefaultPageInput() {
		return getAdvisor().getDefaultPageInput();
	}

	/**
	 * Returns the id of the org.eclipse.jface.preference page that should be presented most
	 * prominently.
	 * 
	 * @return the id of the org.eclipse.jface.preference page, or <code>null</code> if none
	 */
	public String getMainPreferencePageId() {
		String id = getAdvisor().getMainPreferencePageId();
		return id;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.IWorkbench
	 * @since 3.0
	 */
	public IElementFactory getElementFactory(String factoryId) {
		Assert.isNotNull(factoryId);
		return WorkbenchPlugin.getDefault().getElementFactory(factoryId);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.IWorkbench#getProgressService()
	 */
//	public IProgressService getProgressService() {
//		return ProgressManager.getInstance();
//	}

//	private WorkbenchActivitySupport workbenchActivitySupport;

	private WorkbenchCommandSupport workbenchCommandSupport;

//	private WorkbenchContextSupport workbenchContextSupport;

	/**
	 * The single instance of the binding manager used by the workbench. This is
	 * initialized in <code>Workbench.init(Display)</code> and then never
	 * changed. This value will only be <code>null</code> if the
	 * initialization call has not yet completed.
	 * 
	 * @since 3.1
	 */
//	private BindingManager bindingManager;

	/**
	 * The single instance of the command manager used by the workbench. This is
	 * initialized in <code>Workbench.init(Display)</code> and then never
	 * changed. This value will only be <code>null</code> if the
	 * initialization call has not yet completed.
	 * 
	 * @since 3.1
	 */
	private CommandManager commandManager;

	/**
	 * The single instance of the context manager used by the workbench. This is
	 * initialized in <code>Workbench.init(Display)</code> and then never
	 * changed. This value will only be <code>null</code> if the
	 * initialization call has not yet completed.
	 * 
	 * @since 3.1
	 */
	private ContextManager contextManager;

//	public IWorkbenchActivitySupport getActivitySupport() {
//		return workbenchActivitySupport;
//	}

	public IWorkbenchCommandSupport getCommandSupport() {
		return workbenchCommandSupport;
	}

//	public IWorkbenchContextSupport getContextSupport() {
//		return workbenchContextSupport;
//	}

	private final IWindowListener windowListener = new IWindowListener() {

		public void windowActivated(IWorkbenchWindow window) {
			updateActiveWorkbenchWindowMenuManager(true);
		}

		public void windowClosed(IWorkbenchWindow window) {
			updateActiveWorkbenchWindowMenuManager(true);
		}

		public void windowDeactivated(IWorkbenchWindow window) {
			updateActiveWorkbenchWindowMenuManager(true);
		}

		public void windowOpened(IWorkbenchWindow window) {
			updateActiveWorkbenchWindowMenuManager(true);
		}
	};

	private void updateActiveWorkbenchWindowMenuManager(boolean textOnly) {
		final IWorkbenchWindow workbenchWindow = getActiveWorkbenchWindow();

		if (workbenchWindow instanceof WorkbenchWindow) {
			final WorkbenchWindow window = (WorkbenchWindow) workbenchWindow;
			if (window.isClosing()) {
				return;
			}

//			final MenuManager menuManager = org.eclipse.jface.window.getMenuManager();

//			if (textOnly)
//				menuManager.update(IAction.TEXT);
//			else
//				menuManager.updateAll(true);
		}
	}

//	private ActivityPersistanceHelper activityHelper;

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.IWorkbench#getIntroManager()
	 */
/*	public IIntroManager getIntroManager() {
		return getWorkbenchIntroManager();
	}
*/
	/**
	 * @return the workbench intro manager
	 * @since 3.0
	 */
	/* package */
/*	WorkbenchIntroManager getWorkbenchIntroManager() {
		if (introManager == null) {
			introManager = new WorkbenchIntroManager(this);
		}
		return introManager;
	}

	private WorkbenchIntroManager introManager;
*/
	/**
	 * @return the intro extension for this workbench.
	 * 
	 * @since 3.0
	 */
/*	public IntroDescriptor getIntroDescriptor() {
		return introDescriptor;
	}
*/
	/**
	 * This method exists as a test hook. This method should <strong>NEVER</strong>
	 * be called by clients.
	 * 
	 * @param descriptor
	 *            The intro descriptor to use.
	 * @since 3.0
	 */
/*	public void setIntroDescriptor(IntroDescriptor descriptor) {
		if (getIntroManager().getIntro() != null) {
			getIntroManager().closeIntro(getIntroManager().getIntro());
		}
		introDescriptor = descriptor;
	}
*/
	/**
	 * The descriptor for the intro extension that is valid for this workspace,
	 * <code>null</code> if none.
	 */
//	private IntroDescriptor introDescriptor;

	private IExtensionTracker tracker;

	private IRegistryChangeListener startupRegistryListener = new IRegistryChangeListener() {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.core.runtime.IRegistryChangeListener#registryChanged(org.eclipse.core.runtime.IRegistryChangeEvent)
		 */
		public void registryChanged(IRegistryChangeEvent event) {
			final IExtensionDelta[] deltas = event.getExtensionDeltas(
					PlatformUI.PLUGIN_ID, IWorkbenchConstants.PL_STARTUP);
			if (deltas.length == 0)
				return;
			final String disabledPlugins = getPreferenceStore().getString(
					IPreferenceConstants.PLUGINS_NOT_ACTIVATED_ON_STARTUP);
			Runnable runnable = new Runnable() {
				public void run() {
					for (int i = 0; i < deltas.length; i++) {
						IExtension extension = deltas[i].getExtension();
						if (deltas[i].getKind() == IExtensionDelta.REMOVED)
							continue;

						// if the plugin is not in the set of disabled plugins,
						// then
						// execute the code to start it
						if (disabledPlugins.indexOf(extension.getNamespace()) == -1)
							Platform.run(new EarlyStartupRunnable(extension));
					}
				}
			};
			Thread thread = new Thread(runnable);
			thread.start();
		}
	};

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.IWorkbench#getThemeManager()
	 */
//	public IThemeManager getThemeManager() {
//		return WorkbenchThemeManager.getInstance();
//	}

	/**
	 * Returns <code>true</code> if the workbench is running,
	 * <code>false</code> if it has been terminated.
	 * 
	 * @return <code>true</code> if the workbench is running,
	 *         <code>false</code> if it has been terminated.
	 */
	public boolean isRunning() {
		return runEventLoop;
	}

	/**
	 * Return the presentation ID specified by the org.eclipse.jface.preference or the default ID
	 * if undefined.
	 * 
	 * @return the presentation ID
	 * @see IWorkbenchPreferenceConstants#PRESENTATION_FACTORY_ID
	 */
	public String getPresentationId() {
		String factoryId = PrefUtil.getAPIPreferenceStore().getString(
				IWorkbenchPreferenceConstants.PRESENTATION_FACTORY_ID);

		// Workaround for bug 58975 - New org.eclipse.jface.preference mechanism does not properly
		// initialize defaults
		// Ensure that the UI plugin has started too.
		if (factoryId == null || factoryId.equals("")) { //$NON-NLS-1$
			factoryId = "org.eclipse.ui.presentations.default"; //$NON-NLS-1$
		}
		return factoryId;
	}

	/**
	 * <p>
	 * Indicates the start of a large update within the workbench. This is used
	 * to disable CPU-intensive, change-sensitive services that were temporarily
	 * disabled in the midst of large changes. This method should always be
	 * called in tandem with <code>largeUpdateEnd</code>, and the event loop
	 * should not be allowed to spin before that method is called.
	 * </p>
	 * <p>
	 * Important: always use with <code>largeUpdateEnd</code>!
	 * </p>
	 */
	public final void largeUpdateStart() {
		if (largeUpdates++ == 0) {
			// TODO Consider whether these lines still need to be here.
			//workbenchCommandSupport.setProcessing(false);
			//workbenchContextSupport.setProcessing(false);
			
			final IWorkbenchWindow[] windows = getWorkbenchWindows();
			for (int i = 0; i < windows.length; i++) {
				IWorkbenchWindow window = windows[i];
				if (window instanceof WorkbenchWindow) {
					((WorkbenchWindow) window).largeUpdateStart();
				}
			}
		}
	}

	/**
	 * <p>
	 * Indicates the end of a large update within the workbench. This is used to
	 * re-enable services that were temporarily disabled in the midst of large
	 * changes. This method should always be called in tandem with
	 * <code>largeUpdateStart</code>, and the event loop should not be
	 * allowed to spin before this method is called.
	 * </p>
	 * <p>
	 * Important: always protect this call by using <code>finally</code>!
	 * </p>
	 */
	public final void largeUpdateEnd() {
		if (--largeUpdates == 0) {
			// TODO Consider whether these lines still need to be here.
			//workbenchCommandSupport.setProcessing(true);
			//workbenchContextSupport.setProcessing(true);

			// Perform org.eclipse.jface.window-specific blocking.
			final IWorkbenchWindow[] windows = getWorkbenchWindows();
			for (int i = 0; i < windows.length; i++) {
				IWorkbenchWindow window = windows[i];
				if (window instanceof WorkbenchWindow) {
					((WorkbenchWindow) window).largeUpdateEnd();
				}
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.IWorkbench#getExtensionTracker()
	 */
	public IExtensionTracker getExtensionTracker() {
		if (tracker == null) {
			tracker = new UIExtensionTracker(getDisplay());
		}
		return tracker;
	}

	/**
	 * Adds the listener that handles startup plugins
	 * 
	 * @since 3.1
	 */
	private void addStartupRegistryListener() {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		registry.addRegistryChangeListener(startupRegistryListener);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.IWorkbench#getHelpSystem()
	 */
//	public IWorkbenchHelpSystem getHelpSystem() {
//		return WorkbenchHelpSystem.getInstance();
//	}

	public IViewRegistry getViewRegistry() {
		return WorkbenchPlugin.getDefault().getViewRegistry();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.IWorkbench#getNewWizardRegistry()
	 */
//	public IWizardRegistry getNewWizardRegistry() {
//		return WorkbenchPlugin.getDefault().getNewWizardRegistry();
//	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.IWorkbench#getImportWizardRegistry()
	 */
//	public IWizardRegistry getImportWizardRegistry() {
//		return WorkbenchPlugin.getDefault().getImportWizardRegistry();
//	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.IWorkbench#getExportWizardRegistry()
	 */
//	public IWizardRegistry getExportWizardRegistry() {
//		return WorkbenchPlugin.getDefault().getExportWizardRegistry();
//	}

	public final Object getAdapter(final Class key) {
		return services.get(key);
	}
	
	/**
	 * This interface defines a Exception Handler which can be set as a global
	 * handler and will be called if an exception happens in the event loop.
	 */
	public static interface IExceptionHandler {
		/**
		 * Handle the exception.
		 * 
		 * @param t
		 *            The exception that occured.
		 */
		public void handleException(Throwable t);
	}

	/**
	 * Defines a default exception handler.
	 */
	private static class DefaultExceptionHandler implements IExceptionHandler {
		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.window.Window.IExceptionHandler#handleException(java.lang.Throwable)
		 */
		public void handleException(Throwable t) {
			if (t instanceof ThreadDeath)
				// Don't catch ThreadDeath as this is a normal occurrence when
				// the thread dies
				throw (ThreadDeath) t;
			// Try to keep running.
			t.printStackTrace();
		}
	}

	/**
	 * The exception handler for this application.
	 */
	private static IExceptionHandler exceptionHandler = new DefaultExceptionHandler();
	
	/**
	 * Sets the exception handler for this application.
	 * <p>
	 * Note that only one handler may be set. Other calls to this method will be
	 * ignored.
	 * <p>
	 * 
	 * @param handler
	 *            the exception handler for the application.
	 */
	public static void setExceptionHandler(IExceptionHandler handler) {
		if (exceptionHandler instanceof DefaultExceptionHandler)
			exceptionHandler = handler;
	}

	
	
///// new codes for Headless workbench, run applicaions in the top level shell/////
	boolean isOpenShellEnabled() {
		  return openShellEnabled;
	}
	
	public int getReturnCode() {
		return this.returnCode;
	}
	
}
