/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ui.internal.dialogs;

import java.util.Iterator;

import org.eclipse.jface.preference.IPreferenceNode;
import org.eclipse.jface.preference.IPreferencePage;
import org.eclipse.jface.preference.PreferenceDialog;
import org.eclipse.jface.preference.PreferenceManager;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.jface.preference.PreferenceNode;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.internal.misc.Assert;


/**
 * Prefence dialog for the workbench including the ability to load/save
 * preferences.
 */
public class WorkbenchPreferenceDialog extends PreferenceDialog{
	/**
	 * There can only ever be one instance of the workbench's preference dialog.
	 * This keeps a handle on this instance, so that attempts to create a second
	 * dialog should just fail (or return the original instance).
	 * 
	 * @since 3.1
	 */
	private static WorkbenchPreferenceDialog instance = null;
	
	private String rootNodeToDisplayId;

	/**
	 * Creates a workbench preference dialog for a particular root page. It
	 * is the responsibility of the caller to then call <code>open()</code>.
	 * The call to <code>open()</code> will not return until the dialog
	 * closes, so this is the last chance to manipulate the dialog.
	 * 
	 * @param shell
	 * 			The Shell to parent the dialog off of if it is not
	 * 			already created. May be <code>null</code>
	 * 			in which case the active workbench window will be used
	 * 			if available.
	 * @param preferencePageId
	 *            The identifier of the preference page to open; may be
	 *            <code>null</code>. If it is <code>null</code>, then the
	 *            preference page is not selected or modified in any way.
	 * @return The selected preference page.
	 * @since 3.1
	 */
	public static final WorkbenchPreferenceDialog createDialogOn(Shell shell, 
			final String preferencePageId) {
		
		final WorkbenchPreferenceDialog dialog;
		
		if (instance == null) {
			/*
			 * There is no existing preference dialog, so open a new one with
			 * the given selected page.
			 */

			Shell parentShell = shell;
			if (parentShell == null) {
				// Determine a decent parent shell.
				final IWorkbench workbench = PlatformUI.getWorkbench();
				final IWorkbenchWindow workbenchWindow = workbench.getActiveWorkbenchWindow();
				if (workbenchWindow != null) {
					parentShell = workbenchWindow.getShell();
				} 
			}

			// Create the dialog
			
			WorkbenchPreferenceManager preferenceManager = (WorkbenchPreferenceManager)PlatformUI.
				getWorkbench().getPreferenceManager();
			dialog = new WorkbenchPreferenceDialog(parentShell, preferenceManager);
			dialog.setRootNodeToDisplay(preferencePageId);
			
		} else {
			/*
			 * There is an existing preference dialog, so let's just select the
			 * given preference page.
			 */
			dialog = instance;

//			if (preferencePageId != null) {
//				dialog.setCurrentPageId(preferencePageId);
//			}

		}
		
		return dialog;
	}
	
	
	/**
	 * Creates a new preference dialog under the control of the given preference
	 * manager.
	 * 
	 * @param parentShell
	 *            the parent shell
	 * @param manager
	 *            the preference manager
	 */
	public WorkbenchPreferenceDialog(Shell parentShell, PreferenceManager manager) {
		super(parentShell, manager);
		Assert.isTrue((instance == null),
				"There cannot be two preference dialogs at once in the workbench."); //$NON-NLS-1$
		instance = this;
		
	}

	/*
	 *  (non-Javadoc)
	 *  Override default behavior in order to display only the pages
	 *  for a given root node
	 *  
	 * @see org.eclipse.jface.preference.PreferenceDialog#createContents(org.eclipse.jface.preference.PreferenceDialog)
	 */
	protected Control createContents(final PreferenceDialog parent) {
		
	    Composite parentDialog = null;

	    //		System.out.println("here in create contents time:"+ counter++);
//		Display.getDefault().syncExec(new Runnable() {
//			public void run() {
				
				Iterator iterator = preferenceManager.getRootElements().iterator();
				while (iterator.hasNext()) {
					PreferenceNode next = (PreferenceNode) iterator.next();
					if (rootNodeToDisplayId == null ||
						next.getId().equals(rootNodeToDisplayId)) {
						parentDialog = createSubPages(next);
					}
				}	
				return parentDialog;
//			}
//		});
	}

	
	public void close() {
		instance = null;
		 super.close();
	}
	
	public void setRootNodeToDisplay(String nodeId) {
		rootNodeToDisplayId = nodeId;
	}
}
