/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.ui.internal.misc;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;

/**
 * convenient class to open messagebox.
 * 
 */
public class ShowMessage {
	/**
	 * Constant for a dialog with no image (value 0).
	 */
	public final static int NONE = 0;

	/**
	 * Constant for a dialog with an error image (value 1).
	 */
	public final static int ERROR = 1;

	/**
	 * Constant for a dialog with an info image (value 2).
	 */
	public final static int INFORMATION = 2;

	/**
	 * Constant for a dialog with a question image (value 3).
	 */
	public final static int QUESTION = 3;

	/**
	 * Constant for a dialog with a warning image (value 4).
	 */
	public final static int WARNING = 4;

	/**
	 * The label for OK buttons.
	 */
	public String OK_LABEL = JFaceResources.getString("ok"); //$NON-NLS-1$

	/**
	 * The label for cancel buttons.
	 */
	public String CANCEL_LABEL = JFaceResources.getString("cancel"); //$NON-NLS-1$

	
	/**
     * The nesting indent.
     */
    private static final String NESTING_INDENT = "  "; //$NON-NLS-1$
    static String  originalMessage;
    /**
	 * @param parent
	 * @param title
	 * @param message
	 * @return the id of the button that have been selected
	 */
	public static int openInformation(Shell parent, String title, String message) {
		return openBox(parent, title, message, SWT.ICON_INFORMATION | SWT.OK);
	}

	/**
	 * @param parent
	 * @param title
	 * @param message
	 * @return the id of the button that have been selected
	 */
	public static int openError(Shell parent, String title, String message) {
		return openBox(parent, title, message, SWT.ICON_ERROR | SWT.OK);
	}

	/**
	 * @param parent
	 * @param title
	 * @param message
	 * @return true if Yes button is chosen, return false if NO button is chosen
	 */
	public static boolean openQuestion(Shell parent, String title,
			String message) {
		return openBox(parent, title, message, SWT.ICON_ERROR | SWT.YES
				| SWT.NO) == SWT.YES;
	}

	/**
	 * @param parent
	 * @param title
	 * @param message
	 * @return the id of the button that have been selected
	 */
	public static int openWarning(Shell parent, String title, String message) {
		return openBox(parent, title, message, SWT.ICON_WARNING | SWT.OK);
	}

	/**
	 * @param parent
	 * @param title
	 * @param message
	 * @param style
	 * @return the id of the button that have been selected
	 */
	public static int openBox(Shell parent, String title, String message,
			int style) {
		Shell shell = null;
		if (parent != null) {
			shell = parent;
		} else {
			Display display = new Display();
			shell = new Shell(display);
		}

		MessageBox mb = new MessageBox(shell, style);
		mb.setText(title);
		mb.setMessage(message);
		return mb.open();
	}

	
	public static void printError(String message,
            IStatus status){
		 originalMessage = message == null ? status.getMessage()
                : JFaceResources
                        .format(
                                "Reason", new Object[] { message, status.getMessage() }); //$NON-NLS-1$
    
        
		printError(status,0,false);
	}
	
	/**
	 * @param buildingStatus
	 * @param originalMessage
	 * @param nesting
	 * @param includeStatus
	 */
	public static void printError(IStatus buildingStatus, int nesting, boolean includeStatus) {
		
		Throwable t = buildingStatus.getException();
        boolean isCoreException= t instanceof CoreException;
        boolean incrementNesting= false;
        
       	if (includeStatus) {
	        StringBuffer sb = new StringBuffer();
	        for (int i = 0; i < nesting; i++) {
	            sb.append(NESTING_INDENT); //$NON-NLS-1$
	        }
	        String message = buildingStatus.getMessage();
            sb.append(message);
	        System.out.println(sb.toString());
	        incrementNesting= true;
       	}
        	
        if (!isCoreException && t != null) {
        	// Include low-level exception message
	        StringBuffer sb = new StringBuffer();
	        for (int i = 0; i < nesting; i++) {
	            sb.append(NESTING_INDENT); //$NON-NLS-1$
	        }
	        String message = t.getLocalizedMessage();
	        if (message == null)
	        	message = t.toString();
	        	
	        sb.append(message);
	        System.out.println(sb.toString());
	        incrementNesting= true;
        }
        
        if (incrementNesting) 
	        nesting++;
        
        // Look for a nested core exception
        if (isCoreException) {
            CoreException ce = (CoreException)t;
            IStatus eStatus = ce.getStatus();
            // Only print the exception message if it is not contained in the parent message
            if (originalMessage == null || originalMessage.indexOf(eStatus.getMessage()) == -1) {
                printError( eStatus, nesting, true);
            }
        }

        
        // Look for child status
        IStatus[] children = buildingStatus.getChildren();
        for (int i = 0; i < children.length; i++) {
        	printError(children[i], nesting, true);
        }
	}
	
    /**
     * Populate the list with the messages from the given status. Traverse the
     * children of the status deeply and also traverse CoreExceptions that appear
     * in the status.
     * @param listToPopulate the list to populate
     * @param buildingStatus the status being displayed
     * @param nesting the nesting level (increases one level for each level of children)
     * @param includeStatus whether to include the buildingStatus in the display or
     * just its children
     */
    
	

}
