/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *
 * The Javadoc comments are taken from Crimson 1.1 developed by The Apache Software Foundation
 * and are made available under the terms and conditions of the Apache Software License 1.1
 * which accompanies this distribution and is available at
 * http://www.apache.org/licenses/LICENSE-1.1.
 *
 * All other portions of this file including the functional Java code, are made available under 
 * the terms and conditions of the EPL.
 *******************************************************************************/ 
package javax.xml.parsers;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.util.Hashtable;

import org.eclipse.ercp.xml.io.IOSpecificCall;
import org.eclipse.ercp.xml.io.PropertyFileReader;



/**
 * Retrieve a JAXP Factory...
 */
class FactoryFinder {
	private static Hashtable Found = new Hashtable(2);
/**
 * Create an instance of a class
 */
private static Object newInstance(String factoryId, String className) throws FactoryConfigurationError {
    try {
    	Class clazz = Class.forName(className);
    	Found.put(factoryId, clazz);
        return clazz.newInstance(); 
    } catch (ClassNotFoundException x) {
        throw new FactoryConfigurationError();
    } catch (IllegalAccessException e) {
        throw new FactoryConfigurationError();
    } catch (InstantiationException e) {
		throw new FactoryConfigurationError();
    }
}

/**
 * Finds the implementation Class object in the specified order.
 * 
 * @return Class object of factory, never null
 *
 * @param factoryId - Name of the factory to find
 * @param defaultFactoryClassName - Implementation class name, if nothing else is found.
 *
 * @exception FactoryConfigurationError
 */
static Object find(String factoryId, String defaultFactoryClassName) throws FactoryConfigurationError {
	Class clazz = (Class)(Found.get(factoryId));
	String factoryClassName;
    // Check the system property first
    factoryClassName = System.getProperty(factoryId);
    if(factoryClassName != null) {
    	if (clazz == null) return newInstance(factoryId, factoryClassName);
    	// Check if the user didn't change the system property...
    	if (clazz.getName().equals(factoryClassName)) {
    		try {
    			return clazz.newInstance();
    		} catch (IllegalAccessException e) {
    			throw new FactoryConfigurationError();
    		} catch (InstantiationException e) {
    			throw new FactoryConfigurationError();
    		}
    	}
    }
    // We assume that the user didn't change the $java.home/lib/xml.properties
    // or services in CLASSPATH during the current session!
	if (clazz!= null) {
		try {
			return clazz.newInstance();
		} catch (IllegalAccessException e) {
			throw new FactoryConfigurationError();
		} catch (InstantiationException e) {
			throw new FactoryConfigurationError();
		}
	}

    // try to read from $java.home/lib/xml.properties
    try {
    	String temp = System.getProperty("file.separator"); //$NON-NLS-1$
    	char separatorChar = (temp == null)?'\\':temp.charAt(0);
        String javah = System.getProperty("java.home"); //$NON-NLS-1$
        javah = javah.replace(separatorChar, '/');
        String configFile = javah + '/' + "lib" + '/' + "jaxp.properties"; //$NON-NLS-2$ //$NON-NLS-1$
		InputStream in = IOSpecificCall.getDefault().openFileInputStream(configFile);
        Hashtable properties = new Hashtable();
        new PropertyFileReader().read(in, properties);
        in.close();
		factoryClassName = (String)(properties.get(factoryId));
		return newInstance(factoryId, factoryClassName);
    } catch(IOException e) {
    	// Try the next one...
    }

    // try to find services in CLASSPATH
    String serviceId = "META-INF/services/" + factoryId; //$NON-NLS-1$
    try {
        InputStream is = Object.class.getResourceAsStream(serviceId);
    
        if (is != null) {
           	BufferedReader reader = new BufferedReader(new InputStreamReader(is, "UTF-8"));
            factoryClassName = reader.readLine();
            reader.close();

            if (factoryClassName != null && !("".equals(factoryClassName))) {
                return newInstance(factoryId, factoryClassName);
            }
        }
    } catch (UnsupportedEncodingException ex) {
    	// NOP
    } catch (IOException ex) {
    	// NOP
    }

    if (defaultFactoryClassName == null) throw new FactoryConfigurationError();

    return newInstance(factoryId, defaultFactoryClassName);
}


}
