/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *
 * The Javadoc comments are taken from Crimson 1.1 developed by The Apache Software Foundation
 * and are made available under the terms and conditions of the Apache Software License 1.1
 * which accompanies this distribution and is available at
 * http://www.apache.org/licenses/LICENSE-1.1.
 *
 * All other portions of this file including the functional Java code, are made available under 
 * the terms and conditions of the EPL.
 *******************************************************************************/ 
package javax.xml.parsers;

import java.io.*;

import org.eclipse.ercp.xml.parser.EXmlMsg;
import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;


/**
 * Defines the API that represents a simple SAX parser.
 * An instance of this class can be obtained from the
 * SAXParserFactory.newSAXParser() method. Once an instance
 * of this class is obtained, XML can be parsed from an InputStream<p>
 * As the content is parsed by the underlying parser, methods of the
 * given DefaultHandler are called.<p>
 * An implementation of SAXParser is NOT guaranteed to behave as per
 * the specification if it is used concurrently by two or more threads.
 * It is recommended to have one instance of the SAXParser per thread or
 * it is upto the application to make sure about the use of SAXParser from
 * more than one thread. 
 */
public abstract class SAXParser {
	
/**
 * Constructor for SAXParser.
 */
protected SAXParser() {
	super();
}
/**
 * Parse the content of the given InputStream instance as
 * XML using the specified DefaultHandler.
 * 
 * @param is - InputStream containing the content to be parsed.
 * @param dh - The SAX DefaultHandler to use.
 * 
 * @exception java.io.IOException - If any IO errors occur.
 * @exception IllegalArgumentException - If the given InputStream is null.
 * @exception SAXException - If the underlying parser throws a SAXException while parsing.
 */
public abstract void parse(InputSource is, DefaultHandler dh) throws SAXException,IOException;
/**
 * Indicates whether or not this parser is configured to understand namespaces.
 * 
 * @return true if this parser is configured to understand namespaces; false otherwise.
 */
public abstract boolean isNamespaceAware();
/**
 * Indicates whether or not this parser is configured to validate XML documents.
 *
 * @return true if this parser is configured to validate XML documents; false otherwise.
 */
public abstract boolean isValidating();
/**
 * Returns the {@link org.xml.sax.XMLReader} that is encapsulated by the
 * implementation of this class.
 *
 * @return The XMLReader that is encapsulated by the
 *         implementation of this class.
 */
public abstract XMLReader getXMLReader() throws SAXException;
/**
 * Sets the particular property in the underlying implementation of
 * {@link org.xml.sax.XMLReader}.
 * A list of the core features and properties can be found at
 * <a href="http://www.megginson.com/SAX/Java/features.html"> http://www.megginson.com/SAX/Java/features.html </a>
 *
 * @param name The name of the property to be set.
 * @param value The value of the property to be set.
 * @exception SAXNotRecognizedException When the underlying XMLReader does
 *            not recognize the property name.
 *
 * @exception SAXNotSupportedException When the underlying XMLReader
 *            recognizes the property name but doesn't support the
 *            property.
 *
 * @see org.xml.sax.XMLReader#setProperty
 */
public abstract void setProperty(String name, Object value)
    throws SAXNotRecognizedException, SAXNotSupportedException;

/**
 *
 * Returns the particular property requested for in the underlying
 * implementation of {@link org.xml.sax.XMLReader}.
 *
 * @param name The name of the property to be retrieved.
 * @return Value of the requested property.
 *
 * @exception SAXNotRecognizedException When the underlying XMLReader does
 *            not recognize the property name.
 *
 * @exception SAXNotSupportedException When the underlying XMLReader
 *            recognizes the property name but doesn't support the
 *            property.
 *
 * @see org.xml.sax.XMLReader#getProperty
 */
public abstract Object getProperty(String name)
    throws SAXNotRecognizedException, SAXNotSupportedException;
/**
 * Parse the content of the file specified as XML using the
 * specified {@link org.xml.sax.helpers.DefaultHandler}.
 *
 * @param f The file containing the XML to parse
 * @param dh The SAX DefaultHandler to use.
 * @exception IOException If any IO errors occur.
 * @exception IllegalArgumentException If the File object is null.
 * @exception SAXException If the underlying parser throws a
 * SAXException while parsing.
 */

public void parse(File file, DefaultHandler dh) throws SAXException, IOException {
    if (file == null) {
        throw new IllegalArgumentException(EXmlMsg.getDefault().getString(EXmlMsg.CANNOT_BE_NULL, "File")); //$NON-NLS-1$ //$NON-NLS-2$
    }
    InputStream is = new FileInputStream(file);
    parse(is, dh);
    is.close();
}
/**
 * Parse the content of the given {@link java.io.InputStream}
 * instance as XML using the specified
 * {@link org.xml.sax.helpers.DefaultHandler}.
 *
 * @param is InputStream containing the content to be parsed.
 * @param dh The SAX DefaultHandler to use.
 * @exception IOException If any IO errors occur.
 * @exception IllegalArgumentException If the given InputStream is null.
 * @exception SAXException If the underlying parser throws a
 * SAXException while parsing.
 */

public void parse(InputStream is, DefaultHandler dh) throws SAXException, IOException {
    if (is == null) {
        throw new IllegalArgumentException(EXmlMsg.getDefault().getString(EXmlMsg.CANNOT_BE_NULL, "InpuStream")); //$NON-NLS-1$ //$NON-NLS-2$
    }
    parse(new InputSource(is), dh);
}
/**
 * Parse the content of the given {@link java.io.InputStream}
 * instance as XML using the specified
 * {@link org.xml.sax.helpers.DefaultHandler}.
 *
 * @param is InputStream containing the content to be parsed.
 * @param dh The SAX DefaultHandler to use.
 * @param systemId The systemId which is needed for resolving relative URIs.
 * @exception IOException If any IO errors occur.
 * @exception IllegalArgumentException If the given InputStream is null.
 * @exception SAXException If the underlying parser throws a
 * SAXException while parsing.
 * version of this method instead.
 */

public void parse(InputStream is, DefaultHandler dh, String systemId) throws SAXException, IOException {
    if (is == null) {
        throw new IllegalArgumentException(EXmlMsg.getDefault().getString(EXmlMsg.CANNOT_BE_NULL, "InpuStream")); //$NON-NLS-1$ //$NON-NLS-2$
    }
    
    InputSource input = new InputSource(is);
    input.setSystemId(systemId);
    parse(input, dh);
}
/**
 * Parse the content described by the giving Uniform Resource
 * Identifier (URI) as XML using the specified
 * {@link org.xml.sax.helpers.DefaultHandler}.
 *
 * @param uri The location of the content to be parsed.
 * @param dh The SAX DefaultHandler to use.
 * @exception IOException If any IO errors occur.
 * @exception IllegalArgumentException If the uri is null.
 * @exception SAXException If the underlying parser throws a
 * SAXException while parsing.
 */

public void parse(String uri, DefaultHandler dh) throws SAXException, IOException {
    if (uri == null) {
        throw new IllegalArgumentException(EXmlMsg.getDefault().getString(EXmlMsg.CANNOT_BE_NULL, "URI")); //$NON-NLS-1$ //$NON-NLS-2$
    }
    parse(new InputSource(uri), dh);
}
/**
 * Returns the SAX parser that is encapsultated by the
 * implementation of this class.
 *
 * @return The SAX parser that is encapsultated by the
 *         implementation of this class.
 * 
 * @deprecated Parser class is deprecated, use instead getXMLReader()
 */

public abstract Parser getParser() throws SAXException;
}
