/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.jaxp.implForCore;

import org.eclipse.ercp.xml.dom.DocumentImpl;
import org.eclipse.ercp.xml.parser.EXmlMsg;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.ProcessingInstruction;
import org.w3c.dom.Text;
import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.ErrorHandler;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.ext.LexicalHandler;


public class DOMBuilder implements ContentHandler, LexicalHandler {
	private Node currentNode;
	private Document document;
	private Locator locator;
	private ErrorHandler errorHandler;
	private boolean fNamespaces;
	private boolean cdataStarted;
/**
 * Constructor for DOMBuilder
 */
public DOMBuilder() {
	super();
	this.fNamespaces = false;
	this.currentNode = null;
	this.document = null;
	this.locator = null;
	this.errorHandler = null;
	this.cdataStarted = false;
}
public Document getDocument() {
	return document;
}
public void setNamespaces(boolean namespaces) {
	this.fNamespaces = namespaces;
}
public void setNamespacePrefixes(boolean namespacePrefixes) {
	// NOP
}
/**
 * @see ContentHandler#setDocumentLocator(Locator)
 */
public void setDocumentLocator(Locator locator) {
	this.locator = locator;
}
/**
 * @see ContentHandler#startDocument()
 */
public void startDocument() throws SAXException {
	document = new DocumentImpl();
	currentNode = document;
}
/**
 * @see ContentHandler#endDocument()
 */
public void endDocument() throws SAXException {
	if (currentNode == document) {
		currentNode = null;
	} else {
		// It is not a well formed document,
		// at least one closing tag is missing
		warning(EXmlMsg.getDefault().getString(EXmlMsg.WRONG_ENDING_DOC));
	}
}
/**
 * @see ContentHandler#startPrefixMapping(String, String)
 */
public void startPrefixMapping(String prefix, String uri) throws SAXException {
	// NOP
}
/**
 * @see ContentHandler#endPrefixMapping(String)
 */
public void endPrefixMapping(String prefix) throws SAXException {
	// NOP
}
/**
 * @see ContentHandler#startElement(String, String, String, Attributes)
 */
public void startElement(String namespaceURI, String localName, String qName, Attributes atts)
	throws SAXException {
//	if ((currentNode == document) &&
//		(document.getFirstChild() != null)) {
//			throw new SAXException();
//		}
//	}
	Element element;
	if (fNamespaces) {
		element = document.createElementNS(namespaceURI, qName);
	} else {
		element = document.createElement(qName);
	}
	currentNode.appendChild(element);
	currentNode = element;
	int length = atts.getLength();
	if (fNamespaces) {
		for (int i = 0; i < length; i++) {
			element.setAttributeNS(atts.getURI(i), atts.getQName(i), atts.getValue(i));
		}
	} else {
		for (int i = 0; i < length; i++) {
			element.setAttribute(atts.getQName(i), atts.getValue(i));
		}
	}
}
/**
 * @see ContentHandler#endElement(String, String, String)
 */
public void endElement(String namespaceURI, String localName, String qName)
	throws SAXException {
	if (fNamespaces) {
		endElementNS(namespaceURI, localName);
	} else {
		endElement(qName);
	}
}
protected void endElement(String qName) throws SAXException {
	if (currentNode.getNodeName().equals(qName)) {
		currentNode = currentNode.getParentNode();
	} else {
		Node node = retreiveSuperNode(qName, currentNode);
		if (node == document) {
			warning(EXmlMsg.getDefault().getString(EXmlMsg.EndingTagExpected, currentNode.getNodeName()));
			currentNode = document;
		} else if (node == null) {
			warning(EXmlMsg.getDefault().getString(EXmlMsg.WARNING_START_TAG_EXPECTED, qName));
			currentNode = document;
		} else {
			currentNode = node.getParentNode();
		}
	
	}
}
protected Node retreiveSuperNode(String qName, Node node) throws SAXException {
	while ((node != null) && (node != document)) {
		if (node.getNodeName().equals(qName)) {
			return node;
		} else {
			warning(EXmlMsg.getDefault().getString(EXmlMsg.EndingTagExpected, currentNode.getNodeName())); //$NON-NLS-1$
			node = node.getParentNode();
		}
	}
	return null;
}
protected void endElementNS(String namespaceURI, String localName) throws SAXException {
	if ((currentNode.getNamespaceURI().equals(namespaceURI)) &&
		(currentNode.getLocalName().equals(localName))) {
		currentNode = currentNode.getParentNode();
	} else {
		warning(EXmlMsg.getDefault().getString(EXmlMsg.EndingTagExpected, currentNode.getNodeName())); //$NON-NLS-1$
		Node node = retreiveSuperNodeNS(namespaceURI, localName, currentNode.getParentNode());
		if (node != null) {
			currentNode = node.getParentNode();
		}
	}
}
protected Node retreiveSuperNodeNS(String namespaceURI, String localName, Node node) throws SAXException {
	while (node != null) {
		if ((node.getNamespaceURI().equals(namespaceURI)) &&
			(node.getLocalName().equals(localName))) {
			return node;
		} else {
			warning(EXmlMsg.getDefault().getString(EXmlMsg.EndingTagExpected, currentNode.getNodeName())); //$NON-NLS-1$
			node = node.getParentNode();
		}
	}
	return null;
}
/**
 * @see ContentHandler#characters(char[], int, int)
 */
public void characters(char[] ch, int start, int length) throws SAXException {
	if (currentNode != document) {
		Text text = (!cdataStarted)?
						document.createTextNode(new String(ch, start, length)):
						document.createCDATASection(new String(ch, start, length));
		currentNode.appendChild(text);
	}
}
/**
 * @see ContentHandler#ignorableWhitespace(char[], int, int)
 */
public void ignorableWhitespace(char[] ch, int start, int length) throws SAXException {
	// NOP
}
/**
 * @see ContentHandler#processingInstruction(String, String)
 */
public void processingInstruction(String target, String data) throws SAXException {
	if (currentNode != document) {
		ProcessingInstruction pi = document.createProcessingInstruction(target, data);
		currentNode.appendChild(pi);
	}
}
/**
 * @see ContentHandler#skippedEntity(String)
 */
public void skippedEntity(String name) throws SAXException {
	// NOP
}
public void setErrorHandler(ErrorHandler errorHandler) {
	this.errorHandler = errorHandler;
}
/**
 * Generate a warning event.
 */
protected void warning (String message) throws SAXException {
	if (errorHandler != null) {
		errorHandler.warning(new SAXParseException(message, locator));
	} else {
		throw new SAXParseException(message, locator);
	}
}
/**
 * Generate a fatal error event.
 */
protected void fatalError (String message) throws SAXException {
	if (errorHandler != null) {
		errorHandler.fatalError(new SAXParseException(message, locator));
	} else {
		throw new SAXParseException(message, locator);
	}
}
protected String getQNameOf(Node node) {
	return (node.getPrefix().equals(""))?node.getLocalName():node.getPrefix()+":"+node.getLocalName(); //$NON-NLS-1$ //$NON-NLS-2$
}
public void startCDATA() throws SAXException {
	this.cdataStarted = true;
}
public void endCDATA() throws SAXException {
	this.cdataStarted = false;
}
public void comment(char[] arg0, int arg1, int arg2) throws SAXException {
	// NOP
}
public void startDTD(String arg0, String arg1, String arg2) throws SAXException {
	// DTD are not supported
}
public void endDTD() throws SAXException {
	// DTD are not supported
}
public void endEntity(String arg0) throws SAXException {
	// NOP
}
public void startEntity(String arg0) throws SAXException {
	// NOP	
}
}

