/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.jaxp.implForCore;

import java.io.IOException;

import org.eclipse.ercp.xml.parser.*;
import org.xml.sax.*;
import org.xml.sax.ext.LexicalHandler;
import org.xml.sax.helpers.DefaultHandler;



/**
 * SAXParser implementation for jclCore and bigger
 */
public class SAXParserImpl extends AbstractSAXParserImpl implements XMLReader {
	private ContentHandler contentHandler;
	private ErrorHandler errorHandler;
	private LexicalHandler lexicalHandler;
	
	// This instance was created to avoid to test ALL the time if
	// the content handler is not null!
	private static final ContentHandler NullContentHandler = new ContentHandler() {
		public void setDocumentLocator (Locator locator) {}
		public void startDocument () throws SAXException {}
		public void endDocument() throws SAXException {}
		public void startPrefixMapping (String prefix, String uri) throws SAXException {}
		public void endPrefixMapping (String prefix) throws SAXException {}
		public void startElement (String namespaceURI, String localName, String qName, Attributes atts) throws SAXException {}
		public void endElement (String namespaceURI, String localName, String qName) throws SAXException {}
		public void characters (char ch[], int start, int length) throws SAXException {}
		public void ignorableWhitespace (char ch[], int start, int length) throws SAXException {}
		public void processingInstruction (String target, String data) throws SAXException {}
		public void skippedEntity (String name) throws SAXException {}
	};

public SAXParserImpl() {
	super();
	this.contentHandler = NullContentHandler;
	this.errorHandler = null;
}

protected void fatalError(String errorMsg, String parameter) throws SAXException {
		SAXParseException e = new SAXParseException(
									errorMsg(errorMsg, parameter),
									this);
		if (errorHandler != null) errorHandler.fatalError(e);

		// If the errorhandler didn't throw the exception
		// it is time to do it
		throw e;
}

protected void warning(String errorMsg, String parameter) throws SAXException {
		if (errorHandler != null) {
			errorHandler.warning(
				new SAXParseException(
						errorMsg(errorMsg, parameter),
						this));
		}
}

protected boolean canDoStringInterning() {
	return false;
}

protected void endPrefixMapping(NSDeclaration nsDecl) throws SAXException {
	String[][] declarations = nsDecl.getDeclarations();
	if (declarations == null) return;
	int len = declarations.length;
	for (int i = 0; i < len; i++) {
		contentHandler.endPrefixMapping(declarations[i][0]);
	}
}

protected void startPrefixMapping(String prefix, String uri) throws SAXException {
	currentNSDecl.addDeclaration(prefix, uri);
	contentHandler.startPrefixMapping(fStringInterning?prefix.intern():prefix, uri);
}

protected void characters(char[] ch, int start, int length) throws SAXException {
	contentHandler.characters(ch, start, length);
}

protected void startElement(String qName, String namespace, String localName, AttributesImpl attributes) throws SAXException {
	contentHandler.startElement(
						currentNSDecl.getUriFor(namespace),
						fStringInterning?localName.intern():localName,
						fStringInterning?qName.intern():qName,
						attributes);
}

protected void endElement(String qName, String namespace, String localName) throws SAXException {
	contentHandler.endElement(
					currentNSDecl.getUriFor(namespace),
					fStringInterning?localName.intern():localName,
					fStringInterning?qName.intern():qName);
}
protected void processingInstruction (String target, String data) throws SAXException {
	contentHandler.processingInstruction(target, data);
}
/**
 * Start parsing
 */
protected void startDocument() throws SAXException {
	contentHandler.setDocumentLocator(this);
	contentHandler.startDocument();
}
/**
 * End the parsing
 */
protected void endDocument() throws SAXException {
	contentHandler.endDocument();
}
/**
 * Parse the content of the given InputStream instance as
 * XML using the specified DefaultHandler.
 * 
 * @param is - InputStream containing the content to be parsed.
 * @param dh - The SAX DefaultHandler to use.
 * 
 * @exception java.io.IOException - If any IO errors occur.
 * @exception IllegalArgumentException - If the given InputStream is null.
 * @exception SAXException - If the underlying parser throws a SAXException while parsing.
 */
public void parse(InputSource is, DefaultHandler dh) throws SAXException, IOException {
	setFeature(XmlOrgFeaturesPrefix + NamespacesFeature, isNamespaceAware());
	setContentHandler(dh);
	setErrorHandler(dh);
	setDTDHandler(dh);
	setEntityResolver(dh);
	parse(is);
}

public XMLReader getXMLReader() throws SAXException {
	return this;
}
/**
 * Returns the SAX parser that is encapsultated by the
 * implementation of this class.
 *
 * @return The SAX parser that is encapsultated by the
 *         implementation of this class.
 * 
 * @deprecated Parser class is deprecated, use instead getXMLReader()
 */

public Parser getParser() throws SAXException {
	throw new SAXNotSupportedException("org.xml.sax.Parser"); //$NON-NLS-1$
}
////////////////////////////////
/**
 * @see XMLReader#setContentHandler(ContentHandler)
 */
public void setContentHandler(ContentHandler contentHandler) {
	if (contentHandler != null) {
		this.contentHandler = contentHandler;
	} else {
		this.contentHandler = NullContentHandler;
	}
}
/**
 * @see XMLReader#getContentHandler()
 */
public ContentHandler getContentHandler() {
	return (contentHandler == NullContentHandler)?null:contentHandler;
}
/**
 * @see XMLReader#setErrorHandler(ErrorHandler)
 */
public void setErrorHandler(ErrorHandler errorHandler) {
	this.errorHandler = errorHandler;
}
/**
 * @see XMLReader#getErrorHandler()
 */
public ErrorHandler getErrorHandler() {
	return errorHandler;
}
/**
 * @see XMLReader#setDTDHandler(DTDHandler)
 */
public void setDTDHandler(DTDHandler handler) {
	// This parser doesn't support any DTD handler
}
/**
 * @see XMLReader#getDTDHandler()
 */
public DTDHandler getDTDHandler() {
	// This parser doesn't support any DTD handler
	return null;
}
/**
 * @see XMLReader#setEntityResolver(EntityResolver)
 */
public void setEntityResolver(EntityResolver resolver) {
	// This parser doesn't support any EntityResolver
}

//implements LexicalHandler, DeclHandler, EntityResolver2
/**
 * @see XMLReader#getEntityResolver()
 */
public EntityResolver getEntityResolver() {
	// This parser doesn't support any EntityResolver
	return null;
}
protected void setLexicalHandler(Object value) {
	this.lexicalHandler = (LexicalHandler)value;
	
}
protected Object getRecognizedXmlOrgProperty(String name) throws SAXNotSupportedException {
	if (LexicalHandlerID.equals(name)) return lexicalHandler;
	return super.getRecognizedXmlOrgProperty(name);
}
protected void setRecognizedXmlOrgProperty(String name, Object value) throws SAXNotSupportedException {
	if (LexicalHandlerID.equals(name)) {
		if (value instanceof LexicalHandler) {
			this.lexicalHandler = (LexicalHandler)value;
		} else {
			throw new SAXNotSupportedException(name);
		}
	} else {
		super.setRecognizedXmlOrgProperty(name, value);
	}
}
protected Object getLexicalHandler() {
	return lexicalHandler;
}
protected void startCDATA() throws SAXException {
	if (lexicalHandler != null) lexicalHandler.startCDATA();
}
protected void endCDATA() throws SAXException {
	if (lexicalHandler != null) lexicalHandler.endCDATA();
}
}
