/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.dom;

import org.w3c.dom.Attr;
import org.w3c.dom.DOMException;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

public class AttrImpl extends NSNodeImpl implements Attr {
	protected String 	value;
	protected ElementImpl ownerElement;
/**
 * Constructor for AttrImpl
 */
public AttrImpl(DocumentImpl ownerDocument, String qualifiedName) {
	super(ownerDocument, qualifiedName);
	this.value = "";
	this.ownerElement = null;
}
public AttrImpl(DocumentImpl ownerDocument, String namespaceURI, String qualifiedName) {
	super(ownerDocument, namespaceURI, qualifiedName);
	this.value = "";
	this.ownerElement = null;
}
/**
 * Returns the name of this attribute. 
 */
public String getName() {
	return getNodeName();
}
/**
 * If this attribute was explicitly given a value in the original 
 * document, this is <code>true</code>; otherwise, it is 
 * <code>false</code>. Note that the implementation is in charge of this 
 * attribute, not the user. If the user changes the value of the 
 * attribute (even if it ends up having the same value as the default 
 * value) then the <code>specified</code> flag is automatically flipped 
 * to <code>true</code>. To re-specify the attribute as the default 
 * value from the DTD, the user must delete the attribute. The 
 * implementation will then make a new attribute available with 
 * <code>specified</code> set to <code>false</code> and the default 
 * value (if one exists).
 * <br>In summary:  If the attribute has an assigned value in the document 
 * then <code>specified</code> is <code>true</code>, and the value is 
 * the assigned value.  If the attribute has no assigned value in the 
 * document and has a default value in the DTD, then 
 * <code>specified</code> is <code>false</code>, and the value is the 
 * default value in the DTD. If the attribute has no assigned value in 
 * the document and has a value of #IMPLIED in the DTD, then the 
 * attribute does not appear in the structure model of the document. If 
 * the <code>ownerElement</code> attribute is <code>null</code> (i.e. 
 * because it was just created or was set to <code>null</code> by the 
 * various removal and cloning operations) <code>specified</code> is 
 * <code>true</code>. 
 */
public boolean getSpecified() {
	// Always true because we don't support DTD
	// so we can have any default value
	return true;
}
/**
 * On retrieval, the value of the attribute is returned as a string. 
 * Character and general entity references are replaced with their 
 * values. See also the method <code>getAttribute</code> on the 
 * <code>Element</code> interface.
 * <br>On setting, this creates a <code>Text</code> node with the unparsed 
 * contents of the string. I.e. any characters that an XML processor 
 * would recognize as markup are instead treated as literal text. See 
 * also the method <code>setAttribute</code> on the <code>Element</code> 
 * interface.
 * @exception DOMException
 *   NO_MODIFICATION_ALLOWED_ERR: Raised when the node is readonly.
 */
public String getValue() {
	return value;
}
public void setValue(String value) throws DOMException {
	this.value = value;
}
/**
 * The <code>Element</code> node this attribute is attached to or 
 * <code>null</code> if this attribute is not in use.
 * @since DOM Level 2
 */
public Element getOwnerElement() {
	return ownerElement;
}
void setOwnerElement(ElementImpl ownerElement) {
	this.ownerElement = ownerElement;
}
/**
 * A code representing the type of the underlying object, as defined above.
 */
public short getNodeType() {
    return Node.ATTRIBUTE_NODE;
}
/**
 * The value of this node, depending on its type; see the table above. 
 * When it is defined to be <code>null</code>, setting it has no effect.
 * @exception DOMException
 *   NO_MODIFICATION_ALLOWED_ERR: Raised when the node is readonly.
 * @exception DOMException
 *   DOMSTRING_SIZE_ERR: Raised when it would return more characters than 
 *   fit in a <code>DOMString</code> variable on the implementation 
 *   platform.
 */
public String getNodeValue() throws DOMException {
	return getValue();
}
public void setNodeValue(String nodeValue) throws DOMException {
	setValue(nodeValue);
}
/**
 * Returns a duplicate of this node, i.e., serves as a generic copy 
 * constructor for nodes. The duplicate node has no parent; (
 * <code>parentNode</code> is <code>null</code>.).
 * <br>Cloning an <code>Element</code> copies all attributes and their 
 * values, including those generated by the XML processor to represent 
 * defaulted attributes, but this method does not copy any text it 
 * contains unless it is a deep clone, since the text is contained in a 
 * child <code>Text</code> node. Cloning an <code>Attribute</code> 
 * directly, as opposed to be cloned as part of an <code>Element</code> 
 * cloning operation, returns a specified attribute (
 * <code>specified</code> is <code>true</code>). Cloning any other type 
 * of node simply returns a copy of this node.
 * <br>Note that cloning an immutable subtree results in a mutable copy, 
 * but the children of an <code>EntityReference</code> clone are readonly
 * . In addition, clones of unspecified <code>Attr</code> nodes are 
 * specified. And, cloning <code>Document</code>, 
 * <code>DocumentType</code>, <code>Entity</code>, and 
 * <code>Notation</code> nodes is implementation dependent.
 * @param deepIf <code>true</code>, recursively clone the subtree under 
 *   the specified node; if <code>false</code>, clone only the node 
 *   itself (and its attributes, if it is an <code>Element</code>). 
 * @return The duplicate node.
 */
public Node cloneNode(boolean deep) {
	AttrImpl clone = new AttrImpl(ownerDocument, namespaceURI, qualifiedName);
	//clone.ownerElement = ownerElement; The owner should be a copy
	clone.value = value;
	return clone;
}
}

