/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.dom;

import java.util.Enumeration;
import java.util.Hashtable;

import org.eclipse.ercp.xml.parser.EXmlMsg;
import org.w3c.dom.Attr;
import org.w3c.dom.DOMException;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;


/**
 * NamedNodeMap implementation
 */
public class AttributeMap implements NamedNodeMap {
	ElementImpl owner;
	Hashtable attributes;
/**
 * Constructor for AttributeMap.
 */
public AttributeMap(ElementImpl owner) {
	super();
	this.owner = owner;
	this.attributes = new Hashtable();
}
/**
 * Retrieves a node specified by name.
 * @param nameThe <code>nodeName</code> of a node to retrieve.
 * @return A <code>Node</code> (of any type) with the specified 
 *   <code>nodeName</code>, or <code>null</code> if it does not identify 
 *   any node in this map.
 */
public Node getNamedItem(String name) {
	return (Node)attributes.get(name);
}

/**
 * Adds a node using its <code>nodeName</code> attribute. If a node with 
 * that name is already present in this map, it is replaced by the new 
 * one.
 * <br>As the <code>nodeName</code> attribute is used to derive the name 
 * which the node must be stored under, multiple nodes of certain types 
 * (those that have a "special" string value) cannot be stored as the 
 * names would clash. This is seen as preferable to allowing nodes to be 
 * aliased.
 * @param argA node to store in this map. The node will later be 
 *   accessible using the value of its <code>nodeName</code> attribute.
 * @return If the new <code>Node</code> replaces an existing node the 
 *   replaced <code>Node</code> is returned, otherwise <code>null</code> 
 *   is returned.
 * @exception DOMException
 *   WRONG_DOCUMENT_ERR: Raised if <code>arg</code> was created from a 
 *   different document than the one that created this map.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this map is readonly.
 *   <br>INUSE_ATTRIBUTE_ERR: Raised if <code>arg</code> is an 
 *   <code>Attr</code> that is already an attribute of another 
 *   <code>Element</code> object. The DOM user must explicitly clone 
 *   <code>Attr</code> nodes to re-use them in other elements.
 */
public Node setNamedItem(Node arg) throws DOMException {
	if (owner.getOwnerDocument() != arg.getOwnerDocument()) {
		throw new DOMException(DOMException.WRONG_DOCUMENT_ERR, EXmlMsg.getDefault().getString(EXmlMsg.WRONG_DOCUMENT_ERR)); //$NON-NLS-1$
	}
	AttrImpl attr = (AttrImpl)arg;
	if ((attr.getOwnerElement() != null) &&
		(attr.getOwnerElement() != this.owner)) {
		throw new DOMException(DOMException.INUSE_ATTRIBUTE_ERR, EXmlMsg.getDefault().getString(EXmlMsg.INUSE_ATTRIBUTE_ERR)); //$NON-NLS-1$
	}
	
	///////////////////////////////////////////////////////
	// set owner  add by nzhang
	attr.setOwnerElement(this.owner);
	///////////////////////////////////////////////////////
	
	AttrImpl oldAttr = (AttrImpl)getNamedItem(attr.getName());
	attributes.put(attr.getName(), attr);
	if (oldAttr != null) oldAttr.setOwnerElement(null);
	return oldAttr;
}

/**
 * Removes a node specified by name. When this map contains the attributes 
 * attached to an element, if the removed attribute is known to have a 
 * default value, an attribute immediately appears containing the 
 * default value as well as the corresponding namespace URI, local name, 
 * and prefix when applicable.
 * @param nameThe <code>nodeName</code> of the node to remove.
 * @return The node removed from this map if a node with such a name 
 *   exists.
 * @exception DOMException
 *   NOT_FOUND_ERR: Raised if there is no node named <code>name</code> in 
 *   this map.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this map is readonly.
 */
public Node removeNamedItem(String name) throws DOMException {
	AttrImpl oldAttr = (AttrImpl)getNamedItem(name);
	attributes.remove(name);
	if (oldAttr != null) {
		oldAttr.setOwnerElement(null);
	}
	return oldAttr;
}

/**
 * Returns the <code>index</code>th item in the map. If <code>index</code> 
 * is greater than or equal to the number of nodes in this map, this 
 * returns <code>null</code>.
 * @param indexIndex into this map.
 * @return The node at the <code>index</code>th position in the map, or 
 *   <code>null</code> if that is not a valid index.
 */
public Node item(int index) {
	Enumeration anEnumeration = attributes.elements();
	int pos = -1;
	Node node;
	while (anEnumeration.hasMoreElements()) {
		node = (Node)(anEnumeration.nextElement());
		pos++;
		if (pos == index) return node;
	}
	return null;
}

/**
 * The number of nodes in this map. The range of valid child node indices 
 * is <code>0</code> to <code>length-1</code> inclusive. 
 */
public int getLength() {
	return attributes.size();
}

/**
 * Retrieves a node specified by local name and namespace URI. HTML-only 
 * DOM implementations do not need to implement this method.
 * @param namespaceURIThe namespace URI of the node to retrieve.
 * @param localNameThe local name of the node to retrieve.
 * @return A <code>Node</code> (of any type) with the specified local 
 *   name and namespace URI, or <code>null</code> if they do not 
 *   identify any node in this map.
 * @since DOM Level 2
 */
public Node getNamedItemNS(String namespaceURI, String localName) {
	if (attributes == null) return null;
	Enumeration anEnumeration = attributes.elements();
	Node attribute;
	String uri;
	while (anEnumeration.hasMoreElements()) {
		attribute = (Node)(anEnumeration.nextElement());
		uri = attribute.getNamespaceURI();
		if (((namespaceURI == uri) ||
				((namespaceURI != null) && (namespaceURI.equals(uri)))) &&
			(localName.equals(attribute.getLocalName()))) {
			return attribute;
		}
	}
	return null;
}

/**
 * Adds a node using its <code>namespaceURI</code> and 
 * <code>localName</code>. If a node with that namespace URI and that 
 * local name is already present in this map, it is replaced by the new 
 * one.
 * <br>HTML-only DOM implementations do not need to implement this method.
 * @param argA node to store in this map. The node will later be 
 *   accessible using the value of its <code>namespaceURI</code> and 
 *   <code>localName</code> attributes.
 * @return If the new <code>Node</code> replaces an existing node the 
 *   replaced <code>Node</code> is returned, otherwise <code>null</code> 
 *   is returned.
 * @exception DOMException
 *   WRONG_DOCUMENT_ERR: Raised if <code>arg</code> was created from a 
 *   different document than the one that created this map.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this map is readonly.
 *   <br>INUSE_ATTRIBUTE_ERR: Raised if <code>arg</code> is an 
 *   <code>Attr</code> that is already an attribute of another 
 *   <code>Element</code> object. The DOM user must explicitly clone 
 *   <code>Attr</code> nodes to re-use them in other elements.
 * @since DOM Level 2
 */
public Node setNamedItemNS(Node arg) throws DOMException {
	return owner.setAttributeNodeNS((Attr)arg);
}

/**
 * Removes a node specified by local name and namespace URI. A removed 
 * attribute may be known to have a default value when this map contains 
 * the attributes attached to an element, as returned by the attributes 
 * attribute of the <code>Node</code> interface. If so, an attribute 
 * immediately appears containing the default value as well as the 
 * corresponding namespace URI, local name, and prefix when applicable.
 * <br>HTML-only DOM implementations do not need to implement this method.
 * @param namespaceURIThe namespace URI of the node to remove.
 * @param localNameThe local name of the node to remove.
 * @return The node removed from this map if a node with such a local 
 *   name and namespace URI exists.
 * @exception DOMException
 *   NOT_FOUND_ERR: Raised if there is no node with the specified 
 *   <code>namespaceURI</code> and <code>localName</code> in this map.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this map is readonly.
 * @since DOM Level 2
 */
public Node removeNamedItemNS(String namespaceURI, String localName) throws DOMException {
	AttrImpl oldAttr = (AttrImpl)getNamedItemNS(namespaceURI, localName);
	if (oldAttr != null) {
		attributes.remove(oldAttr.getName());
		oldAttr.setOwnerElement(null);
	}
	return oldAttr;
}
//////////////////////////////////////////
Node removeItem(Node node) throws DOMException {
	if (node == null) return null;
	AttrImpl attr = (AttrImpl)node;
	if (attr.getOwnerElement() != owner) {
		throw new DOMException(DOMException.NOT_FOUND_ERR, EXmlMsg.getDefault().getString(EXmlMsg.NOT_FOUND_ERR));
	}
	attributes.remove(attr.getName());
	attr.setOwnerElement(null);

	return attr;
}
void setNamedItemNS(String namespaceURI,String qualifiedName,String value)
	throws DOMException {
	AttrImpl attribute = (AttrImpl)getNamedItem(qualifiedName);
	if (attribute == null) {
		attribute = (AttrImpl)(owner.getOwnerDocument().createAttributeNS(namespaceURI, qualifiedName));
		attributes.put(qualifiedName, attribute);
		attribute.setOwnerElement(owner);
	}
	attribute.setValue(value);
}
void setNamedItem(String name, String value) throws DOMException {
	AttrImpl attribute = (AttrImpl)getNamedItem(name);
	if (attribute == null) {
		attribute = (AttrImpl)(owner.getOwnerDocument().createAttribute(name));
		attributes.put(name, attribute);
		attribute.setOwnerElement(owner);
	}
	attribute.setValue(value);
}
}
