/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.dom;

import org.w3c.dom.DOMException;

public abstract class CharacterDataImpl extends TreeNode {
	protected String data;
/**
 * Constructor for CharacterDataImpl
 */
public CharacterDataImpl(DocumentImpl ownerDocument, String data) {
	super(ownerDocument);
	this.data = data;
}
/**
 * The character data of the node that implements this interface. The DOM 
 * implementation may not put arbitrary limits on the amount of data 
 * that may be stored in a <code>CharacterData</code> node. However, 
 * implementation limits may mean that the entirety of a node's data may 
 * not fit into a single <code>DOMString</code>. In such cases, the user 
 * may call <code>substringData</code> to retrieve the data in 
 * appropriately sized pieces.
 * @exception DOMException
 *   NO_MODIFICATION_ALLOWED_ERR: Raised when the node is readonly.
 * @exception DOMException
 *   DOMSTRING_SIZE_ERR: Raised when it would return more characters than 
 *   fit in a <code>DOMString</code> variable on the implementation 
 *   platform.
 */
public String getData() throws DOMException {
	return data;
}
public void setData(String data) throws DOMException {
	this.data = data;
}
/**
 * Extracts a range of data from the node.
 * @param offsetStart offset of substring to extract.
 * @param countThe number of 16-bit units to extract.
 * @return The specified substring. If the sum of <code>offset</code> and 
 *   <code>count</code> exceeds the <code>length</code>, then all 16-bit 
 *   units to the end of the data are returned.
 * @exception DOMException
 *   INDEX_SIZE_ERR: Raised if the specified <code>offset</code> is 
 *   negative or greater than the number of 16-bit units in 
 *   <code>data</code>, or if the specified <code>count</code> is 
 *   negative.
 *   <br>DOMSTRING_SIZE_ERR: Raised if the specified range of text does 
 *   not fit into a <code>DOMString</code>.
 */
public String substringData(int offset, int count) throws DOMException {
	if(count < 0 || offset < 0 || offset > data.length()-1)
	  throw new DOMException(DOMException.INDEX_SIZE_ERR,null);

	//JCLExtreme limitation: int tailIndex=Math.min(offset+count,value.length());
	int tailIndex = (offset+count<data.length())?offset+count:data.length();
	return data.substring(offset, tailIndex);
}
/**
 * Append the string to the end of the character data of the node. Upon 
 * success, <code>data</code> provides access to the concatenation of 
 * <code>data</code> and the <code>DOMString</code> specified.
 * @param argThe <code>DOMString</code> to append.
 * @exception DOMException
 *   NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 */
public void appendData(String arg) throws DOMException {
	data += arg;
}
/**
 * Remove a range of 16-bit units from the node. Upon success, 
 * <code>data</code> and <code>length</code> reflect the change.
 * @param offsetThe offset from which to start removing.
 * @param countThe number of 16-bit units to delete. If the sum of 
 *   <code>offset</code> and <code>count</code> exceeds 
 *   <code>length</code> then all 16-bit units from <code>offset</code> 
 *   to the end of the data are deleted.
 * @exception DOMException
 *   INDEX_SIZE_ERR: Raised if the specified <code>offset</code> is 
 *   negative or greater than the number of 16-bit units in 
 *   <code>data</code>, or if the specified <code>count</code> is 
 *   negative.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 */
public void insertData(int offset, String arg) throws DOMException {
	try {
		StringBuffer buffer = new StringBuffer(data.length() + arg.length());
		buffer.append(data.substring(0, offset));
		buffer.append(arg);
		buffer.append(data.substring(offset));
		
		data = buffer.toString();
	}
	catch (IndexOutOfBoundsException e) {
		throw new DOMException(DOMException.INDEX_SIZE_ERR, null);
	}
}
/**
 * Remove a range of 16-bit units from the node. Upon success, 
 * <code>data</code> and <code>length</code> reflect the change.
 * @param offsetThe offset from which to start removing.
 * @param countThe number of 16-bit units to delete. If the sum of 
 *   <code>offset</code> and <code>count</code> exceeds 
 *   <code>length</code> then all 16-bit units from <code>offset</code> 
 *   to the end of the data are deleted.
 * @exception DOMException
 *   INDEX_SIZE_ERR: Raised if the specified <code>offset</code> is 
 *   negative or greater than the number of 16-bit units in 
 *   <code>data</code>, or if the specified <code>count</code> is 
 *   negative.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 */
public void deleteData(int offset, int count) throws DOMException {
	if (count < 0) {
		throw new DOMException(DOMException.INDEX_SIZE_ERR, null);
	}
	int tailLength = data.length() - count - offset;
	tailLength = (tailLength > 0)?tailLength:0;
	try {
		data = data.substring(0, offset) + (tailLength > 0 ? data.substring(offset + count, offset + count + tailLength) : "");
	} catch (IndexOutOfBoundsException e) {
		throw new DOMException(DOMException.INDEX_SIZE_ERR, null);
	}
}
/**
 * Replace the characters starting at the specified 16-bit unit offset 
 * with the specified string.
 * @param offsetThe offset from which to start replacing.
 * @param countThe number of 16-bit units to replace. If the sum of 
 *   <code>offset</code> and <code>count</code> exceeds 
 *   <code>length</code>, then all 16-bit units to the end of the data 
 *   are replaced; (i.e., the effect is the same as a <code>remove</code>
 *    method call with the same range, followed by an <code>append</code>
 *    method invocation).
 * @param argThe <code>DOMString</code> with which the range must be 
 *   replaced.
 * @exception DOMException
 *   INDEX_SIZE_ERR: Raised if the specified <code>offset</code> is 
 *   negative or greater than the number of 16-bit units in 
 *   <code>data</code>, or if the specified <code>count</code> is 
 *   negative.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 */
public void replaceData(int offset, int count, String arg) throws DOMException {
	deleteData(offset, count);
	insertData(offset, arg);
}
/**
 * The value of this node, depending on its type; see the table above. 
 * When it is defined to be <code>null</code>, setting it has no effect.
 * @exception DOMException
 *   NO_MODIFICATION_ALLOWED_ERR: Raised when the node is readonly.
 * @exception DOMException
 *   DOMSTRING_SIZE_ERR: Raised when it would return more characters than 
 *   fit in a <code>DOMString</code> variable on the implementation 
 *   platform.
 */
public String getNodeValue() throws DOMException {
	return getData();
}
public void setNodeValue(String nodeValue) throws DOMException {
	setData(nodeValue);
}
}

