/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.dom;

import org.w3c.dom.Attr;
import org.w3c.dom.CDATASection;
import org.w3c.dom.Comment;
import org.w3c.dom.DOMException;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentFragment;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;
import org.w3c.dom.EntityReference;
import org.w3c.dom.Node;
import org.w3c.dom.ProcessingInstruction;
import org.w3c.dom.Text;

public class DocumentImpl extends TreeNode implements Document {
	protected ElementImpl documentElement;
/**
 * Constructor for DocumentImpl
 */
public DocumentImpl() {
	super(null, "#document"); //$NON-NLS-1$
	this.documentElement = null;
}
/**
 * The Document Type Declaration (see <code>DocumentType</code>) 
 * associated with this document. For HTML documents as well as XML 
 * documents without a document type declaration this returns 
 * <code>null</code>. The DOM Level 2 does not support editing the 
 * Document Type Declaration. <code>docType</code> cannot be altered in 
 * any way, including through the use of methods inherited from the 
 * <code>Node</code> interface, such as <code>insertNode</code> or 
 * <code>removeNode</code>.
 */
public DocumentType getDoctype() {
	return null;
}
/**
 * The <code>DOMImplementation</code> object that handles this document. A 
 * DOM application may use objects from multiple implementations.
 */
public DOMImplementation getImplementation() {
	return null;
}
/**
 * This is a convenience attribute that allows direct access to the child 
 * node that is the root element of the document. For HTML documents, 
 * this is the element with the tagName "HTML".
 */
public Element getDocumentElement() {
	return documentElement;
}
/**
 * Creates an element of the type specified. Note that the instance 
 * returned implements the <code>Element</code> interface, so attributes 
 * can be specified directly on the returned object.
 * <br>In addition, if there are known attributes with default values, 
 * <code>Attr</code> nodes representing them are automatically created 
 * and attached to the element.
 * <br>To create an element with a qualified name and namespace URI, use 
 * the <code>createElementNS</code> method.
 * @param tagNameThe name of the element type to instantiate. For XML, 
 *   this is case-sensitive. For HTML, the <code>tagName</code> 
 *   parameter may be provided in any case, but it must be mapped to the 
 *   canonical uppercase form by the DOM implementation. 
 * @return A new <code>Element</code> object with the 
 *   <code>nodeName</code> attribute set to <code>tagName</code>, and 
 *   <code>localName</code>, <code>prefix</code>, and 
 *   <code>namespaceURI</code> set to <code>null</code>.
 * @exception DOMException
 *   INVALID_CHARACTER_ERR: Raised if the specified name contains an 
 *   illegal character.
 */
public Element createElement(String tagName) throws DOMException {
	return new ElementImpl(this, tagName);
}
/**
 * Creates an empty <code>DocumentFragment</code> object. 
 * @return A new <code>DocumentFragment</code>.
 */
public DocumentFragment createDocumentFragment() {
	throw new DOMException(DOMException.NOT_SUPPORTED_ERR, "document fragment"); //$NON-NLS-1$
}
/**
 * Creates a <code>Text</code> node given the specified string.
 * @param dataThe data for the node.
 * @return The new <code>Text</code> object.
 */
public Text createTextNode(String data) {
	return new TextImpl(this, data);
}
/**
 * Creates a <code>Comment</code> node given the specified string.
 * @param dataThe data for the node.
 * @return The new <code>Comment</code> object.
 */
public Comment createComment(String data) {
	// Not supported (our SAX doesn't let you create comments)
	throw new DOMException(DOMException.NOT_SUPPORTED_ERR, "comment"); //$NON-NLS-1$
}
/**
 * Creates a <code>CDATASection</code> node whose value is the specified 
 * string.
 * @param dataThe data for the <code>CDATASection</code> contents.
 * @return The new <code>CDATASection</code> object.
 * @exception DOMException
 *   NOT_SUPPORTED_ERR: Raised if this document is an HTML document.
 */
public CDATASection createCDATASection(String data) throws DOMException {
	return new CDATASectionImpl(this, data);
}
/**
 * Creates a <code>ProcessingInstruction</code> node given the specified 
 * name and data strings.
 * @param targetThe target part of the processing instruction.
 * @param dataThe data for the node.
 * @return The new <code>ProcessingInstruction</code> object.
 * @exception DOMException
 *   INVALID_CHARACTER_ERR: Raised if the specified target contains an 
 *   illegal character.
 *   <br>NOT_SUPPORTED_ERR: Raised if this document is an HTML document.
 */
public ProcessingInstruction createProcessingInstruction(String target,String data) throws DOMException {
	return new ProcessingInstructionImpl(this, target, data);
}
/**
 * Creates an <code>Attr</code> of the given name. Note that the 
 * <code>Attr</code> instance can then be set on an <code>Element</code> 
 * using the <code>setAttributeNode</code> method. 
 * <br>To create an attribute with a qualified name and namespace URI, use 
 * the <code>createAttributeNS</code> method.
 * @param nameThe name of the attribute.
 * @return A new <code>Attr</code> object with the <code>nodeName</code> 
 *   attribute set to <code>name</code>, and <code>localName</code>, 
 *   <code>prefix</code>, and <code>namespaceURI</code> set to 
 *   <code>null</code>. The value of the attribute is the empty string.
 * @exception DOMException
 *   INVALID_CHARACTER_ERR: Raised if the specified name contains an 
 *   illegal character.
 */
public Attr createAttribute(String name) throws DOMException {
	return new AttrImpl(this, name);
}
/**
 * Creates an <code>EntityReference</code> object. In addition, if the 
 * referenced entity is known, the child list of the 
 * <code>EntityReference</code> node is made the same as that of the 
 * corresponding <code>Entity</code> node.If any descendant of the 
 * <code>Entity</code> node has an unbound namespace prefix, the 
 * corresponding descendant of the created <code>EntityReference</code> 
 * node is also unbound; (its <code>namespaceURI</code> is 
 * <code>null</code>). The DOM Level 2 does not support any mechanism to 
 * resolve namespace prefixes.
 * @param nameThe name of the entity to reference. 
 * @return The new <code>EntityReference</code> object.
 * @exception DOMException
 *   INVALID_CHARACTER_ERR: Raised if the specified name contains an 
 *   illegal character.
 *   <br>NOT_SUPPORTED_ERR: Raised if this document is an HTML document.
 */
public EntityReference createEntityReference(String name) throws DOMException {
	// Not supported (our SAX doesn't let you create comments)
	throw new DOMException(DOMException.NOT_SUPPORTED_ERR, "entity reference"); //$NON-NLS-1$
}
/**
 * @see Document#importNode(Node, boolean)
 */
public Node importNode(Node importedNode, boolean deep) throws DOMException {
	return null;
}
/**
 * Creates an element of the given qualified name and namespace URI. 
 * HTML-only DOM implementations do not need to implement this method.
 * @param namespaceURIThe namespace URI of the element to create.
 * @param qualifiedNameThe qualified name of the element type to 
 *   instantiate.
 * @return A new <code>Element</code> object with the following 
 *   attributes:AttributeValue<code>Node.nodeName</code>
 *   <code>qualifiedName</code><code>Node.namespaceURI</code>
 *   <code>namespaceURI</code><code>Node.prefix</code>prefix, extracted 
 *   from <code>qualifiedName</code>, or <code>null</code> if there is 
 *   no prefix<code>Node.localName</code>local name, extracted from 
 *   <code>qualifiedName</code><code>Element.tagName</code>
 *   <code>qualifiedName</code>
 * @exception DOMException
 *   INVALID_CHARACTER_ERR: Raised if the specified qualified name 
 *   contains an illegal character.
 *   <br>NAMESPACE_ERR: Raised if the <code>qualifiedName</code> is 
 *   malformed, if the <code>qualifiedName</code> has a prefix and the 
 *   <code>namespaceURI</code> is <code>null</code>, or if the 
 *   <code>qualifiedName</code> has a prefix that is "xml" and the 
 *   <code>namespaceURI</code> is different from "
 *   http://www.w3.org/XML/1998/namespace" .
 * @since DOM Level 2
 */
public Element createElementNS(String namespaceURI, String qualifiedName)
	throws DOMException {
	return new ElementImpl(this, namespaceURI, qualifiedName);
}
/**
 * Creates an attribute of the given qualified name and namespace URI. 
 * HTML-only DOM implementations do not need to implement this method.
 * @param namespaceURIThe namespace URI of the attribute to create.
 * @param qualifiedNameThe qualified name of the attribute to instantiate.
 * @return A new <code>Attr</code> object with the following attributes:
 *   AttributeValue<code>Node.nodeName</code>qualifiedName
 *   <code>Node.namespaceURI</code><code>namespaceURI</code>
 *   <code>Node.prefix</code>prefix, extracted from 
 *   <code>qualifiedName</code>, or <code>null</code> if there is no 
 *   prefix<code>Node.localName</code>local name, extracted from 
 *   <code>qualifiedName</code><code>Attr.name</code>
 *   <code>qualifiedName</code><code>Node.nodeValue</code>the empty 
 *   string
 * @exception DOMException
 *   INVALID_CHARACTER_ERR: Raised if the specified qualified name 
 *   contains an illegal character.
 *   <br>NAMESPACE_ERR: Raised if the <code>qualifiedName</code> is 
 *   malformed, if the <code>qualifiedName</code> has a prefix and the 
 *   <code>namespaceURI</code> is <code>null</code>, if the 
 *   <code>qualifiedName</code> has a prefix that is "xml" and the 
 *   <code>namespaceURI</code> is different from "
 *   http://www.w3.org/XML/1998/namespace", or if the 
 *   <code>qualifiedName</code> is "xmlns" and the 
 *   <code>namespaceURI</code> is different from "
 *   http://www.w3.org/2000/xmlns/".
 * @since DOM Level 2
 */
public Attr createAttributeNS(String namespaceURI, String qualifiedName)
	throws DOMException {
	return new AttrImpl(this, namespaceURI, qualifiedName);
}
/**
 * Returns the <code>Element</code> whose <code>ID</code> is given by 
 * <code>elementId</code>. If no such element exists, returns 
 * <code>null</code>. Behavior is not defined if more than one element 
 * has this <code>ID</code>. The DOM implementation must have 
 * information that says which attributes are of type ID. Attributes 
 * with the name "ID" are not of type ID unless so defined. 
 * Implementations that do not know whether attributes are of type ID or 
 * not are expected to return <code>null</code>.
 * @param elementIdThe unique <code>id</code> value for an element.
 * @return The matching element.
 * @since DOM Level 2
 */
public Element getElementById(String elementId) {
	// Not supported
	return null;
}
public short getNodeType() {
    return Node.DOCUMENT_NODE;
}
/**
 * Adds the node <code>newChild</code> to the end of the list of children 
 * of this node. If the <code>newChild</code> is already in the tree, it 
 * is first removed.
 * @param newChildThe node to add.If it is a <code>DocumentFragment</code>
 *    object, the entire contents of the document fragment are moved 
 *   into the child list of this node
 * @return The node added.
 * @exception DOMException
 *   HIERARCHY_REQUEST_ERR: Raised if this node is of a type that does not 
 *   allow children of the type of the <code>newChild</code> node, or if 
 *   the node to append is one of this node's ancestors.
 *   <br>WRONG_DOCUMENT_ERR: Raised if <code>newChild</code> was created 
 *   from a different document than the one that created this node.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 */
public Node appendChild(Node newChild) throws DOMException {
	super.appendChild(newChild);
	if (newChild.getNodeType() == Node.ELEMENT_NODE) {
		documentElement = (ElementImpl)newChild;
	}
	return newChild;
}
/**
 * Returns a duplicate of this node, i.e., serves as a generic copy 
 * constructor for nodes. The duplicate node has no parent; (
 * <code>parentNode</code> is <code>null</code>.).
 * <br>Cloning an <code>Element</code> copies all attributes and their 
 * values, including those generated by the XML processor to represent 
 * defaulted attributes, but this method does not copy any text it 
 * contains unless it is a deep clone, since the text is contained in a 
 * child <code>Text</code> node. Cloning an <code>Attribute</code> 
 * directly, as opposed to be cloned as part of an <code>Element</code> 
 * cloning operation, returns a specified attribute (
 * <code>specified</code> is <code>true</code>). Cloning any other type 
 * of node simply returns a copy of this node.
 * <br>Note that cloning an immutable subtree results in a mutable copy, 
 * but the children of an <code>EntityReference</code> clone are readonly
 * . In addition, clones of unspecified <code>Attr</code> nodes are 
 * specified. And, cloning <code>Document</code>, 
 * <code>DocumentType</code>, <code>Entity</code>, and 
 * <code>Notation</code> nodes is implementation dependent.
 * @param deepIf <code>true</code>, recursively clone the subtree under 
 *   the specified node; if <code>false</code>, clone only the node 
 *   itself (and its attributes, if it is an <code>Element</code>). 
 * @return The duplicate node.
 */
public Node cloneNode(boolean deep) {
	DocumentImpl clone = new DocumentImpl();
	if (documentElement != null) {
		clone.documentElement = (ElementImpl)(documentElement.cloneNode(deep));
		clone.documentElement.spreadOwnerDocument(clone);
	}
	return clone;
}

}

