/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.dom;

import java.util.Enumeration;
import java.util.Hashtable;

import org.w3c.dom.Attr;
import org.w3c.dom.DOMException;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

public class ElementImpl extends TreeNode implements Element {
	AttributeMap attributes;
/**
 * Constructor for ElementImpl
 */
public ElementImpl(DocumentImpl ownerDocument, String qualifiedName) {
	super(ownerDocument, qualifiedName);
	this.attributes = new AttributeMap(this);
}
public ElementImpl(DocumentImpl ownerDocument, String namespaceURI, String qualifiedName) {
	super(ownerDocument, namespaceURI, qualifiedName);
	this.attributes = new AttributeMap(this);
}
/**
 * The name of the element. For example, in: 
 * <pre> &lt;elementExample 
 * id="demo"&gt; ... &lt;/elementExample&gt; , </pre>
 *  <code>tagName</code> has 
 * the value <code>"elementExample"</code>. Note that this is 
 * case-preserving in XML, as are all of the operations of the DOM. The 
 * HTML DOM returns the <code>tagName</code> of an HTML element in the 
 * canonical uppercase form, regardless of the case in the source HTML 
 * document. 
 */
public String getTagName() {
	return getNodeName();
}
/**
 * Retrieves an attribute value by name.
 * @param nameThe name of the attribute to retrieve.
 * @return The <code>Attr</code> value as a string, or the empty string 
 *   if that attribute does not have a specified or default value.
 */
public String getAttribute(String name) {
	Attr attribute = getAttributeNode(name);
	return (attribute == null)?"":attribute.getValue();
}
/**
 * Adds a new attribute. If an attribute with that name is already present 
 * in the element, its value is changed to be that of the value 
 * parameter. This value is a simple string; it is not parsed as it is 
 * being set. So any markup (such as syntax to be recognized as an 
 * entity reference) is treated as literal text, and needs to be 
 * appropriately escaped by the implementation when it is written out. 
 * In order to assign an attribute value that contains entity 
 * references, the user must create an <code>Attr</code> node plus any 
 * <code>Text</code> and <code>EntityReference</code> nodes, build the 
 * appropriate subtree, and use <code>setAttributeNode</code> to assign 
 * it as the value of an attribute.
 * <br>To set an attribute with a qualified name and namespace URI, use 
 * the <code>setAttributeNS</code> method.
 * @param nameThe name of the attribute to create or alter.
 * @param valueValue to set in string form.
 * @exception DOMException
 *   INVALID_CHARACTER_ERR: Raised if the specified name contains an 
 *   illegal character.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 */
public void setAttribute(String name, String value) throws DOMException {
	attributes.setNamedItem(name, value);
}
/**
 * Removes an attribute by name. If the removed attribute is known to have 
 * a default value, an attribute immediately appears containing the 
 * default value as well as the corresponding namespace URI, local name, 
 * and prefix when applicable.
 * <br>To remove an attribute by local name and namespace URI, use the 
 * <code>removeAttributeNS</code> method.
 * @param nameThe name of the attribute to remove.
 * @exception DOMException
 *   NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 */
public void removeAttribute(String name) throws DOMException {
	attributes.removeNamedItem(name);
}
/**
 * Retrieves an attribute node by name.
 * <br>To retrieve an attribute node by qualified name and namespace URI, 
 * use the <code>getAttributeNodeNS</code> method.
 * @param nameThe name (<code>nodeName</code>) of the attribute to 
 *   retrieve.
 * @return The <code>Attr</code> node with the specified name (
 *   <code>nodeName</code>) or <code>null</code> if there is no such 
 *   attribute.
 */
public Attr getAttributeNode(String name) {
	return (Attr)attributes.getNamedItem(name);
}
/**
 * Adds a new attribute node. If an attribute with that name (
 * <code>nodeName</code>) is already present in the element, it is 
 * replaced by the new one.
 * <br>To add a new attribute node with a qualified name and namespace 
 * URI, use the <code>setAttributeNodeNS</code> method.
 * @param newAttrThe <code>Attr</code> node to add to the attribute list.
 * @return If the <code>newAttr</code> attribute replaces an existing 
 *   attribute, the replaced <code>Attr</code> node is returned, 
 *   otherwise <code>null</code> is returned.
 * @exception DOMException
 *   WRONG_DOCUMENT_ERR: Raised if <code>newAttr</code> was created from a 
 *   different document than the one that created the element.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 *   <br>INUSE_ATTRIBUTE_ERR: Raised if <code>newAttr</code> is already an 
 *   attribute of another <code>Element</code> object. The DOM user must 
 *   explicitly clone <code>Attr</code> nodes to re-use them in other 
 *   elements.
 */
public Attr setAttributeNode(Attr newAttr) throws DOMException {
	return (Attr)attributes.setNamedItem(newAttr);
}
/**
 * Removes the specified attribute node. If the removed <code>Attr</code> 
 * has a default value it is immediately replaced. The replacing 
 * attribute has the same namespace URI and local name, as well as the 
 * original prefix, when applicable.
 * @param oldAttrThe <code>Attr</code> node to remove from the attribute 
 *   list.
 * @return The <code>Attr</code> node that was removed.
 * @exception DOMException
 *   NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 *   <br>NOT_FOUND_ERR: Raised if <code>oldAttr</code> is not an attribute 
 *   of the element.
 */
public Attr removeAttributeNode(Attr oldAttr) throws DOMException {
	return (Attr)attributes.removeItem(oldAttr);
}

protected void getElementByTagName(String name, NodeListImpl list) {
	if (	("*".equals(name)) || //$NON-NLS-1$
			(getTagName().equals(name))) {
		list.addNode(this);
	}
	super.getElementByTagName(name, list);
}
/**
 * Retrieves an attribute value by local name and namespace URI. HTML-only 
 * DOM implementations do not need to implement this method.
 * @param namespaceURIThe namespace URI of the attribute to retrieve.
 * @param localNameThe local name of the attribute to retrieve.
 * @return The <code>Attr</code> value as a string, or the empty string 
 *   if that attribute does not have a specified or default value.
 * @since DOM Level 2
 */
public String getAttributeNS(String namespaceURI, String localName) {
	Attr attribute = getAttributeNodeNS(namespaceURI, localName);
	return (attribute == null)?"":attribute.getValue();
}
/**
 * Adds a new attribute. If an attribute with the same local name and 
 * namespace URI is already present on the element, its prefix is 
 * changed to be the prefix part of the <code>qualifiedName</code>, and 
 * its value is changed to be the <code>value</code> parameter. This 
 * value is a simple string; it is not parsed as it is being set. So any 
 * markup (such as syntax to be recognized as an entity reference) is 
 * treated as literal text, and needs to be appropriately escaped by the 
 * implementation when it is written out. In order to assign an 
 * attribute value that contains entity references, the user must create 
 * an <code>Attr</code> node plus any <code>Text</code> and 
 * <code>EntityReference</code> nodes, build the appropriate subtree, 
 * and use <code>setAttributeNodeNS</code> or 
 * <code>setAttributeNode</code> to assign it as the value of an 
 * attribute.
 * <br>HTML-only DOM implementations do not need to implement this method.
 * @param namespaceURIThe namespace URI of the attribute to create or 
 *   alter.
 * @param qualifiedNameThe qualified name of the attribute to create or 
 *   alter.
 * @param valueThe value to set in string form.
 * @exception DOMException
 *   INVALID_CHARACTER_ERR: Raised if the specified qualified name 
 *   contains an illegal character.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 *   <br>NAMESPACE_ERR: Raised if the <code>qualifiedName</code> is 
 *   malformed, if the <code>qualifiedName</code> has a prefix and the 
 *   <code>namespaceURI</code> is <code>null</code>, if the 
 *   <code>qualifiedName</code> has a prefix that is "xml" and the 
 *   <code>namespaceURI</code> is different from "
 *   http://www.w3.org/XML/1998/namespace", or if the 
 *   <code>qualifiedName</code> is "xmlns" and the 
 *   <code>namespaceURI</code> is different from "
 *   http://www.w3.org/2000/xmlns/".
 * @since DOM Level 2
 */
public void setAttributeNS(String namespaceURI,String qualifiedName,String value)
	throws DOMException {
		attributes.setNamedItemNS(namespaceURI, qualifiedName, value);
}
/**
 * Removes an attribute by local name and namespace URI. If the removed 
 * attribute has a default value it is immediately replaced. The 
 * replacing attribute has the same namespace URI and local name, as 
 * well as the original prefix.
 * <br>HTML-only DOM implementations do not need to implement this method.
 * @param namespaceURIThe namespace URI of the attribute to remove.
 * @param localNameThe local name of the attribute to remove.
 * @exception DOMException
 *   NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 * @since DOM Level 2
 */
public void removeAttributeNS(String namespaceURI, String localName)
	throws DOMException {
	attributes.removeNamedItemNS(namespaceURI, localName);
}
/**
 * Retrieves an <code>Attr</code> node by local name and namespace URI. 
 * HTML-only DOM implementations do not need to implement this method.
 * @param namespaceURIThe namespace URI of the attribute to retrieve.
 * @param localNameThe local name of the attribute to retrieve.
 * @return The <code>Attr</code> node with the specified attribute local 
 *   name and namespace URI or <code>null</code> if there is no such 
 *   attribute.
 * @since DOM Level 2
 */
public Attr getAttributeNodeNS(String namespaceURI, String localName) {
	return (Attr)attributes.getNamedItemNS(namespaceURI, localName);
}
/**
 * Adds a new attribute. If an attribute with that local name and that 
 * namespace URI is already present in the element, it is replaced by 
 * the new one.
 * <br>HTML-only DOM implementations do not need to implement this method.
 * @param newAttrThe <code>Attr</code> node to add to the attribute list.
 * @return If the <code>newAttr</code> attribute replaces an existing 
 *   attribute with the same local name and namespace URI, the replaced 
 *   <code>Attr</code> node is returned, otherwise <code>null</code> is 
 *   returned.
 * @exception DOMException
 *   WRONG_DOCUMENT_ERR: Raised if <code>newAttr</code> was created from a 
 *   different document than the one that created the element.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 *   <br>INUSE_ATTRIBUTE_ERR: Raised if <code>newAttr</code> is already an 
 *   attribute of another <code>Element</code> object. The DOM user must 
 *   explicitly clone <code>Attr</code> nodes to re-use them in other 
 *   elements.
 * @since DOM Level 2
 */
public Attr setAttributeNodeNS(Attr newAttr) throws DOMException {
	return setAttributeNode(newAttr);
}
void getElementsByTagNameNS(String namespaceURI, String localName, NodeListImpl list) {
	if (	(	("*".equals(namespaceURI)) || //$NON-NLS-1$
				(getNamespaceURI().equals(namespaceURI))) &&
			(	("*".equals(localName)) || //$NON-NLS-1$
				(getLocalName().equals(localName)))) {
		list.addNode(this);
	}
	super.getElementsByTagNameNS(namespaceURI, localName, list);
}
/**
 * Returns <code>true</code> when an attribute with a given name is 
 * specified on this element or has a default value, <code>false</code> 
 * otherwise.
 * @param nameThe name of the attribute to look for.
 * @return <code>true</code> if an attribute with the given name is 
 *   specified on this element or has a default value, <code>false</code>
 *    otherwise.
 * @since DOM Level 2
 */
public boolean hasAttribute(String name) {
	return getAttributeNode(name) != null;
}
/**
 * Returns <code>true</code> when an attribute with a given local name and 
 * namespace URI is specified on this element or has a default value, 
 * <code>false</code> otherwise. HTML-only DOM implementations do not 
 * need to implement this method.
 * @param namespaceURIThe namespace URI of the attribute to look for.
 * @param localNameThe local name of the attribute to look for.
 * @return <code>true</code> if an attribute with the given local name 
 *   and namespace URI is specified or has a default value on this 
 *   element, <code>false</code> otherwise.
 * @since DOM Level 2
 */
public boolean hasAttributeNS(String namespaceURI, String localName) {
	return getAttributeNodeNS(namespaceURI, localName) != null;
}
public short getNodeType() {
    return Node.ELEMENT_NODE;
}
/**
 * A <code>NamedNodeMap</code> containing the attributes of this node (if 
 * it is an <code>Element</code>) or <code>null</code> otherwise. 
 */
public NamedNodeMap getAttributes() {
	return attributes;
}
/**
 * Returns whether this node (if it is an element) has any attributes.
 * @return <code>true</code> if this node has any attributes, 
 *   <code>false</code> otherwise.
 * @since DOM Level 2
 */
public boolean hasAttributes() {
	return attributes.getLength() != 0;
}
/**
 * Returns a duplicate of this node, i.e., serves as a generic copy 
 * constructor for nodes. The duplicate node has no parent; (
 * <code>parentNode</code> is <code>null</code>.).
 * <br>Cloning an <code>Element</code> copies all attributes and their 
 * values, including those generated by the XML processor to represent 
 * defaulted attributes, but this method does not copy any text it 
 * contains unless it is a deep clone, since the text is contained in a 
 * child <code>Text</code> node. Cloning an <code>Attribute</code> 
 * directly, as opposed to be cloned as part of an <code>Element</code> 
 * cloning operation, returns a specified attribute (
 * <code>specified</code> is <code>true</code>). Cloning any other type 
 * of node simply returns a copy of this node.
 * <br>Note that cloning an immutable subtree results in a mutable copy, 
 * but the children of an <code>EntityReference</code> clone are readonly
 * . In addition, clones of unspecified <code>Attr</code> nodes are 
 * specified. And, cloning <code>Document</code>, 
 * <code>DocumentType</code>, <code>Entity</code>, and 
 * <code>Notation</code> nodes is implementation dependent.
 * @param deepIf <code>true</code>, recursively clone the subtree under 
 *   the specified node; if <code>false</code>, clone only the node 
 *   itself (and its attributes, if it is an <code>Element</code>). 
 * @return The duplicate node.
 */
public Node cloneNode(boolean deep) {
	ElementImpl clone = new ElementImpl(ownerDocument, namespaceURI, qualifiedName);
	Hashtable attTable = attributes.attributes;
	Enumeration keys = attTable.keys();
	AttrImpl att, attClone;
	Object key;
	while (keys.hasMoreElements()) {
		key = keys.nextElement();
		att = (AttrImpl)(attTable.get(key));
		attClone = (AttrImpl)(att.cloneNode(deep));
		attClone.ownerElement = clone;
		clone.setAttributeNode(attClone);
	}
	if (deep) {
		int len = getLength();
		Node child, childClone;
		for (int i = 0; i < len; i++) {
			child = item(i);
			childClone = child.cloneNode(deep);
			clone.appendChild(childClone);
		}
	}
	return clone;
}
protected void spreadOwnerDocument(DocumentImpl ownerDocument) {
	if (this.ownerDocument.ownerDocument == ownerDocument) return; // To avoid circular reference issues...
	super.spreadOwnerDocument(ownerDocument);
	Hashtable attTable = attributes.attributes;
	Enumeration keys = attTable.keys();
	AttrImpl att;
	Object key;
	while (keys.hasMoreElements()) {
		key = keys.nextElement();
		att = (AttrImpl)(attTable.get(key));
		att.spreadOwnerDocument(ownerDocument);
	}
	int len = getLength();
	for (int i = 0; i < len; i++) {
		((NodeImpl)item(i)).spreadOwnerDocument(ownerDocument);
	
	}
}
}

