/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.dom;

import org.w3c.dom.DOMException;
import org.w3c.dom.Node;

/**
 * Implement namespaces needs
 */
public abstract class NSNodeImpl extends NodeImpl {
	protected String	prefix,
						localName,
						namespaceURI,
						qualifiedName;
/**
 * Constructor for NSNodeImpl
 */
public NSNodeImpl(DocumentImpl ownerDocument) {
	this(ownerDocument, null, null);
}
public NSNodeImpl(DocumentImpl ownerDocument, String qualifiedName) {
	this(ownerDocument, null, qualifiedName);
}
public NSNodeImpl(DocumentImpl ownerDocument, String namespaceURI, String qualifiedName) {
	super(ownerDocument);
	if (namespaceURI != null) {
		// qualifiedName must be != null
		int index = qualifiedName.indexOf(':');
		if (index == -1) {
			this.prefix = null;
			this.localName = qualifiedName;
		} else {
			this.prefix = qualifiedName.substring(0,index);
			this.localName = qualifiedName.substring(index+1);
		}
	} else {
		this.prefix = null;
		this.localName = null;
	}
		
		
	this.namespaceURI = namespaceURI;
	this.qualifiedName = qualifiedName;
}
/**
 * The namespace prefix of this node, or <code>null</code> if it is 
 * unspecified.
 * <br>Note that setting this attribute, when permitted, changes the 
 * <code>nodeName</code> attribute, which holds the qualified name, as 
 * well as the <code>tagName</code> and <code>name</code> attributes of 
 * the <code>Element</code> and <code>Attr</code> interfaces, when 
 * applicable.
 * <br>Note also that changing the prefix of an attribute that is known to 
 * have a default value, does not make a new attribute with the default 
 * value and the original prefix appear, since the 
 * <code>namespaceURI</code> and <code>localName</code> do not change.
 * <br>For nodes of any type other than <code>ELEMENT_NODE</code> and 
 * <code>ATTRIBUTE_NODE</code> and nodes created with a DOM Level 1 
 * method, such as <code>createElement</code> from the 
 * <code>Document</code> interface, this is always <code>null</code>.
 * @exception DOMException
 *   INVALID_CHARACTER_ERR: Raised if the specified prefix contains an 
 *   illegal character.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 *   <br>NAMESPACE_ERR: Raised if the specified <code>prefix</code> is 
 *   malformed, if the <code>namespaceURI</code> of this node is 
 *   <code>null</code>, if the specified prefix is "xml" and the 
 *   <code>namespaceURI</code> of this node is different from "
 *   http://www.w3.org/XML/1998/namespace", if this node is an attribute 
 *   and the specified prefix is "xmlns" and the 
 *   <code>namespaceURI</code> of this node is different from "
 *   http://www.w3.org/2000/xmlns/", or if this node is an attribute and 
 *   the <code>qualifiedName</code> of this node is "xmlns" .
 * @since DOM Level 2
 */
public String getPrefix() {
	return prefix;
}
public void setPrefix(String prefix) throws DOMException {
	if ((namespaceURI == null) ||
		(	("xml".equals(prefix)) && //$NON-NLS-1$
			(!"http://www.w3.org/XML/1998/namespace".equals(namespaceURI))) || //$NON-NLS-1$
		(	(getNodeType() == Node.ATTRIBUTE_NODE) &&
			("xmlns".equals(prefix)) && //$NON-NLS-1$
			("http://www.w3.org/2000/xmlns/".equals(namespaceURI))) || //$NON-NLS-1$
		(	(getNodeType() == Node.ATTRIBUTE_NODE) &&
			("xmlns".equals(qualifiedName)))) { //$NON-NLS-1$
		throw new DOMException(DOMException.NAMESPACE_ERR, prefix);
	}

	this.prefix = prefix;
}
/**
 * The name of this node, depending on its type; see the table above. 
 */
public String getNodeName() {
	return qualifiedName;
}
/**
 * Returns the local part of the qualified name of this node.
 * <br>For nodes of any type other than <code>ELEMENT_NODE</code> and 
 * <code>ATTRIBUTE_NODE</code> and nodes created with a DOM Level 1 
 * method, such as <code>createElement</code> from the 
 * <code>Document</code> interface, this is always <code>null</code>.
 * @since DOM Level 2
 */
public String getLocalName() {
	return localName;
}
/**
 * The namespace URI of this node, or <code>null</code> if it is 
 * unspecified.
 * <br>This is not a computed value that is the result of a namespace 
 * lookup based on an examination of the namespace declarations in 
 * scope. It is merely the namespace URI given at creation time.
 * <br>For nodes of any type other than <code>ELEMENT_NODE</code> and 
 * <code>ATTRIBUTE_NODE</code> and nodes created with a DOM Level 1 
 * method, such as <code>createElement</code> from the 
 * <code>Document</code> interface, this is always <code>null</code>.Per 
 * the Namespaces in XML Specification  an attribute does not inherit 
 * its namespace from the element it is attached to. If an attribute is 
 * not explicitly given a namespace, it simply has no namespace.
 * @since DOM Level 2
 */
public String getNamespaceURI() {
	return namespaceURI;
}
}

