/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.dom;

import org.eclipse.ercp.xml.parser.EXmlMsg;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;


public abstract class NodeImpl implements Node, NodeList {
	protected DocumentImpl	ownerDocument;
/**
 * Constructor for NodeImpl
 */
public NodeImpl(DocumentImpl ownerDocument) {
	super();
	this.ownerDocument = ownerDocument;
}
/**
 * The value of this node, depending on its type; see the table above. 
 * When it is defined to be <code>null</code>, setting it has no effect.
 * @exception DOMException
 *   NO_MODIFICATION_ALLOWED_ERR: Raised when the node is readonly.
 * @exception DOMException
 *   DOMSTRING_SIZE_ERR: Raised when it would return more characters than 
 *   fit in a <code>DOMString</code> variable on the implementation 
 *   platform.
 */
public String getNodeValue() throws DOMException {
	return null;	// overridden in ChracterDataImpl and AttrImpl
}
public void setNodeValue(String nodeValue) throws DOMException {
	// overridden in ChracterDataImpl and AttrImpl
}
/**
 * The parent of this node. All nodes, except <code>Attr</code>, 
 * <code>Document</code>, <code>DocumentFragment</code>, 
 * <code>Entity</code>, and <code>Notation</code> may have a parent. 
 * However, if a node has just been created and not yet added to the 
 * tree, or if it has been removed from the tree, this is 
 * <code>null</code>.
 */
public Node getParentNode() {
	return null;
}
/**
 * A <code>NodeList</code> that contains all children of this node. If 
 * there are no children, this is a <code>NodeList</code> containing no 
 * nodes.
 */
public NodeList getChildNodes() {
	return this;
}
/**
 * The first child of this node. If there is no such node, this returns 
 * <code>null</code>.
 */
public Node getFirstChild() {
	return null;
}
/**
 * The last child of this node. If there is no such node, this returns 
 * <code>null</code>.
 */
public Node getLastChild() {
	return null;
}
/**
 * The node immediately preceding this node. If there is no such node, 
 * this returns <code>null</code>.
 */
public Node getPreviousSibling() {
	return null;
}
/**
 * The node immediately following this node. If there is no such node, 
 * this returns <code>null</code>.
 */
public Node getNextSibling() {
	return null;
}
/**
 * A <code>NamedNodeMap</code> containing the attributes of this node (if 
 * it is an <code>Element</code>) or <code>null</code> otherwise. 
 */
public NamedNodeMap getAttributes() {
	return null; // overridden in ElementImpl
}
/**
 * The <code>Document</code> object associated with this node. This is 
 * also the <code>Document</code> object used to create new nodes. When 
 * this node is a <code>Document</code> or a <code>DocumentType</code> 
 * which is not used with any <code>Document</code> yet, this is 
 * <code>null</code>.
 * @version DOM Level 2
 */
public Document getOwnerDocument() {
	return ownerDocument;
}
/*
 * Sets the ownerDocument (internal API)
 */
void setOwnerDocument(DocumentImpl ownerDocument) {
	this.ownerDocument = ownerDocument;
}
protected void spreadOwnerDocument(DocumentImpl ownerDocument) {
	this.ownerDocument = ownerDocument;
}
/**
 * Inserts the node <code>newChild</code> before the existing child node 
 * <code>refChild</code>. If <code>refChild</code> is <code>null</code>, 
 * insert <code>newChild</code> at the end of the list of children.
 * <br>If <code>newChild</code> is a <code>DocumentFragment</code> object, 
 * all of its children are inserted, in the same order, before 
 * <code>refChild</code>. If the <code>newChild</code> is already in the 
 * tree, it is first removed.
 * @param newChildThe node to insert.
 * @param refChildThe reference node, i.e., the node before which the new 
 *   node must be inserted.
 * @return The node being inserted.
 * @exception DOMException
 *   HIERARCHY_REQUEST_ERR: Raised if this node is of a type that does not 
 *   allow children of the type of the <code>newChild</code> node, or if 
 *   the node to insert is one of this node's ancestors.
 *   <br>WRONG_DOCUMENT_ERR: Raised if <code>newChild</code> was created 
 *   from a different document than the one that created this node.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly or 
 *   if the parent of the node being inserted is readonly.
 *   <br>NOT_FOUND_ERR: Raised if <code>refChild</code> is not a child of 
 *   this node.
 */
public Node insertBefore(Node newChild, Node refChild) throws DOMException {
	throw new DOMException(DOMException.HIERARCHY_REQUEST_ERR, EXmlMsg.getDefault().getString(EXmlMsg.INSERTING_NODE)); //$NON-NLS-1$
}
/**
 * Replaces the child node <code>oldChild</code> with <code>newChild</code>
 *  in the list of children, and returns the <code>oldChild</code> node.
 * <br>If <code>newChild</code> is a <code>DocumentFragment</code> object, 
 * <code>oldChild</code> is replaced by all of the 
 * <code>DocumentFragment</code> children, which are inserted in the 
 * same order. If the <code>newChild</code> is already in the tree, it 
 * is first removed.
 * @param newChildThe new node to put in the child list.
 * @param oldChildThe node being replaced in the list.
 * @return The node replaced.
 * @exception DOMException
 *   HIERARCHY_REQUEST_ERR: Raised if this node is of a type that does not 
 *   allow children of the type of the <code>newChild</code> node, or if 
 *   the node to put in is one of this node's ancestors.
 *   <br>WRONG_DOCUMENT_ERR: Raised if <code>newChild</code> was created 
 *   from a different document than the one that created this node.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node or the parent of 
 *   the new node is readonly.
 *   <br>NOT_FOUND_ERR: Raised if <code>oldChild</code> is not a child of 
 *   this node.
 */
public Node replaceChild(Node newChild, Node oldChild) throws DOMException {
	throw new DOMException(DOMException.HIERARCHY_REQUEST_ERR,EXmlMsg.getDefault().getString(EXmlMsg.REPLACING_NODE));
}
/**
 * Removes the child node indicated by <code>oldChild</code> from the list 
 * of children, and returns it.
 * @param oldChildThe node being removed.
 * @return The node removed.
 * @exception DOMException
 *   NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 *   <br>NOT_FOUND_ERR: Raised if <code>oldChild</code> is not a child of 
 *   this node.
 */
public Node removeChild(Node oldChild) throws DOMException {
	throw new DOMException(DOMException.HIERARCHY_REQUEST_ERR,EXmlMsg.getDefault().getString(EXmlMsg.REMOVING_NODE)); //$NON-NLS-1$
}
/**
 * Adds the node <code>newChild</code> to the end of the list of children 
 * of this node. If the <code>newChild</code> is already in the tree, it 
 * is first removed.
 * @param newChildThe node to add.If it is a <code>DocumentFragment</code>
 *    object, the entire contents of the document fragment are moved 
 *   into the child list of this node
 * @return The node added.
 * @exception DOMException
 *   HIERARCHY_REQUEST_ERR: Raised if this node is of a type that does not 
 *   allow children of the type of the <code>newChild</code> node, or if 
 *   the node to append is one of this node's ancestors.
 *   <br>WRONG_DOCUMENT_ERR: Raised if <code>newChild</code> was created 
 *   from a different document than the one that created this node.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 */
public Node appendChild(Node newChild) throws DOMException {
	throw new DOMException(DOMException.HIERARCHY_REQUEST_ERR,EXmlMsg.getDefault().getString(EXmlMsg.APPENDING_NODE)); //$NON-NLS-1$
}
/**
 * Returns whether this node has any children.
 * @return  <code>true</code> if this node has any children, 
 *   <code>false</code> otherwise.
 */
public boolean hasChildNodes() {
	return false;
}
/**
 * Puts all <code>Text</code> nodes in the full depth of the sub-tree 
 * underneath this <code>Node</code>, including attribute nodes, into a 
 * "normal" form where only structure (e.g., elements, comments, 
 * processing instructions, CDATA sections, and entity references) 
 * separates <code>Text</code> nodes, i.e., there are neither adjacent 
 * <code>Text</code> nodes nor empty <code>Text</code> nodes. This can 
 * be used to ensure that the DOM view of a document is the same as if 
 * it were saved and re-loaded, and is useful when operations (such as 
 * XPointer  lookups) that depend on a particular document tree 
 * structure are to be used.In cases where the document contains 
 * <code>CDATASections</code>, the normalize operation alone may not be 
 * sufficient, since XPointers do not differentiate between 
 * <code>Text</code> nodes and <code>CDATASection</code> nodes.
 * @version DOM Level 2
 */
public void normalize() {
	// ElementImpl and DocumentImpl overrides this method
}
/**
 * Tests whether the DOM implementation implements a specific feature and 
 * that feature is supported by this node.
 * @param featureThe name of the feature to test. This is the same name 
 *   which can be passed to the method <code>hasFeature</code> on 
 *   <code>DOMImplementation</code>.
 * @param versionThis is the version number of the feature to test. In 
 *   Level 2, version 1, this is the string "2.0". If the version is not 
 *   specified, supporting any version of the feature will cause the 
 *   method to return <code>true</code>.
 * @return Returns <code>true</code> if the specified feature is 
 *   supported on this node, <code>false</code> otherwise.
 * @since DOM Level 2
 */
public boolean isSupported(String feature, String version) {
	return ownerDocument.getImplementation().hasFeature(feature, version);
}
/**
 * The namespace URI of this node, or <code>null</code> if it is 
 * unspecified.
 * <br>This is not a computed value that is the result of a namespace 
 * lookup based on an examination of the namespace declarations in 
 * scope. It is merely the namespace URI given at creation time.
 * <br>For nodes of any type other than <code>ELEMENT_NODE</code> and 
 * <code>ATTRIBUTE_NODE</code> and nodes created with a DOM Level 1 
 * method, such as <code>createElement</code> from the 
 * <code>Document</code> interface, this is always <code>null</code>.Per 
 * the Namespaces in XML Specification  an attribute does not inherit 
 * its namespace from the element it is attached to. If an attribute is 
 * not explicitly given a namespace, it simply has no namespace.
 * @since DOM Level 2
 */
public String getNamespaceURI() {
	return null;
}
/**
 * The namespace prefix of this node, or <code>null</code> if it is 
 * unspecified.
 * <br>Note that setting this attribute, when permitted, changes the 
 * <code>nodeName</code> attribute, which holds the qualified name, as 
 * well as the <code>tagName</code> and <code>name</code> attributes of 
 * the <code>Element</code> and <code>Attr</code> interfaces, when 
 * applicable.
 * <br>Note also that changing the prefix of an attribute that is known to 
 * have a default value, does not make a new attribute with the default 
 * value and the original prefix appear, since the 
 * <code>namespaceURI</code> and <code>localName</code> do not change.
 * <br>For nodes of any type other than <code>ELEMENT_NODE</code> and 
 * <code>ATTRIBUTE_NODE</code> and nodes created with a DOM Level 1 
 * method, such as <code>createElement</code> from the 
 * <code>Document</code> interface, this is always <code>null</code>.
 * @exception DOMException
 *   INVALID_CHARACTER_ERR: Raised if the specified prefix contains an 
 *   illegal character.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 *   <br>NAMESPACE_ERR: Raised if the specified <code>prefix</code> is 
 *   malformed, if the <code>namespaceURI</code> of this node is 
 *   <code>null</code>, if the specified prefix is "xml" and the 
 *   <code>namespaceURI</code> of this node is different from "
 *   http://www.w3.org/XML/1998/namespace", if this node is an attribute 
 *   and the specified prefix is "xmlns" and the 
 *   <code>namespaceURI</code> of this node is different from "
 *   http://www.w3.org/2000/xmlns/", or if this node is an attribute and 
 *   the <code>qualifiedName</code> of this node is "xmlns" .
 * @since DOM Level 2
 */
public String getPrefix() {
	return null;
}
public void setPrefix(String prefix) throws DOMException {
	throw new DOMException(DOMException.NAMESPACE_ERR, prefix);
}
/**
 * Returns the local part of the qualified name of this node.
 * <br>For nodes of any type other than <code>ELEMENT_NODE</code> and 
 * <code>ATTRIBUTE_NODE</code> and nodes created with a DOM Level 1 
 * method, such as <code>createElement</code> from the 
 * <code>Document</code> interface, this is always <code>null</code>.
 * @since DOM Level 2
 */
public String getLocalName() {
	return null;
}
/**
 * Returns whether this node (if it is an element) has any attributes.
 * @return <code>true</code> if this node has any attributes, 
 *   <code>false</code> otherwise.
 * @since DOM Level 2
 */
public boolean hasAttributes() {
	return false;	// Overwrite ElementImpl
}
/**
 * Returns the <code>index</code>th item in the collection. If 
 * <code>index</code> is greater than or equal to the number of nodes in 
 * the list, this returns <code>null</code>.
 * @param indexIndex into the collection.
 * @return The node at the <code>index</code>th position in the 
 *   <code>NodeList</code>, or <code>null</code> if that is not a valid 
 *   index.
 */
public Node item(int index) {
	return null;
}
/**
 * The number of nodes in the list. The range of valid child node indices 
 * is 0 to <code>length-1</code> inclusive. 
 */
public int getLength() {
	return 0;
}
}

