/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.dom;

import org.w3c.dom.DOMException;
import org.w3c.dom.Node;
import org.w3c.dom.Text;

public class TextImpl extends CharacterDataImpl implements Text {
/**
 * Constructor for TextImpl
 */
public TextImpl(DocumentImpl ownerDocument, String data) {
	super(ownerDocument, data);
}
/**
 * Breaks this node into two nodes at the specified <code>offset</code>, 
 * keeping both in the tree as siblings. After being split, this node 
 * will contain all the content up to the <code>offset</code> point. A 
 * new node of the same type, which contains all the content at and 
 * after the <code>offset</code> point, is returned. If the original 
 * node had a parent node, the new node is inserted as the next sibling 
 * of the original node. When the <code>offset</code> is equal to the 
 * length of this node, the new node has no data.
 * @param offsetThe 16-bit unit offset at which to split, starting from 
 *   <code>0</code>.
 * @return The new node, of the same type as this node.
 * @exception DOMException
 *   INDEX_SIZE_ERR: Raised if the specified offset is negative or greater 
 *   than the number of 16-bit units in <code>data</code>.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 */
public Text splitText(int offset) throws DOMException {
	String data = getData();
	if (offset < 0 || offset > data.length() - 1) {
		throw new DOMException(DOMException.INDEX_SIZE_ERR, null);
	}
		
	TextImpl newText = 
		(TextImpl) ownerDocument.createTextNode(data.substring(offset));
	setData(data.substring(0, offset));
	Node parent = getParentNode();
	if (parent != null) {
		parent.insertBefore(newText, getNextSibling());
	}
	return newText;
}
public short getNodeType() {
	return Node.TEXT_NODE;
}
public String getNodeName() {
	return "#text"; //$NON-NLS-1$
}
/**
 * Returns a duplicate of this node, i.e., serves as a generic copy 
 * constructor for nodes. The duplicate node has no parent; (
 * <code>parentNode</code> is <code>null</code>.).
 * <br>Cloning an <code>Element</code> copies all attributes and their 
 * values, including those generated by the XML processor to represent 
 * defaulted attributes, but this method does not copy any text it 
 * contains unless it is a deep clone, since the text is contained in a 
 * child <code>Text</code> node. Cloning an <code>Attribute</code> 
 * directly, as opposed to be cloned as part of an <code>Element</code> 
 * cloning operation, returns a specified attribute (
 * <code>specified</code> is <code>true</code>). Cloning any other type 
 * of node simply returns a copy of this node.
 * <br>Note that cloning an immutable subtree results in a mutable copy, 
 * but the children of an <code>EntityReference</code> clone are readonly
 * . In addition, clones of unspecified <code>Attr</code> nodes are 
 * specified. And, cloning <code>Document</code>, 
 * <code>DocumentType</code>, <code>Entity</code>, and 
 * <code>Notation</code> nodes is implementation dependent.
 * @param deepIf <code>true</code>, recursively clone the subtree under 
 *   the specified node; if <code>false</code>, clone only the node 
 *   itself (and its attributes, if it is an <code>Element</code>). 
 * @return The duplicate node.
 */
public Node cloneNode(boolean deep) {
	return new TextImpl(ownerDocument, data);
}
}

