/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.dom;

import org.eclipse.ercp.xml.parser.EXmlMsg;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

/**
 * Support previous and next siblings
 */
public abstract class TreeNode extends NSNodeImpl {
	protected TreeNode	nextSibling,
						previousSibling,
						parentNode,
						firstChild,
						lastChild;
	protected int		length;
/**
 * Constructor for SiblingNode
 */
public TreeNode(DocumentImpl ownerDocument) {
	super(ownerDocument);
	this.nextSibling = null;
	this.previousSibling = null;
	this.parentNode = null;
	this.firstChild = null;
	this.lastChild = null;
	this.length = 0;
}
public TreeNode(DocumentImpl ownerDocument, String qualifiedName) {
	super(ownerDocument, qualifiedName);
	this.nextSibling = null;
	this.previousSibling = null;
	this.parentNode = null;
	this.firstChild = null;
	this.lastChild = null;
	this.length = 0;
}
public TreeNode(DocumentImpl ownerDocument, String namespaceURI, String qualifiedName) {
	super(ownerDocument, namespaceURI, qualifiedName);
	this.nextSibling = null;
	this.previousSibling = null;
	this.parentNode = null;
	this.firstChild = null;
	this.lastChild = null;
	this.length = 0;
}
/**
 * Inserts the node <code>newChild</code> before the existing child node 
 * <code>refChild</code>. If <code>refChild</code> is <code>null</code>, 
 * insert <code>newChild</code> at the end of the list of children.
 * <br>If <code>newChild</code> is a <code>DocumentFragment</code> object, 
 * all of its children are inserted, in the same order, before 
 * <code>refChild</code>. If the <code>newChild</code> is already in the 
 * tree, it is first removed.
 * @param newChildThe node to insert.
 * @param refChildThe reference node, i.e., the node before which the new 
 *   node must be inserted.
 * @return The node being inserted.
 * @exception DOMException
 *   HIERARCHY_REQUEST_ERR: Raised if this node is of a type that does not 
 *   allow children of the type of the <code>newChild</code> node, or if 
 *   the node to insert is one of this node's ancestors.
 *   <br>WRONG_DOCUMENT_ERR: Raised if <code>newChild</code> was created 
 *   from a different document than the one that created this node.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly or 
 *   if the parent of the node being inserted is readonly.
 *   <br>NOT_FOUND_ERR: Raised if <code>refChild</code> is not a child of 
 *   this node.
 */
public Node insertBefore(Node newChild, Node refChild) throws DOMException {
	try {
		if (newChild == null) return null;
		// Check that the node to insert is not one of the receiver ancestors
		checkHierarchy(newChild);
		// Check that they have been created by the same document!
		checkOwnerDocument(newChild);
		TreeNode newChildImpl = (TreeNode)newChild;
		if (refChild == null) {
			if (firstChild == null) { 						// Very first child!
				this.firstChild = this.lastChild = newChildImpl;
			} else {										// Append at the end of the child list
				lastChild.nextSibling = newChildImpl;
				newChildImpl.previousSibling = lastChild;	//////
				this.lastChild = newChildImpl;
			}
			// Sets the parent
			newChildImpl.parentNode = this;
		} else {
			// Check that refChild is a child of the receiver
			if (refChild.getParentNode() != this) {
				throw new DOMException(DOMException.NOT_FOUND_ERR, EXmlMsg.getDefault().getString(EXmlMsg.NOT_FOUND_ERR));
			}
			TreeNode refChildImpl = (TreeNode)refChild;
			// Insert
			newChildImpl.previousSibling = refChildImpl.previousSibling;
			newChildImpl.nextSibling = refChildImpl;
			refChildImpl.previousSibling = newChildImpl;
			// Sets the parent
			newChildImpl.parentNode = this;
			// Check if refChild was the first child
			if (firstChild == refChildImpl) {
				this.firstChild = newChildImpl;
			}
		}
		length++;
		return newChild;
	} catch (ClassCastException e) {
		throw new DOMException(DOMException.HIERARCHY_REQUEST_ERR, EXmlMsg.getDefault().getString(EXmlMsg.HIERARCHY_REQUEST_ERR));
	}
}
/**
 * Removes the child node indicated by <code>oldChild</code> from the list 
 * of children, and returns it.
 * @param oldChildThe node being removed.
 * @return The node removed.
 * @exception DOMException
 *   NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 *   <br>NOT_FOUND_ERR: Raised if <code>oldChild</code> is not a child of 
 *   this node.
 */
public Node removeChild(Node oldChild) throws DOMException {
	TreeNode oldChildImpl = (TreeNode)oldChild;

	if (oldChildImpl.previousSibling != null) {
		oldChildImpl.previousSibling.nextSibling = oldChildImpl.nextSibling;
	}
	if (oldChildImpl.nextSibling != null) {
		oldChildImpl.nextSibling.previousSibling = oldChildImpl.previousSibling;
	}
	if (oldChildImpl.parentNode.firstChild == oldChild) {
		oldChildImpl.parentNode.firstChild = oldChildImpl.nextSibling;
	}
	if (oldChildImpl.parentNode.lastChild == oldChild) {
		oldChildImpl.parentNode.lastChild = oldChildImpl.previousSibling;
	}
	length--;

	return oldChild;
}
/**
 * Adds the node <code>newChild</code> to the end of the list of children 
 * of this node. If the <code>newChild</code> is already in the tree, it 
 * is first removed.
 * @param newChildThe node to add.If it is a <code>DocumentFragment</code>
 *    object, the entire contents of the document fragment are moved 
 *   into the child list of this node
 * @return The node added.
 * @exception DOMException
 *   HIERARCHY_REQUEST_ERR: Raised if this node is of a type that does not 
 *   allow children of the type of the <code>newChild</code> node, or if 
 *   the node to append is one of this node's ancestors.
 *   <br>WRONG_DOCUMENT_ERR: Raised if <code>newChild</code> was created 
 *   from a different document than the one that created this node.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
 */
public Node appendChild(Node newChild) throws DOMException {
	return insertBefore(newChild, null);
}
/**
 * Replaces the child node <code>oldChild</code> with <code>newChild</code>
 *  in the list of children, and returns the <code>oldChild</code> node.
 * <br>If <code>newChild</code> is a <code>DocumentFragment</code> object, 
 * <code>oldChild</code> is replaced by all of the 
 * <code>DocumentFragment</code> children, which are inserted in the 
 * same order. If the <code>newChild</code> is already in the tree, it 
 * is first removed.
 * @param newChildThe new node to put in the child list.
 * @param oldChildThe node being replaced in the list.
 * @return The node replaced.
 * @exception DOMException
 *   HIERARCHY_REQUEST_ERR: Raised if this node is of a type that does not 
 *   allow children of the type of the <code>newChild</code> node, or if 
 *   the node to put in is one of this node's ancestors.
 *   <br>WRONG_DOCUMENT_ERR: Raised if <code>newChild</code> was created 
 *   from a different document than the one that created this node.
 *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node or the parent of 
 *   the new node is readonly.
 *   <br>NOT_FOUND_ERR: Raised if <code>oldChild</code> is not a child of 
 *   this node.
 */
public Node replaceChild(Node newChild, Node oldChild) throws DOMException {
	insertBefore(newChild, oldChild);
	removeChild(oldChild);
	return oldChild;
}
/**
 * The first child of this node. If there is no such node, this returns 
 * <code>null</code>.
 */
public Node getFirstChild() {
	return firstChild;
}
/**
 * The last child of this node. If there is no such node, this returns 
 * <code>null</code>.
 */
public Node getLastChild() {
	return lastChild;
}
/**
 * The node immediately preceding this node. If there is no such node, 
 * this returns <code>null</code>.
 */
public Node getPreviousSibling() {
	return previousSibling;
}
/**
 * The node immediately following this node. If there is no such node, 
 * this returns <code>null</code>.
 */
public Node getNextSibling() {
	return nextSibling;
}
/**
 * Returns whether this node has any children.
 * @return  <code>true</code> if this node has any children, 
 *   <code>false</code> otherwise.
 */
public boolean hasChildNodes() {
	return firstChild != null;
}
/**
 * The number of nodes in the list. The range of valid child node indices 
 * is 0 to <code>length-1</code> inclusive. 
 */
public int getLength() {
	return length;
}
/**
 * The parent of this node. All nodes, except <code>Attr</code>, 
 * <code>Document</code>, <code>DocumentFragment</code>, 
 * <code>Entity</code>, and <code>Notation</code> may have a parent. 
 * However, if a node has just been created and not yet added to the 
 * tree, or if it has been removed from the tree, this is 
 * <code>null</code>.
 */
public Node getParentNode() {
	return parentNode;
}
/**
 * Returns the <code>index</code>th item in the collection. If 
 * <code>index</code> is greater than or equal to the number of nodes in 
 * the list, this returns <code>null</code>.
 * @param index Index into the collection.
 * @return The node at the <code>index</code>th position in the 
 *   <code>NodeList</code>, or <code>null</code> if that is not a valid 
 *   index.
 */
public Node item(int index) {
	TreeNode node = firstChild;
	for(int i=0; i<index && node!=null; ++i) {
	  node = node.nextSibling;
	}
	return node;
}
/**
 * Returns a <code>NodeList</code> of all descendant <code>Elements</code> 
 * with a given tag name, in the order in which they are encountered in 
 * a preorder traversal of this <code>Element</code> tree.
 * @param nameThe name of the tag to match on. The special value "*" 
 *   matches all tags.
 * @return A list of matching <code>Element</code> nodes.
 */
public NodeList getElementsByTagName(String name) {
	NodeListImpl list = new NodeListImpl();
	getElementByTagName(name, list);
	return list;
}
protected void getElementByTagName(String name, NodeListImpl list) {
	TreeNode node = firstChild;
	while (node != null) {
		if (node.getNodeType() == Node.ELEMENT_NODE) {
			((ElementImpl)node).getElementByTagName(name, list);
		}
		node = node.nextSibling;
	}
}
/**
 * Returns a <code>NodeList</code> of all the descendant 
 * <code>Elements</code> with a given local name and namespace URI in 
 * the order in which they are encountered in a preorder traversal of 
 * this <code>Element</code> tree.
 * <br>HTML-only DOM implementations do not need to implement this method.
 * @param namespaceURIThe namespace URI of the elements to match on. The 
 *   special value "*" matches all namespaces.
 * @param localNameThe local name of the elements to match on. The 
 *   special value "*" matches all local names.
 * @return A new <code>NodeList</code> object containing all the matched 
 *   <code>Elements</code>.
 * @since DOM Level 2
 */
public NodeList getElementsByTagNameNS(String namespaceURI, String localName) {
	NodeListImpl list = new NodeListImpl();
	getElementsByTagNameNS(namespaceURI, localName, list);
	return list;
}
void getElementsByTagNameNS(String namespaceURI, String localName, NodeListImpl list) {
	TreeNode node = firstChild;
	while (node != null) {
		if (node.getNodeType() == Node.ELEMENT_NODE) {
			((ElementImpl)node).getElementsByTagNameNS(namespaceURI, localName, list);
		}
		node = node.nextSibling;
	}
}
/**
 * Puts all <code>Text</code> nodes in the full depth of the sub-tree 
 * underneath this <code>Node</code>, including attribute nodes, into a 
 * "normal" form where only structure (e.g., elements, comments, 
 * processing instructions, CDATA sections, and entity references) 
 * separates <code>Text</code> nodes, i.e., there are neither adjacent 
 * <code>Text</code> nodes nor empty <code>Text</code> nodes. This can 
 * be used to ensure that the DOM view of a document is the same as if 
 * it were saved and re-loaded, and is useful when operations (such as 
 * XPointer  lookups) that depend on a particular document tree 
 * structure are to be used.In cases where the document contains 
 * <code>CDATASections</code>, the normalize operation alone may not be 
 * sufficient, since XPointers do not differentiate between 
 * <code>Text</code> nodes and <code>CDATASection</code> nodes.
 * @version DOM Level 2
 */
public void normalize() {
	Node child = getFirstChild();
	Node sibling;
	Text lastText = null;
	while (child != null) {
		sibling = child.getNextSibling();
		switch (child.getNodeType()) {
			case Node.TEXT_NODE:
				if (child.getNodeValue().length() == 0) {
					// Remove the empty Text nodes
					this.removeChild(child);
				} else				
				if (lastText == null) {
					lastText = (Text)child;
				} else {
					lastText.appendData(child.getNodeValue());
					this.removeChild(child);
				}
				break;
			case Node.ELEMENT_NODE:
				child.normalize();
				break;
			default:
				break;
		}
		child = sibling;
	}
}
/*
 * Checks that the receiver and teh argument have been created by the same document!
 * @exception DOMException
 *   WRONG_DOCUMENT_ERR: Raised if <code>newChild</code> was created 
 *   from a different document than the one that created this node.
 */
private void checkOwnerDocument(Node childNode) throws DOMException {
	Document childDoc = childNode.getOwnerDocument();
	Document myDoc = getOwnerDocument();
	if (myDoc == null && (this instanceof DocumentImpl)) {
		myDoc = (DocumentImpl)this;
	}
	if (childDoc == null) {
		((NodeImpl)childNode).setOwnerDocument((DocumentImpl)myDoc);
		return;
	}
	if (childDoc != myDoc) {
		throw new DOMException(DOMException.WRONG_DOCUMENT_ERR, EXmlMsg.getDefault().getString(EXmlMsg.WRONG_DOCUMENT_ERR));
	}
	
}
/*
 * Check that the node to insert is not one of the receiver ancestors
 * 
 * @exception DOMException
 *   HIERARCHY_REQUEST_ERR: Raised if this node is of a type that does not 
 *   allow children of the type of the <code>newChild</code> node, or if 
 *   the node to insert is one of this node's ancestors.
 */
private void checkHierarchy(Node childNode) {
	if (!(childNode instanceof TreeNode)) {
		throw new DOMException(DOMException.HIERARCHY_REQUEST_ERR, EXmlMsg.getDefault().getString(EXmlMsg.HIERARCHY_REQUEST_ERR));
	}
	Node parent = this;
	while (parent != null) {
		if (parent == childNode)
			throw new DOMException(DOMException.HIERARCHY_REQUEST_ERR, EXmlMsg.getDefault().getString(EXmlMsg.HIERARCHY_REQUEST_ERR));
		parent = parent.getParentNode();
	}	
}
}

