/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.io;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.Hashtable;

import org.eclipse.ercp.xml.parser.EXmlMsg;

/**
 * Implementation of AbstractIOSpecificCall against Core and Higher
 */
public class IOSpecificCall extends AbstractIOSpecificCall {

	static final IOSpecificCall Default = new IOSpecificCall();

	public static IOSpecificCall getDefault() {
		return Default;
	}

	private IOSpecificCall() {
		super();
	}

	/**
	 * Constructs a new InputStream on the file named <code>fileName</code>.
	 */
	public InputStream openFileInputStream(String filename) throws IOException {
		return new FileInputStream(filename);
	}

	/**
	 * Reads a property file and returns a corresponding Hashtable. If the
	 * properpyfile is not defined, returns null
	 */
	public Hashtable readPropertyFile(final Class aClass,
			final String resource, final Hashtable table) {
		// must encapuslate the file access inside a doPrivileged in case anyone
		// on the call stack does not have enough permissions
		try {
			AccessController.doPrivileged(new PrivilegedExceptionAction() {
				public java.lang.Object run() throws IOException {
					InputStream is = aClass.getResourceAsStream(resource);
					try {
						readPropertyFile(is, table);
					} finally {
						try {
							is.close();
						} catch (IOException e) {
							// NOP
						}
					}
					return null;
				}
			});
		} catch (PrivilegedActionException ex) {
			// NOP
		}
		return table;
	}

	/**
	 * Reads a property file and returns a corresponding Hashtable. If the
	 * properpyfile is not defined, returns null
	 */
	public Hashtable readPropertyFile(final String filename, final Hashtable table) {
		final Hashtable[] result = new Hashtable[1];
		// must encapuslate the file access inside a doPrivileged in case anyone
		// on the call stack does not have enough permissions

		try {
			AccessController.doPrivileged(new PrivilegedExceptionAction() {
				public java.lang.Object run() throws IOException {

					InputStream is = openFileInputStream(filename);
					try {
						result[0] = (is != null) ? new PropertyFileReader()
								.read(is, table) : null;
					} catch (IOException e) {
						// NOP
					} finally {
						try {
							is.close();
						} catch (IOException e) {
							// NOP
						}
					}
					return null;
				}
			});
		} catch (PrivilegedActionException ex) {
			// NOP
		}
		return result[0];
	}

	public String getSystemProperty(final String property) {
		return (String) AccessController.doPrivileged(new PrivilegedAction() {
			public Object run() {
				return System.getProperty(property);
			}
		});

	}

	public Reader getReaderFromSystemID(String systemID, String encoding)
			throws IOException {
		try {
			URL url = new URL(systemID);
			InputStream is = url.openStream();
			return new XmlInputStreamReader(is, encoding);
		} catch (MalformedURLException e) {
			throw new IllegalArgumentException(EXmlMsg.getDefault().getString(EXmlMsg.SYSTEMID_SET_ERROR, systemID));
		}

	}
}
