/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.io;

import java.io.IOException;
import java.io.InputStream;
import java.util.Hashtable;


/**
 * Because CLDC doesn't provide the properties class, we have to write
 * our own Property file reader...
 */
public class PropertyFileReader {

	private static final int INIT_STATE = 0;
	private static final int SLASH_MET_STATE = 1;
	private static final int UNICODE_STATE = 2;
	private static final int RC_STATE = 3;
	private static final int END_OF_PROP_STATE = 4;
	private static final int SKIP_NEXT_CHAR_STATE = 4;
/**
 * Constructor
 */
public PropertyFileReader() {
	super();
}
public Hashtable read(InputStream in, Hashtable table) throws IOException {
	return readBIS(new SimpleBufferedInputStream(in), table);
}
Hashtable readBIS(SimpleBufferedInputStream in, Hashtable table) throws IOException {
	int currentState = INIT_STATE;
	int unicodeValue = 0;
	int unicodeLength = 0;
	int c;
	char[] buf = new char[40];
	int offset = 0;
	int keyLength = -1;
	boolean firstChar = true;

	while ((c = in.read()) != -1) {
		if (offset == buf.length) {
			char[] newBuf = new char[buf.length * 2];
			System.arraycopy(buf, 0, newBuf, 0, offset);
			buf = newBuf;
		}
		if (currentState == UNICODE_STATE) {
			int digit = Character.digit((char)c, 16);
			if (digit >= 0) {
				unicodeValue = (unicodeValue << 4) + digit;
				if (++unicodeLength < 4) continue;
			}
			currentState = INIT_STATE;
			buf[offset++] = (char)unicodeValue;
			if (c != '\n') continue;
		}
		if (currentState == SLASH_MET_STATE) {
			currentState = INIT_STATE;
			switch (c) {
				case '\r':
					currentState = RC_STATE; // Look for a following \n
					continue;
				case '\n':
					currentState = SKIP_NEXT_CHAR_STATE; // Ignore whitespace on the next line
					continue;
				case 'b':
					c = '\b';
					break;
				case 'f':
					c = '\f';
					break;
				case 'n':
					c = '\n';
					break;
				case 'r':
					c = '\r';
					break;
				case 't':
					c = '\t';
					break;
				case 'u':
					currentState = UNICODE_STATE;
					unicodeValue = unicodeLength = 0;
					continue;
			}
		} else {
			switch (c) {
				case '#':
				case '!':
					if (firstChar) {
						while ((c = in.read()) != -1)
							if (c == '\r' || c == '\n') break;
						continue;
					}
					break;
				case '\n':
					if (currentState == RC_STATE) { // Part of a \r\n sequence
						currentState = SKIP_NEXT_CHAR_STATE; // Ignore whitespace on the next line
						continue;
					}
					// fall into the next case
				case '\r':
					currentState = INIT_STATE;
					firstChar = true;
					if (keyLength >= 0) {
						String temp = new String(buf, 0, offset);
						table.put(temp.substring(0,keyLength), temp.substring(keyLength));
					}
					keyLength = -1;
					offset = 0;
					continue;
				case '\\':
					currentState = SLASH_MET_STATE;
					continue;
				case ':':
				case '=':
					if (keyLength == -1) { // if parsing the key
						keyLength = offset;
						continue;
					}
					break;
			}
			if (isWhitespace((char)c)) {
				if (currentState == RC_STATE) currentState = SKIP_NEXT_CHAR_STATE;
				// if key length == 0 or value length == 0
				if (offset == 0 || offset == keyLength || currentState == SKIP_NEXT_CHAR_STATE) continue;
				if (keyLength == -1) { // if parsing the key
					currentState = END_OF_PROP_STATE;
					continue;
				}
			}
			if (currentState == SKIP_NEXT_CHAR_STATE || currentState == RC_STATE) currentState = INIT_STATE;
		}
		firstChar = false;
		if (currentState == END_OF_PROP_STATE) {
			keyLength = offset;
			currentState = INIT_STATE;
		}
		buf[offset++] = (char)c;
	}
	if (keyLength >= 0) {
		String temp = new String(buf, 0, offset);
		table.put(temp.substring(0,keyLength), temp.substring(keyLength));
	}
	return table;
}
/**
 * Answers whether the character is a Java whitespace character.
 */
static boolean isWhitespace(char c) {
	// Optimized case for ASCII
	if ((c >= 0x1c && c <= 0x20) || (c >= 0x9 && c <= 0xd)) return true;
	return false;
}
}
