/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.io;

import java.io.IOException;
import java.io.InputStream;


/**
 * Simplified version of a BufferedInputStream
 * allowing to rewind read bytes by providing them!
 */
public class SimpleBufferedInputStream extends InputStream {
	InputStream in;
	byte[] buf;
	int bytesAvailable;
	int pos;
	static final int DefaultBufferSize = 2048;
/**
 * Constructor for SimpleBufferedInputStream.
 */
public SimpleBufferedInputStream(InputStream in) throws IOException {
	super();
	this.in = in;
	this.pos = 0;
	this.buf = new byte[DefaultBufferSize];
	fillByteBuffer();
}
/**
 * Reads a single byte from this InputStream and returns the result as
 * an int.  The low-order byte is returned or -1 of the end of stream was
 * encountered.  If the underlying buffer does not contain any available bytes
 * then it is filled and the first byte is returned.
 *
 * @return 		the byte read or -1 if end of stream.
 *
 * @exception 	java.io.IOException If the stream is already closed or another IOException occurs.
 */
public int read() throws IOException {
	if (pos < bytesAvailable) return buf[pos++] & 0xFF;
	if (fillByteBuffer() != -1) return buf[pos++] & 0xFF;
	return -1;
}
/**
 * Reads at most <code>length</code> bytes from this InputStream and stores them in byte
 * array <code>buffer</code> starting at offset <code>offset</code>. Answer the number of bytes
 * actually read or -1 if no bytes were read and end of stream was encountered.  If all the
 * buffered bytes have been used, a mark has not been set, and the requested number
 * of bytes is larger than the receiver's buffer size, this implementation bypasses
 * the buffer and simply places the results directly into <code>buffer</code>.
 *
 * @param		buffer	the byte array in which to store the read bytes.
 * @param		offset	the offset in <code>buffer</code> to store the read bytes.
 * @param		length	the maximum number of bytes to store in <code>buffer</code>.
 * @return 		the number of bytes actually read or -1 if end of stream.
 *
 * @exception 	java.io.IOException If the stream is already closed or another IOException occurs.
 */
public int read(byte[] buffer, int offset, int length) throws IOException {
	// avoid int overflow
	if (0 <= offset && offset <= buffer.length && 0 <= length && length <= buffer.length - offset) {
		if (length == 0) return 0;

		int required;
		if (pos < bytesAvailable) {
			/* There are bytes available in the buffer. */
			int copylength = bytesAvailable - pos >= length ? length : bytesAvailable - pos;
			System.arraycopy(buf, pos, buffer, offset, copylength);
			pos += copylength;
			if (copylength == length || in.available() == 0)
				return copylength;
			offset += copylength;
			required = length - copylength;
		} else required = length;

		while (true) {
			int read;
			/* If we're not marked and the required size is greater than
			 * the buffer, simply read the bytes directly bypassing the
			 * buffer. */
			if (required >= buf.length) {
				read = in.read(buffer, offset, required);
				if (read == -1)
					return required == length ? -1 : length - required;
			} else {
				if (fillByteBuffer() == -1)
					return required == length ? -1 : length - required;
				read = bytesAvailable - pos >= required ? required : bytesAvailable - pos;
				System.arraycopy(buf, pos, buffer, offset, read);
				pos += read;
			}
			required -= read;
			if (required == 0) return length;
			if (in.available() == 0) return length - required;
			offset += read;
		}
	} else {
		throw new ArrayIndexOutOfBoundsException();
	}
}
/**
 * Restore a set of bytes in the buffer
 */
public void unread(byte[] bytes, int length) {
	if (pos >= length) {
		pos -= length;
	} else {
		int bytesToRestore = length - pos;
		byte[] newBuffer = new byte[buf.length + bytesToRestore];
		System.arraycopy(bytes, 0, newBuffer, 0, bytesToRestore);
		System.arraycopy(buf, 0, newBuffer, bytesToRestore, buf.length);
		this.buf = newBuffer;
	}
}
int fillByteBuffer() throws IOException {	
	int result = in.read(buf, 0, buf.length);
	this.bytesAvailable = result == -1 ? 0:result;
	this.pos = 0;
	return result;
}

/**
 * Close the InputStream.  Concrete implementations of this class should
 * free any resources during close.  This implementation does nothing.
 *
 * @version		initial
 *
 * @exception 	java.io.IOException	If an error occurs attempting to close this InputStream.
 */
public void close() throws IOException {
	if (in != null) in.close();
	this.buf = null;
}
/**
 * Answers a int representing then number of bytes that are available
 * before this InputStream will block.  This method always returns 0.
 * Subclasses should override and indicate the correct number of bytes
 * available.
 *
 * @version		initial
 *
 * @return 		the number of bytes available before blocking.
 *
 * @exception 	java.io.IOException	If an error occurs in this InputStream.
 */
public int available() throws IOException {
	return bytesAvailable - pos +  in.available();
}
/**
 * Skips <code>amount</code> number of bytes in this BufferedInputStream.  Subsequent
 * <code>read()</code>'s will not return these bytes unless <code>reset()</code>
 * is used.
 *
 * @version		initial
 *
 * @param 		amount		the number of bytes to skip.
 * @return		the number of bytes actually skipped.
 *
 * @exception 	java.io.IOException If the stream is already closed or another IOException occurs.
 */
public synchronized long skip(long amount) throws IOException {
	if (amount < 1) return 0;

	if (bytesAvailable - pos >= amount) {
		pos += amount;
		return amount;
	}
	long read = bytesAvailable - pos;
	pos = bytesAvailable;
	return read + in.skip(amount - read);
}
}
