/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.io;

import java.io.IOException;

import org.eclipse.ercp.xml.parser.EXmlMsg;


/**
 * StreamDecoder for the UTF8 encoding
 */
public class StreamDecoder_UTF8 extends StreamDecoder {
	byte lastByte2, lastByte3;
	boolean readOneMoreChar;

/**
 * Constructor for StreamDecoder_UTF8.
 * @param stream
 */
public StreamDecoder_UTF8(SimpleBufferedInputStream stream) {
	super(stream, UTF8_Encoding);
	this.readOneMoreChar = false;
}
/**
 * @see org.eclipse.ercp.xml.io.StreamDecoder#read()
 */
public int read() throws IOException {
	if (readOneMoreChar) {
		int temp = (((lastByte2 & 0xF) << 6) + (lastByte3 & 0x3F) + 0xDC00);
		this.readOneMoreChar = false;
		return temp;
	}
	byte b0 = (byte)(stream.read());
	if (b0 == -1)  return -1;
	if (b0 >= 0) return (char)b0;
	byte b1 = (byte)(stream.read());
	if (b1 == -1) throw new IOException(EXmlMsg.getDefault().getString(EXmlMsg.MALFORMED_UTF8_1, Integer.toHexString(b0)));			 //$NON-NLS-1$
	if ((b0 & 0xE0) == 0xC0) {
		return /*(char)*/((b0 & 0x1F) << 6) + (b1 & 0x3F);
	} else {
		byte b2 = (byte)(stream.read());
		if (b2 == -1) throw new IOException(
									EXmlMsg.getDefault().getString(
															EXmlMsg.MALFORMED_UTF8_2, 
															new String[] {
																	Integer.toHexString(b0),
																	Integer.toHexString(b1)
															}));			 //$NON-NLS-1$
		if ((b0 & 0xF0) == 0xE0) {
			return /*(char)*/((b0 & 0xF) << 12) + ((b1 & 0x3F) << 6) + (b2 & 0x3F);
		} else {
			byte b3 = (byte)(stream.read());
			if (b2 == -1) throw new IOException(EXmlMsg.getDefault().getString(EXmlMsg.MALFORMED_UTF8_3,			 //$NON-NLS-1$
																					new String[] {
																						Integer.toHexString(b0),
																						Integer.toHexString(b1),
																						Integer.toHexString(b2)
																					}));
			this.lastByte2 = b2;
			this.lastByte3 = b3;
			this.readOneMoreChar = true;
			return /*(char)*/((((b0 & 7) << 2) + ((b1 >> 4) & 3) - 1) << 6) + ((b1 & 0xF) << 2) + ((b2 >> 4) & 3) + 0xD800;
			
		}
	}
}
/**
 * @see org.eclipse.ercp.xml.io.StreamDecoder#read(char[], int, int)
 */
public int read(char[] buf, int offset, int count) throws IOException {
	if (count <= 0) return count;
	int end = offset+count;
	int charOffset = offset;
	if (readOneMoreChar) {
		buf[charOffset++] = (char)(((lastByte2 & 0xF) << 6) + (lastByte3 & 0x3F) + 0xDC00);
		this.readOneMoreChar = false;
	}
	byte[] buffer = new byte[end-charOffset];
	if (-1 == stream.read(buffer)) return ((charOffset - offset) == 0)?-1:charOffset - offset;
	int index = 0;
	int missingBytes = 0;
	byte b0, b1, b2, b3;
	while (charOffset < end) {
		b0 = buffer[index++];
		if (b0 >= 0) {// >= 0x80 is negative
			buf[charOffset++] = (char)b0;
		} else {
			if (index == buffer.length) {
				// ==> missingBytes > 0
				buffer = new byte[missingBytes];
				index = 0;
				missingBytes = 0;
				if (-1 == stream.read(buffer)) {
					throw new IOException(EXmlMsg.getDefault().getString(
																	EXmlMsg.MALFORMED_UTF8_1,
																	Integer.toHexString(b0)));
				} 
			}
			b1 = buffer[index++];
			missingBytes++;
			if ((b0 & 0xE0) == 0xC0) {
				buf[charOffset++] = (char)(((b0 & 0x1F) << 6) + (b1 & 0x3F));
			} else {
				if (index == buffer.length) {
					// ==> missingBytes > 0
					buffer = new byte[missingBytes];
					index = 0;
					missingBytes = 0;
					if (-1 == stream.read(buffer)) {
						throw new IOException(EXmlMsg.getDefault().getString(
																		EXmlMsg.MALFORMED_UTF8_2, 
																		new String[] {
																			Integer.toHexString(b0),
																			Integer.toHexString(b1)
																		}));
					}
				}
				b2 = buffer[index++];
				missingBytes++;
				if ((b0 & 0xF0) == 0xE0) {
					buf[charOffset++] = (char)(((b0 & 0xF) << 12) + ((b1 & 0x3F) << 6) + (b2 & 0x3F));
				} else {
					if (index == buffer.length) {
						// ==> missingBytes > 0
						buffer = new byte[missingBytes];
						index = 0;
						missingBytes = 0;
						if (-1 == stream.read(buffer)) { 
							throw new IOException(EXmlMsg.getDefault().getString(
																			EXmlMsg.MALFORMED_UTF8_3,
																			new String[] {
																				Integer.toHexString(b0),
																				Integer.toHexString(b1),
																				Integer.toHexString(b2)
																			}));
						}
					}
					b3 = buffer[index++];
					missingBytes++;
					buf[charOffset++] = (char)(((((b0 & 7) << 2) + ((b1 >> 4) & 3) - 1) << 6) + ((b1 & 0xF) << 2) + ((b2 >> 4) & 3) + 0xD800);
					if (charOffset < end) {
						buf[charOffset++] = (char)(((b2 & 0xF) << 6) + (b3 & 0x3F) + 0xDC00);
					} else {
						this.lastByte2 = b2;
						this.lastByte3 = b3;
						this.readOneMoreChar = true;
					}
				}
			}
		}
	}
	return count;
}
/**
 * If the decoder read too much bytes during the last read
 * (cf. UTF8) restore them!
 */
public void restore() {
	if (readOneMoreChar) {
		stream.unread(new byte[] {lastByte2, lastByte3}, 2);
		this.readOneMoreChar = false;
	}
}
}
