/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.io;

import java.io.*;



/**
 * Implement a dedicated InputStreamReader for an XML file
 * which can change dynamically for an other encoding
 * Used when reading an XML file...
 */
public class XmlInputStreamReader extends Reader {
	SimpleBufferedInputStream in;
	Reader decoder;
	

/**
 * Constructs a new XmlInputStreamReader on the InputStream <code>in</code>.  The default
 * buffer size (2K) is allocated and all reads can now be filtered through this stream.
 *
 * @param		in		the InputStream to buffer reads on.
 *
 */
public XmlInputStreamReader(InputStream in) throws IOException {
	this(in, null);
}

/**
 * Constructs a new XmlInputStreamReader on the InputStream <code>in</code>.  The
 * buffer size is specified by the parameter <code>size</code> and all reads can
 * now be filtered through this XmlInputStreamReader.
 *
 * @param		in		the InputStream to buffer reads on.
 * @param		enc		the java encoding name.
 */
public XmlInputStreamReader(InputStream in, String ianaEncoding) throws IOException {
	super();
	this.in = new SimpleBufferedInputStream(in);
	this.decoder = createFirstDecoder(ianaEncoding);
}
Reader createFirstDecoder(String ianaEncoding) throws UnsupportedEncodingException {
	// read the first four bytes to guess the encoding
    int bytesToUnread = 3;
	if (ianaEncoding == null) {
	    byte[] b = new byte[bytesToUnread];
    	int available = -1;
    	try {
    		available = in.read(b, 0, bytesToUnread);
    	} catch (IOException e) {
    		// NOP
    	}
	    if (available == -1) {
	    	ianaEncoding = StreamDecoder.UTF8_Encoding;
	    } else
	    if (available < bytesToUnread) {
    		in.unread(b, available);
	    } else {
	        int b0 = b[0] & 0xFF;
	        int b1 = b[1] & 0xFF;
	        int b2 = b[2] & 0xFF;
//	        int b3 = b[3] & 0xFF;
	
	        // UTF-16, with BOM
	        if (b0 == 0xFE && b1 == 0xFF) {
	            // UTF-16, big-endian
	            ianaEncoding = StreamDecoder.UTF16BE_Encoding;
				// ignore first two bytes...
	            bytesToUnread -= 2;
	        } else
	        if (b0 == 0xFF && b1 == 0xFE) {
	            // UTF-16, little-endian
	            ianaEncoding = StreamDecoder.UTF16LE_Encoding;
				// ignore first two bytes...
	            bytesToUnread -= 2;
	        } else 
	        // UTF-8 with a BOM
	        if (b0 == 0xEF && b1 == 0xBB && b2 == 0xBF) {
	            ianaEncoding = StreamDecoder.UTF8_Encoding;
				// ignore first three bytes...
	            bytesToUnread -= 3;
//	        } else
//	        // other encodings
//	        if (b0 == 0x00 && b1 == 0x3C && b2 == 0x00 && b3 == 0x3F) {
//	            // UTF-16, big-endian, no BOM
//	            // (or could turn out to be UCS-2...
//	            // REVISIT: What should this be?
//	            ianaEncoding = StreamDecoder.UTF16BE_Encoding;
//	        } else
//	        if (b0 == 0x3C && b1 == 0x00 && b2 == 0x3F && b3 == 0x00) {
//	            // UTF-16, little-endian, no BOM
//	            // (or could turn out to be UCS-2...
//	            ianaEncoding = StreamDecoder.UTF16LE_Encoding;
	        } else {
		        // default encoding
		        ianaEncoding = StreamDecoder.UTF8_Encoding;
	        }
		    in.unread(b, bytesToUnread);
	    }
	}
	
	return createDecoder(ianaEncoding);
}
Reader createDecoder(String ianaEncoding) throws UnsupportedEncodingException {
	//if (decoder != null) decoder.restore();
	if (ianaEncoding == null) {
		ianaEncoding = StreamDecoder.UTF8_Encoding;
	} else {
		ianaEncoding = ianaEncoding.toUpperCase();
	}
	// Try to use an optimized decoder
	if (ianaEncoding.equals(StreamDecoder.UTF8_Encoding)) {
		return new StreamDecoder_UTF8(in);
	}
	if (ianaEncoding.equals(StreamDecoder.ASCII_Encoding)) {
		return new StreamDecoder_ASCII(in);
	}
	if (ianaEncoding.equals(StreamDecoder.ISO8859_1_Encoding)) {
		return new StreamDecoder_ISO8859_1(in);
	}
	if (ianaEncoding.equals(StreamDecoder.UTF16BE_Encoding)) {
		return new StreamDecoder_UTF16BE(in);
	}
	if (ianaEncoding.equals(StreamDecoder.UTF16LE_Encoding)) {
		return new StreamDecoder_UTF16LE(in);
	}
	if (ianaEncoding == null) return new InputStreamReader(in);
	
	String javaEncoding = IanaJavaMap.getIana2Java(ianaEncoding);
	if (javaEncoding == null) throw new UnsupportedEncodingException(ianaEncoding);
	return new InputStreamReader(in, javaEncoding);
}
public void setEncoding(String ianaEncoding) throws UnsupportedEncodingException {
	if ((decoder != null) && (decoder instanceof StreamDecoder)) {
		StreamDecoder sDecoder = (StreamDecoder)decoder;
		if ((!sDecoder.getEncoding().equals(ianaEncoding)) &&
			(sDecoder.supportNewEncoding())) {
			sDecoder.restore();
			this.decoder = createDecoder(ianaEncoding);
		}
	}			
}

/**
 * Reads a single character from this reader and returns the result as
 * an int.  The 2 higher-order characters are set to 0. If the end of reader was encountered
 * then return -1.
 *
 * @return 		the character read or -1 if end of reader.
 *
 * @exception 	java.io.IOException If the Reader is already closed or some other IO error occurs.
 */
public final int read() throws IOException {
	return decoder.read();
}
/**
 * Reads at most <code>count</code> characters from this Reader and stores them
 * at <code>offset</code> in the character array <code>buf</code>.  Returns the
 * number of characters actually read or -1 if the end of reader was encountered.
 *
 * @version		initial
 *
 * @param 		buf		character array to store the read characters
 * @param 		offset 	offset in buf to store the read characters
 * @param 		count 	maximum number of characters to read
 * @return		the number of characters read or -1 if end of reader.
 *
 * @exception 	java.io.IOException If the Reader is already closed or some other IO error occurs.
 */
public final int read(char buf[], int offset, int count) throws IOException {
	return decoder.read(buf, offset, count);
}
/**
 * Close this Reader.  This must be implemented by any concrete subclasses.  The
 * implementation should free any resources associated with the Reader.
 *
 * @exception 	java.io.IOException	If an error occurs attempting to close this Reader.
 */
public void close() throws IOException {
	if (decoder != null) {
		decoder.close();
	} else {
		in.close();
	}
}
InputStream getInputStream() {
	return in;
}
}
