/*******************************************************************************
 * Copyright (c) 2005, 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.parser;

import java.io.IOException;
import java.io.Reader;
import java.io.UnsupportedEncodingException;

import javax.xml.parsers.SAXParser;

import org.eclipse.ercp.xml.io.IOSpecificCall;
import org.eclipse.ercp.xml.io.XmlInputStreamReader;
import org.xml.sax.*;



/**
 * This class implements the default behavior for
 * a non-validating XML parser using SAX 2.0 APIs
 * Subset for JSR 172
 * 
 * This class is abstract to supports several concrete subclesses:
 * 	- SAXParserImpl for CLDC (JSR 172)
 * 	- SAXParserImpl for Core and bigger
 * 	- XMLStateMachineParser
 * 
 * Grammar:
 * 	- <http://www.w3.org/TR/REC-xml>
 *  - <http://www.w3.org/TR/REC-xml-names/>
 * 
 * The assumptions of this parser is:
 *		- The XML file is lexicaly and syntaxy correct
 * 		- There is not twice the same Namespace prefix declared
 *		- The prefix 'xml' is never used to declare a namespace
 * 			(this namespace is always associated to <http://www.w3.org/XML/1998/namespace>
 * 		- The XML file prolog doesn't contain any:
 * 			- <!DOCTYPE ...> declaration
 * 		- The characters used as element, attribute or CData respect the XML specs
 * 			defined in <http://www.w3.org/TR/REC-xml> 
 */
public abstract class AbstractSAXParserImpl extends SAXParser implements Locator {
	// IO variables
	Reader reader;
	InputSource source;
	CharDecoder charDecoder;
	int lineNumber;
	int columnNumber;
	
	// Intermediate variables: while tokenizing
	int currentState;
	char[] writeBuffer;
	int writeBufferIndex, columnIndex;
	char[] refBuffer;
	int refBufferIndex;
	int cdataHeaderIndex;
	final static String CDATA_HEADER = "[CDATA[";
	int doctypeHeaderIndex;
	final static String DOCTYPE_HEADER = "DOCTYPE";
	int openSquareBrackets;
	int openAngleBrackets;
	
	// Token variables
	String elementQName, elementNamespace, elementLocalName;
	String attributeQName, attributeNamespace, attributeLocalName;
	String attributeValue;
	AttributesImpl attributes;
	protected NSDeclaration currentNSDecl;
	int elementsDepth;
	boolean parsingYet;
	int closingValue;
	boolean checkXmlDeclaration;
	
	// Features
	/**
	 * If true ==> the Parser will identify the namespace ids!
	 */
	boolean fNamespaces;
	/**
	 * If true ==> the attributes will also contain the namespace declaration xmlns:...
	 */
	boolean fNamespacesPrefixes;
	protected boolean fStringInterning, fXmlNSUris;

	public static int WriteBufferSize = 5 * 1024; //5Kb
	
	// Features (See <http://www.saxproject.org/apidoc/org/xml/sax/package-summary.html>)
	public static final String XmlOrgFeaturesPrefix = "http://xml.org/sax/features/";	 //$NON-NLS-1$
	public static final String NamespacesFeature = "namespaces";	 //$NON-NLS-1$
	public static final String NamespacePrefixesFeature = "namespace-prefixes";	 //$NON-NLS-1$
	public static final String StringInterningFeature = "string-interning";	 //$NON-NLS-1$
	public static final String XmlNSUris = "xmlns-uris";	 //$NON-NLS-1$
	static final String[] XmlOrgFeatureIDs = {
		NamespacesFeature,
		NamespacePrefixesFeature,
		"validation",	 //$NON-NLS-1$
		"external-general-entities",	 //$NON-NLS-1$
		"external-parameter-entities",	 //$NON-NLS-1$
		"is-standalone",	 //$NON-NLS-1$
		"lexical-handler/parameter-entities",	 //$NON-NLS-1$
		"resolve-dtd-uris",	 //$NON-NLS-1$
		StringInterningFeature,
		"use-attributes2",	 //$NON-NLS-1$
		"use-locator2",	 //$NON-NLS-1$
		"use-entity-resolver2", //$NON-NLS-1$
		XmlNSUris	 //$NON-NLS-1$
	};
	
	// Properties
	// See <http://www.saxproject.org/apidoc/org/xml/sax/package-summary.html>
	public static final String XmlOrgPropertiesPrefix = "http://xml.org/sax/properties/"; //$NON-NLS-1$
	public static final String LexicalHandlerID = "lexical-handler";
	static final String[] XmlOrgPropertyIDs = {
							"declaration-handler",
							"dom-node",
							LexicalHandlerID,
							"xml-string",
						};

public AbstractSAXParserImpl() {
	super();
	// Set a Default CharDecoder to avoid
	// any kind of test like:
	//		if (charDecoder != null) ...
	this.charDecoder = CharDecoder.forXml();
	// Default features!
	this.fNamespaces 			= true;						// default
	this.fNamespacesPrefixes	= false;					// default
	this.fStringInterning		= canDoStringInterning();	// default
	this.fXmlNSUris				= false;					// default
	resetParser();
}
public void resetParser() {
	this.currentState = DOC_STATE;
	
	this.writeBuffer = new char[WriteBufferSize];
	resetWriteBuffer();

	this.refBuffer = new char[WriteBufferSize];
	this.refBufferIndex = 0;
	// Install the default name spaces
	this.currentNSDecl = new NSDeclaration();
	
	this.elementsDepth = 0;
	this.parsingYet = false;
	
	this.lineNumber = 0;
	this.columnNumber = 0;
	this.attributes = null;
}
///////////////////////////////////////////
final void resetWriteBufferWith(int c) {
	resetWriteBuffer();
	writeBuffer[writeBufferIndex++] = (char)c;	
}
final void resetWriteBuffer() {
	this.writeBufferIndex = 0;
	this.columnIndex = -1;
}
final void appendToWriteBuffer(int c) {
	writeBuffer[writeBufferIndex++] = (char)c;	
}
final void appendToWriteBuffer(char[] text, int start, int length) {
	while ((writeBufferIndex + length) > writeBuffer.length) {
		// Double the size of the write buffer
		int max = writeBuffer.length + WriteBufferSize;
		// Create a new write buffer
		char[] newWBuffer = new char[max];
		// Move the write buffer to the new one
		System.arraycopy(writeBuffer, 0, newWBuffer, 0, writeBuffer.length);
		this.writeBuffer = newWBuffer;
	}
	System.arraycopy(text, start, writeBuffer, writeBufferIndex-1, length);
	this.writeBufferIndex += length;
}
final void appendAndGrowToWriteBuffer(int c) {
	try {
		writeBuffer[writeBufferIndex++] = (char)c;
	} catch (ArrayIndexOutOfBoundsException e) {
		// Double the size of the write buffer
		int max = writeBuffer.length + WriteBufferSize;
		// Create a new write buffer
		char[] newWBuffer = new char[max];
		// Move the write buffer to the new one
		System.arraycopy(writeBuffer, 0, newWBuffer, 0, writeBuffer.length);
		writeBuffer = newWBuffer;
		// Set the write buffer with the char
		writeBuffer[writeBufferIndex-1] = (char)c;
	}
}
///////////////////////////////////////////
final void resetRefBufferWith(int c) {
	this.refBufferIndex = 0;
	refBuffer[refBufferIndex++] = (char)c;	
}
final void appendToRefBuffer(int c) {
	refBuffer[refBufferIndex++] = (char)c;	
}
final void appendRefBufferToWriteBuffer() {
	appendToWriteBuffer('&');
	System.arraycopy(refBuffer, 0, writeBuffer, writeBufferIndex, refBufferIndex);
	this.writeBufferIndex += refBufferIndex;
}
final void appendRefValueToWriteBuffer() throws SAXException {
	String charName = new String(refBuffer, 0, refBufferIndex);
	String charValue = charDecoder.getCharValue(charName);
	if (charValue == null) {
		warning(EXmlMsg.getDefault().getString(EXmlMsg.WARNING_UNKNOWN_CHARACTER_DEF), "&" + charName + ";");
		appendToWriteBuffer ('&');
		appendToWriteBuffer(refBuffer, 0, refBufferIndex);
		appendToWriteBuffer(';');
	} else {
		appendToWriteBuffer(charValue.charAt(0));
	}
}
///////////////////////////////////////////
final void resetElementQName() {
	this.elementQName = null;
	this.elementNamespace = null;
	this.elementLocalName = null;
}
final void endStartElementName() {
	endElementLocalName();
	if (fNamespaces) pushNamespaceDeclaration();
}
final void endElementLocalName() {
	this.elementQName = new String(writeBuffer, 0, writeBufferIndex);
	if (columnIndex != -1) {
		this.elementNamespace = new String(writeBuffer, 0, columnIndex-1);
		this.elementLocalName = !fNamespaces?"":new String(writeBuffer, columnIndex, writeBufferIndex - columnIndex);
	} else {
		this.elementNamespace = null;
		this.elementLocalName = !fNamespaces?"":this.elementQName;
	}
	resetWriteBuffer();
	// Reset the list of attributes...
	this.attributes = null;
}

final void endElementNamespace() {
	this.columnIndex = writeBufferIndex;
}
///////////////////////////////////////////
final void resetAttributeQName() {
	this.attributeQName = null;
	this.attributeNamespace = null;
	this.attributeLocalName = null;
}
final void endAttributeLocalName() {
	this.attributeQName = new String(writeBuffer, 0, writeBufferIndex);
	if (columnIndex != -1) {
		this.attributeNamespace = new String(writeBuffer, 0, columnIndex-1);
		this.attributeLocalName = new String(writeBuffer, columnIndex, writeBufferIndex - columnIndex);
	} else {
		this.attributeNamespace = null;
		this.attributeLocalName = this.attributeQName;
	}
	// reset the write buffer
	resetWriteBuffer();
}
final void endAttributeNamespace() {
	this.columnIndex = writeBufferIndex;
}
///////////////////////////////////////////
final void resetAttributeValue() {
	this.attributeValue = null;
}
final void endAttValue() throws SAXException {
	this.attributeValue = new String(writeBuffer, 0, writeBufferIndex);
	resetWriteBuffer();
	addAttribute(attributeNamespace, attributeQName, attributeLocalName, attributeValue);
}
protected void addAttribute(String namespace, String qName, String localName, String attvalue) throws SAXException {
	// It should be never null
	//if (attributes == null) this.attributes = new AttributesImpl();
	if (!fNamespaces) {
		addAttribute(null, namespace, qName, "", AttributesImpl.CDATA, attvalue);
	} else {
		// ==> fNamespaces == true!
		
		/////////////////////////////////////////////////////////////////////////////
		// modified by nzhang@CRL at 2006/01/19
		// extend MicroXML to meet case xmlns="http://..."
		if ("xmlns".equals(localName)) {
			namespace = localName;
			localName = "";
		}
		/////////////////////////////////////////////////////////////////////////////
		
		if ("xmlns".equals(namespace)) { //$NON-NLS-1$
			startPrefixMapping(localName, attvalue);
			if (fNamespacesPrefixes) {
				addAttribute((fXmlNSUris?currentNSDecl:null), namespace, qName, localName, AttributesImpl.CDATA, attvalue);
			}
		} else {
			addAttribute((fXmlNSUris?currentNSDecl:null), namespace, qName, localName, AttributesImpl.CDATA, attributeValue);
		}
	}
}
void addAttribute(NSDeclaration nsDeclaration, String prefix, String qName, String localName, String type, String value) {
	if (attributes == null) this.attributes = new AttributesImpl();
	attributes.addAttribute(nsDeclaration, prefix, qName, localName, type, value);
}
///////////////////////////////////////////
final int endPITargetName() {
	this.elementQName = new String(writeBuffer, 0, writeBufferIndex);
	boolean wasXMLTarget = isXMLTargetNameWasFound();
	// reset the write buffer
	resetWriteBuffer();
	// Reset the list of attributes...
	this.attributes = null;
	return (wasXMLTarget)?XPI_XML_TARGET_STATE:XPI_DATA_STATE;
}
final boolean isXMLTargetNameWasFound() {
	return (writeBufferIndex == 3) &&
			((writeBuffer[0]=='x') || (writeBuffer[0]=='X')) &&
			((writeBuffer[1]=='m') || (writeBuffer[1]=='M')) &&
			((writeBuffer[2]=='l') || (writeBuffer[2]=='L'));
}
final void endPIAttributeName() {
	this.attributeQName = new String(writeBuffer, 0, writeBufferIndex);
}
final void endPIAttValue() {
	this.attributeValue = new String(writeBuffer, 0, writeBufferIndex);
	addAttribute(null, "", attributeQName, attributeQName, AttributesImpl.CDATA, attributeValue);
	// reset the write buffer
	resetWriteBuffer();
}
final void checkXmlDeclaration() throws SAXException {
	if ((elementQName != null) && (attributes != null)) {
		String ianaEncoding = attributes.getValue("", "encoding");
		if (ianaEncoding != null) {
			if (source.getByteStream() != null) {
				try {
					((XmlInputStreamReader)reader).setEncoding(ianaEncoding);
				} catch (UnsupportedEncodingException e) {
					warning(EXmlMsg.getDefault().getString(EXmlMsg.WARNING_UNSUPPORTED_ENCODING), ianaEncoding); //$NON-NLS-1$
				}
				
			}
		}
	}
}

/**
 * @see XMLReader#parse(String)
 */
public void parse(String systemId) throws IOException, SAXException {
	parse(new InputSource(systemId));
}
/**
 *Internal call
 */
public void parse(InputSource inputSource) throws IOException, SAXException {
	Reader openedReader = null;
	try {
		if (parsingYet) throw new NullPointerException(EXmlMsg.getDefault().getString(EXmlMsg.Parsing_yet)); //$NON-NLS-1$
		parsingYet = true;
		resetParser();
		this.reader = null;
		this.source = inputSource;
		// Try to access to the character stream (encoding included)
		if (inputSource.getCharacterStream() != null) {
			this.reader = inputSource.getCharacterStream();
		} else
		if (inputSource.getByteStream() != null) {
			this.reader = new XmlInputStreamReader(inputSource.getByteStream(), inputSource.getEncoding());
		} else // Fixing PR#171546
		if (inputSource.getSystemId() != null) {
			this.reader = openedReader = IOSpecificCall.getDefault().getReaderFromSystemID(inputSource.getSystemId(), inputSource.getEncoding());
		}
		if (reader == null) 
			throw new IllegalArgumentException(EXmlMsg.getDefault().getString(EXmlMsg.CS_OR_BS_SET_ERROR));

		doParse();
	} finally {
		parsingYet = false;
		if (openedReader != null) {
			// this Reader has been opened locally and must be closed locally too!
			try {
				openedReader.close();
			} catch (IOException e) {
				// NOP
			}
		}
	}
}
/**
 * Effective parsing...
 * 
 * @throws IOException
 * @throws SAXException
 */
void doParse() throws IOException, SAXException {
	this.closingValue = -1;
	this.checkXmlDeclaration = false;
	boolean proceed = true;
	startDocument();
	while (proceed) {
		proceed = parseNextCharacter(reader.read());
	}
}
final void newLine() {
	this.columnNumber = 0;
	this.lineNumber++;
}
public boolean parseNextCharacter(int c) throws SAXException {
	columnNumber++;
	switch (currentState) {
		case DOC_STATE:	// Beginning of the Document
			if ((c == ' ') || (c == '\t') || (c == '\n')) {
				// this.currentState = DOC_STATE;
				return true;
			} else
			if (c == '\r') {
				newLine();
				// this.currentState = DOC_STATE;
				return true;
			} else
			if (c == '<') {
				this.currentState = PROLOG_STATE;
				return true;
			} else
			if (c == -1) {
				return false;
			} else {
				fatalError(EXmlMsg.ERROR_LT_EXPECTED);
				return false;
			}
		case PROLOG_STATE: // "<" has been identified
			if (c == '?') {
				this.checkXmlDeclaration = true;
				this.currentState = XPI_STATE;
				return true;
			} else
			if (c == '!') {
				this.currentState = DT_DECL_STATE;
				return true;
			} else
			if (isNameStartChar(c)) {
				resetWriteBufferWith(c);
				this.currentState = START_ELEMENT_STATE;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_TAG_NAME_EXPECTED);
				return false;
			}
		case XPI_STATE: // "<?" has been identified
			if (isNameStartChar(c)) {
				resetWriteBufferWith(c);
				this.currentState = XPI_TARGET_NAME_STATE;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_WRONG_PI);
				return false;
			}
		case MISC_STATE:  // We are between 2 elements
			if ((c == ' ') || (c == '\t') || (c == '\n')) {
				//this.currentState = MISC_STATE;
				appendAndGrowToWriteBuffer(c);
				return true;
			} else
			if (c == '\r') {
				newLine();
				//this.currentState = MISC_STATE;
				appendAndGrowToWriteBuffer(c);
				return true;
			} else
			if (c == '<') {
				fireCharacters();
				resetWriteBuffer();
				this.currentState = MISC2_STATE;
				return true;
			} else
			if (c == '&') {
				this.currentState = REF2_STATE;
				return true;
			}
			if (c == '>') {
				fatalError(EXmlMsg.ERROR_LT_EXPECTED);
				return false;
			} else
			if (c == -1) {
				endDocument();
				return false;
			} else {
				//this.currentState = MISC_STATE;
				appendAndGrowToWriteBuffer(c);
				return true;
			}
		case MISC2_STATE: // "<" has been identified
			if (c == '?') {
				this.checkXmlDeclaration = false;
				this.currentState = XPI_STATE;
				return true;
			} else
			if (c == '!') {
				this.currentState = DT_DECL_STATE;
				return true;
			} else
			if (c == '/') {
				this.currentState = END_ELEMENT_STATE;
				return true;
			} else
			if (isNameStartChar(c)) {
				resetElementQName();
				resetWriteBufferWith(c);
				this.currentState = START_ELEMENT_STATE;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_TAG_NAME_EXPECTED);
				return false;						
			}
		case DT_DECL_STATE: // "<!" has been identified
			if (c == '-') {
				this.currentState = CMT2_STATE;
				return true;
			} else
			if (c == '[') {
				this.currentState = CDATA_STATE;
				this.cdataHeaderIndex = 1;
				return true;
			}
			if (c == 'D') {
				this.currentState = DOCTYPE_STATE;
				this.doctypeHeaderIndex = 1;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_TAG_COMMENT_EXPECTED);
				return false;
			}
		case CMT2_STATE: // "<!-" has been identified
			if (c == '-') {
				this.currentState = CMT3_STATE;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_TAG_COMMENT_EXPECTED);
				return false;
			}
		case CMT3_STATE: // "<!--xxxx" has been identified
			if (c == '-') {
				this.currentState = CMT4_STATE;
				return true;
			} else
			if (c == -1) {
				fatalError(EXmlMsg.ERROR_END_OF_TAG_EXPECTED);
			} else {
				if (c == '\r') newLine();
				//this.currentState = CMT3_STATE;
				return true;
			}
		case CMT4_STATE: // "<!--xxxx-" has been identified
			if (c == '-') {
				this.currentState = CMT5_STATE;
				return true;
			} else
			if (c == -1) {
				fatalError(EXmlMsg.ERROR_END_OF_TAG_EXPECTED);
				return false;
			} else {
				if (c == '\r') newLine();
				this.currentState = CMT3_STATE;
				return true;
			}
		case CMT5_STATE: // "<!--xxxx--" has been identified
			if (c == '>') {
				this.currentState = MISC_STATE;
				resetWriteBuffer();
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_GT_EXPECTED);
				return false;
			}
		case START_ELEMENT_STATE: // "<aaa" has been identified
			if ((c == ' ') || (c == '\t') || (c == '\n')) {
				endStartElementName();
				this.currentState = ATTLIST_STATE;
				return true;
			} else
			if (c == '\r') {
				newLine();	
				endStartElementName();
				this.currentState = ATTLIST_STATE;
				return true;
			} else
			if (c == '>') {
				endStartElementName();
				fireStartElement();
				this.currentState = MISC_STATE;
				resetWriteBuffer();
				return true;
			} else
			if (c == '/') {
				endStartElementName();
				this.currentState = ENDTAG_STATE;
				return true;
			} else
			if (c == ':') {
				appendToWriteBuffer(c); // We suppose that the tag name will not be longer than 5120 chars
				endElementNamespace();
				//this.currentState = TAGNAME_STATE;
				return true;
			} else
			if (isNameChar(c)) {
				appendToWriteBuffer(c); // We suppose that the tag name will not be longer than 5120 chars
				//this.currentState = TAGNAME_STATE;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_TAG_NAME_EXPECTED);
				return false;						
			}
		case END_ELEMENT_STATE: // "</" has been identified
			if (isNameStartChar(c)) {
				resetWriteBufferWith(c);
				this.currentState = END_ELEMENT_NAME_STATE;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_TAG_NAME_EXPECTED);
				return false;					
			}
		case END_ELEMENT_NAME_STATE: // "</aaa" has been identified
			if ((c == ' ') || (c == '\t') || (c == '\n')) {
				endElementLocalName();
				this.currentState = END_ELEMENT2_STATE;
				return true;
			} else
			if (c == '\r') {
				newLine();	
				endElementLocalName();
				this.currentState = END_ELEMENT2_STATE;
				return true;
			} else
			if (c == ':') {
				appendToWriteBuffer(c); // We suppose that the tag name will not be longer than 5120 chars
				endElementNamespace();
				//this.currentState = END_ELEMENT_NAME_STATE;
				return true;
			} else
			if (c == '>') {
				endElementLocalName();
				fireEndElement();
				this.currentState = MISC_STATE;
				resetWriteBuffer();
				return true;
			} else
			if (isNameChar(c)) {
				//this.currentState = END_ELEMENT_NAME_STATE;
				appendToWriteBuffer(c); // We suppose that the tag name will not be longer than 5120 chars
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_TAG_NAME_EXPECTED);
				return false;					
			}
		case END_ELEMENT2_STATE: // "</aaa " has been identified
			if ((c == ' ') || (c == '\t') || (c == '\n')) {
				//this.currentState = END_ELEMENT2_STATE;
				return true;
			} else
			if (c == '\r') {
				newLine();	
				//this.currentState = END_ELEMENT2_STATE;
				return true;
			} else
			if (c == '>') {
				fireEndElement();
				this.currentState = MISC_STATE;
				resetWriteBuffer();
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_TAG_NAME_EXPECTED);
				return false;
			}
		case ENDTAG_STATE: // "<aaa ... /" has been identified
			if (c == '>') {
				fireStartElement();
				fireEndElement();	
				this.currentState = MISC_STATE;
				resetWriteBuffer();
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_GT_EXPECTED);
				return false;
			}
		case ATTLIST_STATE: // "<aaa ... " has been identified
			if ((c == ' ') || (c == '\t') || (c == '\n')) {
				//this.currentState = ATTLIST_STATE;
				return true;
			} else
			if (c == '\r') {
				newLine();	
				//this.currentState = ATTLIST_STATE;
				return true;
			} else
			if (c == '>') {
				fireStartElement();
				this.currentState = MISC_STATE;
				resetWriteBuffer();
				return true;
			} else
			if (c == '/') {
				this.currentState = ENDTAG_STATE;
				return true;
			} else
			if (isNameStartChar(c)) {
				resetAttributeQName();
				resetWriteBufferWith(c);
				this.currentState = ATTNAME_STATE;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_GT_EXPECTED);
				return false;						
			}
		case ATTNAME_STATE: // "<aaa ... bbb" has been identified
			if ((c == ' ') || (c == '\t') || (c == '\n')) {
				endAttributeLocalName();
				this.currentState = ATTEQUAL_STATE;
				return true;
			} else
			if (c == '\r') {
				newLine();	
				endAttributeLocalName();
				this.currentState = ATTEQUAL_STATE;
				return true;
			} else
			if (c == ':') {
				appendToWriteBuffer(c); // We suppose that the attribute name will not be longer than 5120 chars
				endAttributeNamespace();
				//this.currentState = ATTNAME_STATE;
				return true;
			} else
			if (c == '=') {
				endAttributeLocalName();
				this.currentState = ATTVALUE_STATE;
				return true;
			} else
			if (isNameChar(c)) {
				appendToWriteBuffer(c); // We suppose that the attribute name will not be longer than 5120 chars
				//this.currentState = ATTNAME_STATE;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_GT_EXPECTED);
				return false;						
			}
		case ATTEQUAL_STATE: // "<aaa ... bbb " has been identified
			if ((c == ' ') || (c == '\t') || (c == '\n')) {
				//this.currentState = ATTEQUAL_STATE;
				return true;
			} else
			if (c == '\r') {
				newLine();	
				//this.currentState = ATTEQUAL_STATE;
				return true;
			} else
			if (c == '=') {
				this.currentState = ATTVALUE_STATE;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_EQUAL_EXPECTED);
				return false;
			}
		case ATTVALUE_STATE: // "<aaa ... bbb = " has been identified
			if ((c == ' ') || (c == '\t') || (c == '\n')) {
				//this.currentState = ATTVALUE_STATE;
				return true;
			} else
			if (c == '\r') {
				newLine();	
				//this.currentState = ATTVALUE_STATE;
				return true;
			} else
			if ((c == '"') || (c == '\'')) {
				resetAttributeValue();
				resetWriteBuffer();
				this.currentState = VALUE_STATE;
				closingValue = c;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_EQUAL_EXPECTED);
				return false;
			}
		case VALUE_STATE: // "<aaa ... bbb = '" has been identified
			if (c == closingValue) {
				endAttValue();
				this.currentState = ATTLIST_STATE;
				return true;
			} else
			if (c == '&') {
				this.currentState = REF_STATE;
				return true;
			} else
			if ((c == '>') || (c == '<')) {
				fatalError(EXmlMsg.ERROR_END_QUOTE_EXPECTED);
				return false;
			} else
			if (c == -1) {
				fatalError(EXmlMsg.ERROR_END_QUOTE_EXPECTED);
				return false;
			} else {
				if (c == '\r') newLine();
				appendAndGrowToWriteBuffer(c); // The value could be bigger than 5120 chars
				//this.currentState = VALUE_STATE;
				return true;
			}
		case REF_STATE: // "<aaa ... bbb = '...&" has been identified
			if (c == closingValue) {
				appendAndGrowToWriteBuffer('&'); // The value could be bigger than 5120 chars
				endAttValue();
				this.currentState = ATTLIST_STATE;
				return true;
			} else
			if (c == '&') {
				appendAndGrowToWriteBuffer('&'); // The previous one! The value could be bigger than 5120 chars
				//this.currentState = REF_STATE;
				return true;
			} else
			if ((c == '<') || (c == '>')) {
				fatalError(EXmlMsg.ERROR_END_QUOTE_EXPECTED);
				return false;						
			} else
			if (isNameStartChar(c)) {
				resetRefBufferWith(c);
				this.currentState = REFNAME_STATE;
				return true;
			} else
			if (c == -1) {
				fatalError(EXmlMsg.ERROR_END_QUOTE_EXPECTED);
				return false;
			} else {
				if (c == '\r') newLine();
				appendAndGrowToWriteBuffer('&'); // The value could be bigger than 5120 chars
				appendAndGrowToWriteBuffer(c); // The value could be bigger than 5120 chars
				this.currentState = VALUE_STATE;
				return true;
			}
		case REFNAME_STATE: // "<aaa ... bbb = '...&cc" has been identified
			if (c == closingValue) {
				appendRefBufferToWriteBuffer();
				endAttValue();
				this.currentState = ATTLIST_STATE;
				return true;
			} else
			if (c == '&') {
				appendRefBufferToWriteBuffer();
				this.currentState = REF_STATE;
				return true;
			} else
			if (c == ';') {
				appendRefValueToWriteBuffer();
				this.currentState = VALUE_STATE;
				return true;
			} else
			if ((c == '<') || (c == '>')) {
				fatalError(EXmlMsg.ERROR_END_QUOTE_EXPECTED);
				return false;
			} else
			if (isNameChar(c)) {
				appendToRefBuffer(c);
				//this.currentState = REFNAME_STATE;
				return true;
			} else
			if (c == -1) {
				fatalError(EXmlMsg.ERROR_END_QUOTE_EXPECTED);
				return false;
			} else {
				if (c == '\r') newLine();
				appendRefBufferToWriteBuffer();
				this.currentState = VALUE_STATE;
				return true;
			}
		case REF2_STATE: // "...&" has been identified
			if (c == '<') {
				appendAndGrowToWriteBuffer('&'); // The value could be bigger than 5120 chars
				this.currentState = MISC2_STATE;
				return true;
			} else
			if (c == '&') {
				appendAndGrowToWriteBuffer('&'); // The previous one! The value could be bigger than 5120 chars
				//this.currentState = REF2_STATE;
				return true;
			} else
			if (c == '>') {
				fatalError(EXmlMsg.ERROR_TAG_EXPECTED);
				return false;						
			} else
			if (isNameChar(c) || (c == '#')) {	//bug 275526 
				resetRefBufferWith(c);
				this.currentState = REF2NAME_STATE;
				return true;
			} else
			if (c == -1) {
				fatalError(EXmlMsg.ERROR_END_QUOTE_EXPECTED);
				return false;
			} else {
				if (c == '\r') newLine();
				appendAndGrowToWriteBuffer('&'); // The value could be bigger than 5120 chars
				appendAndGrowToWriteBuffer(c); // The value could be bigger than 5120 chars
				this.currentState = MISC_STATE;
				return true;
			}
		case REF2NAME_STATE: // "...&a" has been identified
			if (c == '<') {
				appendRefBufferToWriteBuffer();
				this.currentState = MISC2_STATE;
				return true;
			} else
			if (c == '&') {
				appendRefBufferToWriteBuffer();
				this.currentState = REF2_STATE;
				return true;
			} else
			if (c == ';') {
				appendRefValueToWriteBuffer();
				this.currentState = MISC_STATE;
				return true;
			} else
			if (c == '>') {
				fatalError(EXmlMsg.ERROR_TAG_EXPECTED);
				return false;
			} else
			if (isNameChar(c)) {
				appendToRefBuffer(c);
				//this.currentState = REF2NAME_STATE;
				return true;
			} else
			if (c == -1) {
				fatalError(EXmlMsg.ERROR_END_QUOTE_EXPECTED);
				return false;
			} else {
				if (c == '\r') newLine();
				appendRefBufferToWriteBuffer();
				this.currentState = MISC_STATE;
				return true;
			}
		case XPI_TARGET_NAME_STATE: // "<?aa" has been identified
			if ((c == ' ') || (c == '\t') || (c == '\n')) {
				this.currentState = endPITargetName();
				//this.currentState = XPI_XML_TARGET_STATE;
				return true;
			} else
			if (c == '\r') {
				newLine();
				this.currentState = endPITargetName();
				//this.currentState = XPI_XML_TARGET_STATE;
				return true;
			} else
			if (c == '?') {
				this.currentState = endPITargetName();
				this.currentState = (currentState == XPI_XML_TARGET_STATE)?XPI_XML_END_STATE:XPI_DATA_END_STATE;
				return true;
			} else
			if (isNameChar(c)) {
				appendToWriteBuffer(c); // We suppose that the target name will not be bigger than 5120 chars
				//this.currentState = XPI_TARGET_NAME_STATE;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_WRONG_PI);
				return false;
			}
		case XPI_DATA_STATE: // "<?aaa " has been identified with (aaa != xml)
			if ((c == ' ') || (c == '\t') || (c == '\n')) {
				//this.currentState = XPI_DATA_STATE;
				return true;
			} else
			if (c == '\r') {
				newLine();
				//this.currentState = XPI_DATA_STATE;
				return true;
			} else
			if (c == '?') {
				this.currentState = XPI_DATA_END_STATE;
				return true;
			} else
			if (c == -1) {
				fatalError(EXmlMsg.ERROR_WRONG_PI);
				return false;
			} else {
				resetWriteBufferWith(c);
				this.currentState = XPI_DATA_VALUE_STATE;
				return true;				
			}
		case XPI_DATA_VALUE_STATE: // "<?aaa xxx" has been identified with (aaa != xml)
			if (c == -1) {
				fatalError(EXmlMsg.ERROR_WRONG_PI);
				return false;
			} else
			if (c == '\r') {
				newLine();
				//this.currentState = XPI_DATA_VALUE_STATE;
				return true;
			} else
			if (c == '?') {
				this.currentState = XPI_DATA_END_STATE;
				return true;
			} else {
				appendToWriteBuffer(c);	// We suppose that the target attribute will not be bigger than 5120 chars
				//this.currentState = XPI_DATA_VALUE_STATE;
				return true;
			}
		case XPI_DATA_END_STATE: // "<?aaa ...?" has been identified with (aaa != xml)
			if (c == '>') {
				fireProcessingInstruction();
				resetWriteBuffer();
				this.currentState = MISC_STATE;
				return true;
			} else {
				appendToWriteBuffer('?');// We suppose that the target attribute will not be bigger than 5120 chars
				appendToWriteBuffer(c);// We suppose that the target attribute will not be bigger than 5120 chars
				this.currentState = XPI_DATA_VALUE_STATE;
				//fatalError(EXmlMsg.ERROR_WRONG_PI);
				return true;
			}
		case XPI_XML_END_STATE: // "<?xml ... ?" has been identified
			if (c == '>') {
				if (checkXmlDeclaration) checkXmlDeclaration();
				resetWriteBuffer();
				this.currentState = MISC_STATE;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_WRONG_PI);
				return false;
			}
		case XPI_XML_TARGET_STATE: // "<?xml " has been identified
			if ((c == ' ') || (c == '\t') || (c == '\n')) {
				//this.currentState = XPI_XML_TARGET_STATE;
				return true;
			} else
			if (c == '\r') {
				newLine();
				//this.currentState = XPI_XML_TARGET_STATE;
				return true;
			} else
			if (c == '?') {
				this.currentState = XPI_XML_END_STATE;
				return true;
			} else
			if (isNameStartChar(c)) {
				resetWriteBufferWith(c);
				this.currentState = XPI_ATTNAME_STATE;
				return true;				
			} else {
				fatalError(EXmlMsg.ERROR_WRONG_PI);
				return false;
			}
		case XPI_ATTNAME_STATE: // "<?xml bbb" has been identified
			if ((c == ' ') || (c == '\t') || (c == '\n')) {
				endPIAttributeName();
				this.currentState = XPI_ATTEQUAL_STATE;
				return true;
			} else
			if (c == '\r') {
				newLine();
				endPIAttributeName();
				this.currentState = XPI_ATTEQUAL_STATE;
				return true;
			} else
			if (c == '=') {
				endPIAttributeName();
				this.currentState = XPI_ATTVALUE_STATE;
				return true;
			} else
			if (isNameChar(c)) {
				appendToWriteBuffer(c);// We suppose that the target attribute will not be bigger than 5120 chars
				//this.currentState = XPI_ATTNAME_STATE;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_WRONG_PI);
				return false;
			}
		case XPI_ATTEQUAL_STATE: // "<?xml ...bbb " has been identified
			if ((c == ' ') || (c == '\t') || (c == '\n')) {
				//this.currentState = XPI_ATTEQUAL_STATE;
				return true;
			} else
			if (c == '\r') {
				newLine();
				//this.currentState = XPI_ATTEQUAL_STATE;
				return true;
			} else
			if (c == '=') {
				this.currentState = XPI_ATTVALUE_STATE;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_WRONG_PI);
				return false;
			}			
		case XPI_ATTVALUE_STATE: // "<?xml ...bbb =" has been identified
			if ((c == ' ') || (c == '\t') || (c == '\n')) {
				//this.currentState = XPI_ATTVALUE_STATE;
				return true;
			} else
			if (c == '\r') {
				newLine();
				//this.currentState = XPI_ATTVALUE_STATE;
				return true;
			} else
			if ((c == '"') || (c == '\'')) {
				resetAttributeValue();
				resetWriteBuffer();
				this.currentState = XPI_VALUE_STATE;
				closingValue = c;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_WRONG_PI);
				return false;
			}
		case XPI_VALUE_STATE: // "<?xml ...bbb = 'ccc" has been identified
			if (c == closingValue) {
				endPIAttValue();
				this.currentState = XPI_XML_TARGET_STATE;
				return true;
			} else
			if ((c == '>') || (c == '<')) {
				fatalError(EXmlMsg.ERROR_END_QUOTE_EXPECTED);
				return false;
			}else
			if (c == -1) {
				fatalError(EXmlMsg.ERROR_END_OF_TAG_EXPECTED);
				return false;
			}  else {
				if (c == '\r') newLine();
				appendAndGrowToWriteBuffer(c); // The value could be bigger than 5120 chars
				//this.currentState = VALUE_STATE;
				return true;
			}
		case CDATA_STATE: // "<![" has been identified
			if (CDATA_HEADER.charAt(cdataHeaderIndex) == c) {
				cdataHeaderIndex++;
				if (cdataHeaderIndex == CDATA_HEADER.length()) {
					resetWriteBuffer();
					startCDATA();
					this.currentState = CDATA_VALUE_STATE;
					return true;
				} else {
					//this.currentState = CDATA_STATE;
					return true;
				}
			} else {
				fatalError(EXmlMsg.ERROR_CDATA);
				return false;
			}
		 case DOCTYPE_STATE: // "<!D" has been identified
			if (DOCTYPE_HEADER.charAt(doctypeHeaderIndex) == c) {
				doctypeHeaderIndex++;
				if (doctypeHeaderIndex == DOCTYPE_HEADER.length()) {
					this.openAngleBrackets = 0;
					this.openSquareBrackets = 0;
					this.currentState = DOCTYPE_VALUE_STATE;
					return true;
				} else {
					//this.currentState = DOCTYPE_STATE;
					return true;
				}
			} else {
				fatalError(EXmlMsg.ERROR_DOCTYPE);
				return false;
			}
		case CDATA_VALUE_STATE:  // "<![CDATA[...." has been identified
			if (c == ']') {
				this.currentState = CDATA_END_STATE;
				return true;				
			} else 
			if (c == -1) {
				fatalError(EXmlMsg.ERROR_CDATA);
				return false;
			} else {
				if (c == '\r') newLine();
				appendAndGrowToWriteBuffer(c);
				//this.currentState = CDATA_VALUE_STATE;
				return true;
			}
		case CDATA_END_STATE:  // "<![CDATA[....]" has been identified
			if (c == ']') {
				this.currentState = CDATA_END2_STATE;
				return true;
			} else 
			if (c == -1) {
				fatalError(EXmlMsg.ERROR_CDATA);
				return false;
			} else {
				if (c == '\r') newLine();
				appendAndGrowToWriteBuffer(']');
				appendAndGrowToWriteBuffer(c);
				this.currentState = CDATA_VALUE_STATE;
				return true;
			}
		case CDATA_END2_STATE:  // "<![CDATA[....]]" has been identified
			if (c == '>') {
				fireCharacters();
				resetWriteBuffer();
				endCDATA();
				this.currentState = MISC_STATE;
				return true;
			} else {
				fatalError(EXmlMsg.ERROR_CDATA);
				return false;
			}			
		case DOCTYPE_VALUE_STATE:  // "<!DOCTYPE...
			if (c == '>') {
				if (openAngleBrackets > 0) {
					openAngleBrackets--;
					//this.currentState = DOCTYPE_VALUE_STATE;
					return true;
				} else // openAngleBrackets == 0
				if (openSquareBrackets == 0) {
					this.currentState = MISC_STATE;
					return true;
				} else {
					fatalError(EXmlMsg.ERROR_DOCTYPE);
					return false;
				}
			} else
			if (c == '<') {
				openAngleBrackets++;
				//this.currentState = DOCTYPE_VALUE_STATE;
				return true;				
			} else
			if (c == '[') {
				openSquareBrackets++;
				//this.currentState = DOCTYPE_VALUE_STATE;
				return true;			
			} else
			if (c == ']') {
				openSquareBrackets--;
				if (openSquareBrackets >= 0) {
					//this.currentState = DOCTYPE_VALUE_STATE;
					return true;
				} else {
					fatalError(EXmlMsg.ERROR_DOCTYPE);
					return false;
				}
			} else
			if (c == -1) {
				fatalError(EXmlMsg.ERROR_DOCTYPE);
				return false;
			} else {
				if (c == '\r') newLine();
				//this.currentState = DOCTYPE_VALUE_STATE;
				return true;
			}
		default:
			fatalError(EXmlMsg.ERROR_TAG_EXPECTED);
			return false;
	}
}
/**
 * Specs: <a href="http://www.w3.org/TR/xml11/#NT-NameStartChar">Extensible Markup Language (XML) 1.1</a>
 * 
 * NameStartChar    ::=	":" | [A-Z] | "_" | [a-z] |
 * 						[#xC0-#xD6] | [#xD8-#xF6] | [#xF8-#x2FF] | [#x370-#x37D] |
 * 						[#x37F-#x1FFF] | [#x200C-#x200D] | [#x2070-#x218F] |
 * 						[#x2C00-#x2FEF] | [#x3001-#xD7FF] | [#xF900-#xFDCF] |
 * 						[#xFDF0-#xFFFD] | [#x10000-#xEFFFF] 
 * 
 * NameChar    		::= NameStartChar | "-" | "." | [0-9] |
 * 						#xB7 | [#x0300-#x036F] | [#x203F-#x2040] 
 * 
 * We don't consider ':' because it is identified separatly by the parser.
 */
final boolean isNameStartChar(int c) {
	return isIn(c, 'a', 'z')		||
			isIn(c, 'A', 'Z')		||
			(c == '_')				||
			isIn(c, 0xC0, 0xD6)		||
			isIn(c, 0xD8, 0xF6)		||
			isIn(c, 0xF8, 0x2FF)	||
			isIn(c, 0x370, 0x37D)	||
			isIn(c, 0x37F, 0x1FFF)	||
			isIn(c, 0x200C, 0x200D)	||
			isIn(c, 0x2070, 0x218F)	||
			isIn(c, 0x2C00, 0x2FEF)	||
			isIn(c, 0x3001, 0xD7FF)	||
			isIn(c, 0xF900, 0xFDCF)	||
			isIn(c, 0xFDF0, 0xFFFD)	||
			isIn(c, 0x10000, 0xEFFFF);
}
final boolean isNameChar(int c) {
	return isIn(c, 'a', 'z')		||
			isIn(c, 'A', 'Z')		||
			isIn(c, '0', '9')		||
			(c == '_')				||
			(c == '-')				||
			(c == '.')				||
			(c == ':')				||
			(c == 0xB7)				||
			isIn(c, 0xC0, 0xD6)		||
			isIn(c, 0xD8, 0xF6)		||
			isIn(c, 0xF8, 0x37D)	||
			isIn(c, 0x37F, 0x1FFF)	||
			isIn(c, 0x200C, 0x200D)	||
			isIn(c, 0x203F, 0x2040)	||
			isIn(c, 0x2070, 0x218F)	||
			isIn(c, 0x2C00, 0x2FEF)	||
			isIn(c, 0x3001, 0xD7FF)	||
			isIn(c, 0xF900, 0xFDCF)	||
			isIn(c, 0xFDF0, 0xFFFD)	||
			isIn(c, 0x10000, 0xEFFFF);
}
final boolean isIn(int c, int min, int max) {
	return (c >= min) && (c <= max);
}
/**
 * Sets the XML character decoder.
 */
public void setCharDecoder(CharDecoder charDecoder) {
	this.charDecoder = charDecoder;
}
/**
 * Returns the XML character decoder.
 */
public CharDecoder getCharDecoder() {
	return charDecoder;
}
void fatalError(int errorID) throws SAXException {
	fatalError(errorID, null);
}
void fatalError(int errorID, String parameter) throws SAXException {
	fatalError(EXmlMsg.getDefault().getString(errorID), parameter);
}
void warning(int errorID, String parameter) throws SAXException {
	warning(EXmlMsg.getDefault().getString(errorID), parameter);
}
///////// Other dedicated APIs ////////////////
/**
 * Returns the current parsed (but not analyzed) token
 */
public String getWriteBufferAsString() {
	return new String(writeBuffer, 0, writeBufferIndex);	
}
/**
 * Returns the current element local name
 * The element is not already anlyzed 
 */
public String getElementLocalName() {
	return elementLocalName;
}
/**
 * Returns the current parser state (state machine)
 * @return byte = 
 */
public int getCurrentState() {
	return currentState;
}

//////////////////////////////////
/**
 * Return a String describing the parsing error.
 * This method can be overwrite by the subclasses
 * to provide a better decription.
 * By default it is the error number.
 */
protected static String errorMsg(String errorMsg, String parameter) {
	return (parameter != null)?errorMsg+": "+parameter:errorMsg; //$NON-NLS-1$
}
/**
 * Indicates whether or not this parser is configured to validate XML documents.
 * 
 * @return true if this parser is configured to validate XML documents; false otherwise.
 * 			but eXML is a non validating parser so return always 'false'
 */
public boolean isValidating() {
	return false;
}
/**
 * Indicates whether or not this parser is configured to understand namespaces.
 * 
 * @return true if this parser is configured to understand namespaces; false otherwise.
 */
public boolean isNamespaceAware() {
	try {
		return getFeature(XmlOrgFeaturesPrefix+NamespacesFeature);
	} catch (SAXNotSupportedException e) {
		return false;
	} catch (SAXNotRecognizedException e) {
		return false;
	}
}
/**
 * Specifies that the parser produced by this code will
 * provide support for XML namespaces. By default the value of this is set
 * to <code>false</code>.
 *
 * @param awareness true if the parser produced by this code will
 *                  provide support for XML namespaces; false otherwise.
 */

public void setNamespaceAware(boolean awareness) {
	try {
		setFeature(XmlOrgFeaturesPrefix+NamespacesFeature, awareness);
		// The following feature must me set separatly by the user (see PR#143138)
		//setFeature(XmlOrgFeaturesPrefix+NamespacePrefixesFeature, awareness);
	} catch (SAXNotSupportedException e) {
		// NOP
	} catch (SAXNotRecognizedException e) {
		// NOP
	}
}
/**
 * @see XMLReader#getFeature(String)
 */
public boolean getFeature(String name) throws SAXNotRecognizedException, SAXNotSupportedException {
	String feature;
	if ((feature = checkID(name, XmlOrgFeaturesPrefix, XmlOrgFeatureIDs)) == null) throw new SAXNotRecognizedException(name);
	if (NamespacesFeature.equals(feature)) return fNamespaces;
	if (NamespacePrefixesFeature.equals(feature)) return fNamespacesPrefixes;
	if (StringInterningFeature.equals(feature)) {
		if (canDoStringInterning()) return fStringInterning;
		throw new SAXNotSupportedException(name);
	}
	if (XmlNSUris.equals(feature)) {
		return fXmlNSUris;
	} else {
		throw new SAXNotSupportedException(name);
	}
}
/**
 * @see XMLReader#setFeature(String, boolean)
 */
public void setFeature(String name, boolean value) throws SAXNotRecognizedException, SAXNotSupportedException {
	String feature;
	if ((feature = checkID(name, XmlOrgFeaturesPrefix, XmlOrgFeatureIDs)) == null) throw new SAXNotRecognizedException(name);
	
	if (NamespacesFeature.equals(feature)) {
		this.fNamespaces = value;
	} else
	if (NamespacePrefixesFeature.equals(feature)) {
		this.fNamespacesPrefixes = value;
	} else
	if (StringInterningFeature.equals(feature)) {
		if (canDoStringInterning()) {
			this.fStringInterning = value;
		} else {
			throw new SAXNotSupportedException(name);
		}
	} else
	if (XmlNSUris.equals(feature)) {
		this.fXmlNSUris = value;
	} else {
		throw new SAXNotSupportedException(name);
	}
}
/**
 * @see XMLReader#getProperty(String)
 */
public Object getProperty(String name) throws SAXNotRecognizedException, SAXNotSupportedException {
	String id = checkID(name, XmlOrgPropertiesPrefix, XmlOrgPropertyIDs);
	if (id != null) {
		return getRecognizedXmlOrgProperty(id);
	} else {
		throw new SAXNotRecognizedException(name);
	}
}
/**
 * By default, we don't support the default properties...
 */
protected Object getRecognizedXmlOrgProperty(String name) throws SAXNotSupportedException {
	throw new SAXNotSupportedException(name);
}
/**
 * @see XMLReader#setProperty(String, Object)
 */
public void setProperty(String name, Object value) throws SAXNotRecognizedException, SAXNotSupportedException {
	String id = checkID(name, XmlOrgPropertiesPrefix, XmlOrgPropertyIDs);
	if (id != null) {
		setRecognizedXmlOrgProperty(id, value);
	} else {
		throw new SAXNotRecognizedException(name);
	}
}
/**
 * By default, we don't support the default properties...
 */
protected void setRecognizedXmlOrgProperty(String name, Object value) throws SAXNotSupportedException {
	throw new SAXNotSupportedException(name);
}
protected abstract boolean canDoStringInterning();

String checkID(String name, String prefix, String[] list) {
	if (name == null) return null;
	if (!name.startsWith(prefix)) return null;
	String id = name.substring(prefix.length());
	int len = list.length;
	for (int i = 0; i < len; ++i) {
		if (id.startsWith(list[i])) return id;
	}
	return null;
}
/**
 * Pop NamespacesDeclaration
 */
final void popNamespaceDeclaration() throws SAXException {
	while (!currentNSDecl.isNamed(elementNamespace, elementLocalName)) {
		endPrefixMapping(currentNSDecl);
		currentNSDecl = currentNSDecl.parent;
		if (currentNSDecl == null) {
			fatalError(EXmlMsg.WARNING_START_TAG_EXPECTED, elementLocalName);
		}
	}
	endPrefixMapping(currentNSDecl);
	this.currentNSDecl = currentNSDecl.parent;
}
/**
 * Push NamespacesDeclaration
 */
final void pushNamespaceDeclaration() {
	this.currentNSDecl = new NSDeclaration(elementNamespace, elementLocalName, currentNSDecl);
}
/**
 * Fires endElement event...
 */
final void fireEndElement() throws SAXException {
	endElement(elementQName, elementNamespace, elementLocalName);
	if (fNamespaces) popNamespaceDeclaration();
	elementsDepth--;
}
/**
 * Fires startElement event...
 */
final void fireStartElement() throws SAXException {
	startElement(
			elementQName, elementNamespace, elementLocalName,
			(attributes == null)?AttributesImpl.EmptyList:attributes);
	elementsDepth++;
}
/**
 * Match the SAX2 API: Fire startCDATA event
 */
protected abstract void startCDATA() throws SAXException;
/**
 * Match the SAX2 API: Fire endCDATA event
 */
protected abstract void endCDATA() throws SAXException;
/**
 * Fires processingInstruction event
 */
final void fireProcessingInstruction() throws SAXException {
	String data = (writeBufferIndex!=0)?new String(writeBuffer, 0, writeBufferIndex):null;
	processingInstruction (elementQName, data);
}
/**
 * Fires characters event...
 */
final void fireCharacters() throws SAXException {
	if (elementsDepth > 0) characters(writeBuffer, 0, writeBufferIndex);
}
/**
 * Match the SAX API
 * 
 * @param target
 * @param data
 * @throws SAXException
 */
protected abstract void processingInstruction (String target, String data) throws SAXException;
/**
 * Match the SAX API
 * 
 * @param ch
 * @param start
 * @param length
 * @throws SAXException
 */
protected abstract void characters (char ch[], int start, int length) throws SAXException;
/**
 * Match the SAX API
 * 
 * @param ch
 * @param start
 * @param length
 * @throws SAXException
 */
protected abstract void startElement(String qName, String namespace, String localName, AttributesImpl attributes)  throws SAXException;
/**
 * Match the SAX API
 * 
 * @param ch
 * @param start
 * @param length
 * @throws SAXException
 */
protected abstract void endElement(String qName, String namespace, String localName) throws SAXException;
/**
 * Start parsing
 */
protected abstract void startDocument() throws SAXException;
/**
 * End the parsing
 */
protected abstract void endDocument() throws SAXException;
/**
 * Ending prefix Mapping
 */
protected abstract void endPrefixMapping(NSDeclaration nsDecl) throws SAXException;
/**
 * Starting prefix mapping
 */
protected abstract void startPrefixMapping(String prefix, String uri) throws SAXException;
/**
 * A fatal error was encountered.
 */
protected abstract void fatalError(String errorMsg, String parameter) throws SAXException;
/**
 * A warning case was encountered.
 */
protected abstract void warning(String errorMsg, String parameter) throws SAXException;
//////////// Locator APIs ///////////////////
public int getColumnNumber() {
	return columnNumber;
}
public int getLineNumber() {
	return lineNumber;
}
public String getPublicId() {
	return source.getPublicId();
}
public String getSystemId() {
	return source.getSystemId();
}
// Parser states
public final static int DOC_STATE 					= 0;
public final static int PROLOG_STATE				= 1;
public final static int XPI_STATE					= 2;
public final static int MISC_STATE					= 3;
public final static int MISC2_STATE					= 4;
public final static int DT_DECL_STATE				= 5;
public final static int CMT2_STATE					= 6;
public final static int CMT3_STATE					= 7;
public final static int CMT4_STATE					= 8;
public final static int CMT5_STATE					= 9;
public final static int START_ELEMENT_STATE			= 10;
public final static int END_ELEMENT_STATE			= 11;
public final static int END_ELEMENT_NAME_STATE 		= 12;
public final static int END_ELEMENT2_STATE			= 13;
public final static int ENDTAG_STATE				= 14;
public final static int ATTLIST_STATE				= 15;
public final static int ATTNAME_STATE				= 16;
public final static int ATTEQUAL_STATE				= 17;
public final static int ATTVALUE_STATE				= 18;
public final static int VALUE_STATE					= 19;
public final static int REF_STATE					= 20;
public final static int REFNAME_STATE				= 21;
public final static int REF2_STATE					= 22;
public final static int REF2NAME_STATE				= 23;
public final static int XPI_TARGET_NAME_STATE		= 24;
public final static int XPI_DATA_STATE				= 25;
public final static int XPI_DATA_VALUE_STATE		= 26;
public final static int XPI_DATA_END_STATE			= 27;
public final static int XPI_XML_END_STATE			= 28;
public final static int XPI_XML_TARGET_STATE		= 29;
public final static int XPI_ATTNAME_STATE			= 30;
public final static int XPI_ATTEQUAL_STATE			= 31;
public final static int XPI_ATTVALUE_STATE			= 32;
public final static int XPI_VALUE_STATE				= 33;
public final static int CDATA_STATE					= 34;
public final static int DOCTYPE_STATE				= 35;
public final static int CDATA_VALUE_STATE			= 36;
public final static int DOCTYPE_VALUE_STATE			= 37;
public final static int CDATA_END_STATE				= 38;
public final static int CDATA_END2_STATE			= 39;
}
